// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.jetbrains.rhizomedb

import kotlinx.serialization.builtins.serializer
import com.jetbrains.rhizomedb.impl.*

/**
 * All interfaces used with entity apis should extend this interface.
 *
 * The implementation of this interface is generated by rhizome compiler plugin.
 * Object of an [Entity] type represents the identity of an entity.
 * One should think of it as of a closure over [EID] and a schema.
 * Properties of this interface will be mapped to attributes.
 * Values of the properties are being looked up into the [DB] snapshot bound to thread using [asOf], see [DbContext].
 *
 * Under the hood rhizome stores triplets of [eid, attribute_ident, value] for each such property and denormalizes them if needed to facilitate queries.
 * New entities are created with DbContext<Mut>.new. Be aware that returned object is _free_ i.e. holds no data, see [asOf]
 *
 * ### Relations
 * If property type is derived from `Entity` it is called `relation` or `ref`. `Set<Entity>` is a special case, see [Many] for details.
 *
 * ### Validation
 *
 * All non-nullable properties must be initialized on entity creation or you will get an exception.
 * This invariant is enforced on subsequent writes to db so you will get an exception while attempting incorrect write rather than later on read.
 * References to other entities are a bit special in this regard: you cannot delete an entity while there are non-nullable references pointing to it,
 * you should delete referer first. See also [CascadeDeleteBy] and [CascadeDelete]
 *
 * ### Inheritance
 *
 * It is possible to extend Entity interfaces with other interfaces, entity or non-entity.
 * You are free to add methods/properties with default implementations and override them as usual.
 * ```
 * interface Foo : LegacyEntity {
 *   var foo: String
 *   val bar: String get() = "bar"
 *
 *   fun describe(): String = foo + bar
 * }
 *
 * interface IAmNotAEntity { }
 *
 * interface SpecialFoo : Foo {
 *   override fun describe(): String = "override " + foo + bar
 * }
 *
 * val foo = new(SpecialFoo::class) {
 *   foo = "foo"
 * }
 * assertTrue(foo is Foo)
 * assertTrue(foo is IAmNotAEntity)
 * assertTrue(foo is SpecialFoo)
 * assertEquals("override foobar", foo.describe())
 * ```
 */
interface Entity {
  companion object : Mixin<Entity>("Entity", "rhizome") {
    /**
     * A reference to the corresponding [EntityType] entity
     * */
    val Type = requiredRef<EntityType<*>>("type")

    /**
     * The attribute storing the instance, returned by [EntityType.reify], of a user-defined implementation of [Entity] interface.
     * An entity might exist without [EntityObject] if user code defining the corresponding [EntityType] is not loaded.
     * e.g. entity is restored from a snapshot, or received over the wire.
     * */
    val EntityObject = optionalTransient<Entity>("entityObject")

    /**
     * An arbitrary indexed value associated with an entity, it is used to track all entities contributed by plugins to clean them up.
     * */
    val Module = optionalValue("module", String.serializer(), Indexing.INDEXED)
  }

  val eid: EID
}


/**
 * Verify that this entity exists in context database.
 *
 * Entity object is just a pointer that retrieves its properties from context db,
 * as such it is possible that there is no knowledge of this entity in current db:
 * perhaps it was already deleted or this entity object comes from alternative db version.
 *
 * Always returns true for bound entities
 */
fun Entity.exists(): Boolean = DbContext.threadBound.impl.entityExists(eid)

fun Q.entityExists(eid: EID): Boolean =
  queryIndex(IndexQuery.GetOne(eid, Entity.Type.attr as Attribute<EID>)) != null

fun <T : Entity> T?.takeIfExists(): T? = if (this?.exists() == true) this else null

fun entity(eid: EID): Entity? = with(DbContext.threadBound) { entity(eid) }
