/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.history.core;

import com.intellij.history.core.ChangeListStorage;
import com.intellij.history.core.ChangeSetHolder;
import com.intellij.history.core.LocalHistoryStorage;
import com.intellij.history.core.changes.ChangeSet;
import com.intellij.history.utils.LocalHistoryLog;
import com.intellij.ide.BrowserUtil;
import com.intellij.ide.actions.ShowFilePathAction;
import com.intellij.notification.Notification;
import com.intellij.notification.NotificationListener;
import com.intellij.notification.NotificationType;
import com.intellij.notification.Notifications;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.application.PathManager;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.io.FileUtil;
import com.intellij.openapi.vfs.newvfs.ManagingFS;
import com.intellij.util.Consumer;
import com.intellij.util.io.storage.AbstractStorage;
import gnu.trove.TIntHashSet;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.File;
import java.io.IOException;
import java.text.DateFormat;
import java.text.MessageFormat;
import javax.swing.event.HyperlinkEvent;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class ChangeListStorageImpl
implements ChangeListStorage {
    private static final int VERSION = 6;
    private static final String STORAGE_FILE = "changes";
    private final File myStorageDir;
    private LocalHistoryStorage myStorage;
    private long myLastId;
    private boolean isCompletelyBroken = false;

    public ChangeListStorageImpl(File storageDir) throws IOException {
        this.myStorageDir = storageDir;
        this.initStorage(this.myStorageDir);
    }

    private synchronized void initStorage(File storageDir) throws IOException {
        boolean timestampMismatch;
        String path = storageDir.getPath() + "/" + STORAGE_FILE;
        boolean fromScratch = ApplicationManager.getApplication().isUnitTestMode() && !new File(path).exists();
        LocalHistoryStorage result = new LocalHistoryStorage(path);
        long fsTimestamp = ChangeListStorageImpl.getVFSTimestamp();
        int storedVersion = result.getVersion();
        boolean versionMismatch = storedVersion != 6;
        boolean bl = timestampMismatch = result.getFSTimestamp() != fsTimestamp;
        if (versionMismatch || timestampMismatch) {
            if (!fromScratch) {
                if (versionMismatch) {
                    LocalHistoryLog.LOG.info(MessageFormat.format("local history version mismatch (was: {0}, expected: {1}), rebuilding...", storedVersion, 6));
                }
                if (timestampMismatch) {
                    LocalHistoryLog.LOG.info("FS has been rebuild, rebuilding local history...");
                }
                result.dispose();
                if (!FileUtil.delete((File)storageDir)) {
                    throw new IOException("cannot clear storage dir: " + storageDir);
                }
                result = new LocalHistoryStorage(path);
            }
            result.setVersion(6);
            result.setFSTimestamp(fsTimestamp);
        }
        this.myLastId = result.getLastId();
        this.myStorage = result;
    }

    private static long getVFSTimestamp() {
        return ManagingFS.getInstance().getCreationTimestamp();
    }

    private void handleError(Throwable e, @Nullable String message) {
        long storageTimestamp = -1L;
        long vfsTimestamp = ChangeListStorageImpl.getVFSTimestamp();
        long timestamp = System.currentTimeMillis();
        try {
            storageTimestamp = this.myStorage.getFSTimestamp();
        }
        catch (Exception ex) {
            LocalHistoryLog.LOG.warn("cannot read storage timestamp", (Throwable)ex);
        }
        LocalHistoryLog.LOG.error("Local history is broken(version:6,current timestamp:" + DateFormat.getDateTimeInstance().format(timestamp) + ",storage timestamp:" + DateFormat.getDateTimeInstance().format(storageTimestamp) + ",vfs timestamp:" + DateFormat.getDateTimeInstance().format(vfsTimestamp) + ")\n" + message, e);
        this.myStorage.dispose();
        try {
            FileUtil.delete((File)this.myStorageDir);
            this.initStorage(this.myStorageDir);
        }
        catch (Throwable ex) {
            LocalHistoryLog.LOG.error("cannot recreate storage", ex);
            this.isCompletelyBroken = true;
        }
        ChangeListStorageImpl.notifyUser("Local History storage file has become corrupted and will be rebuilt.");
    }

    public static void notifyUser(String message) {
        final String logFile = PathManager.getLogPath();
        Notifications.Bus.notify((Notification)new Notification("System Messages", "Local History is broken", message, NotificationType.ERROR, new NotificationListener(){

            public void hyperlinkUpdate(@NotNull Notification notification, @NotNull HyperlinkEvent event) {
                if (notification == null) {
                    throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "notification", "com/intellij/history/core/ChangeListStorageImpl$1", "hyperlinkUpdate"));
                }
                if (event == null) {
                    throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "event", "com/intellij/history/core/ChangeListStorageImpl$1", "hyperlinkUpdate"));
                }
                if (event.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
                    if ("url".equals(event.getDescription())) {
                        BrowserUtil.browse((String)"http://youtrack.jetbrains.net/issue/IDEA-71270");
                    } else {
                        File file = new File(logFile);
                        ShowFilePathAction.openFile(file);
                    }
                }
            }
        }), null);
    }

    @Override
    public synchronized void close() {
        this.myStorage.dispose();
    }

    @Override
    public synchronized long nextId() {
        return ++this.myLastId;
    }

    @Override
    @Nullable
    public synchronized ChangeSetHolder readPrevious(int id, TIntHashSet recursionGuard) {
        if (this.isCompletelyBroken) {
            return null;
        }
        int prevId = 0;
        try {
            int n = prevId = id == -1 ? this.myStorage.getLastRecord() : this.doReadPrevSafely(id, recursionGuard);
            if (prevId == 0) {
                return null;
            }
            return this.doReadBlock(prevId);
        }
        catch (Throwable e) {
            String message = null;
            if (prevId != 0) {
                try {
                    Pair<Long, Integer> prevOS = this.myStorage.getOffsetAndSize(prevId);
                    long prevRecordTimestamp = this.myStorage.getTimestamp(prevId);
                    int lastRecord = this.myStorage.getLastRecord();
                    Pair<Long, Integer> lastOS = this.myStorage.getOffsetAndSize(lastRecord);
                    long lastRecordTimestamp = this.myStorage.getTimestamp(lastRecord);
                    message = "invalid record is: " + prevId + " offset: " + prevOS.first + " size: " + prevOS.second + " (created " + DateFormat.getDateTimeInstance().format(prevRecordTimestamp) + ") " + "last record is: " + lastRecord + " offset: " + lastOS.first + " size: " + lastOS.second + " (created " + DateFormat.getDateTimeInstance().format(lastRecordTimestamp) + ")";
                }
                catch (Exception e1) {
                    message = "cannot retrieve more debug info: " + e1.getMessage();
                }
            }
            this.handleError(e, message);
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @NotNull
    private ChangeSetHolder doReadBlock(int id) throws IOException {
        DataInputStream in = this.myStorage.readStream(id);
        ChangeSetHolder changeSetHolder = new ChangeSetHolder(id, new ChangeSet(in));
        ChangeSetHolder changeSetHolder2 = changeSetHolder;
        if (changeSetHolder2 == null) {
            throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "com/intellij/history/core/ChangeListStorageImpl", "doReadBlock"));
        }
        return changeSetHolder2;
        finally {
            in.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized void writeNextSet(ChangeSet changeSet) {
        if (this.isCompletelyBroken) {
            return;
        }
        try {
            AbstractStorage.StorageDataOutput out = this.myStorage.writeStream(this.myStorage.createNextRecord(), true);
            try {
                changeSet.write((DataOutput)out);
            }
            finally {
                out.close();
            }
            this.myStorage.setLastId(this.myLastId);
            this.myStorage.force();
        }
        catch (IOException e) {
            this.handleError(e, null);
        }
    }

    @Override
    public synchronized void purge(long period, int intervalBetweenActivities, Consumer<ChangeSet> processor) {
        if (this.isCompletelyBroken) {
            return;
        }
        TIntHashSet recursionGuard = new TIntHashSet(1000);
        try {
            int firstObsoleteId = this.findFirstObsoleteBlock(period, intervalBetweenActivities, recursionGuard);
            if (firstObsoleteId == 0) {
                return;
            }
            int eachBlockId = firstObsoleteId;
            while (eachBlockId != 0) {
                processor.consume((Object)this.doReadBlock((int)eachBlockId).changeSet);
                eachBlockId = this.doReadPrevSafely(eachBlockId, recursionGuard);
            }
            this.myStorage.deleteRecordsUpTo(firstObsoleteId);
            this.myStorage.force();
        }
        catch (IOException e) {
            this.handleError(e, null);
        }
    }

    private int findFirstObsoleteBlock(long period, int intervalBetweenActivities, TIntHashSet recursionGuard) throws IOException {
        long prevTimestamp = 0L;
        long length = 0L;
        int last = this.myStorage.getLastRecord();
        while (last != 0) {
            long t = this.myStorage.getTimestamp(last);
            if (prevTimestamp == 0L) {
                prevTimestamp = t;
            }
            long delta = prevTimestamp - t;
            prevTimestamp = t;
            if ((length += delta < (long)intervalBetweenActivities ? delta : 1L) >= period) {
                return last;
            }
            last = this.doReadPrevSafely(last, recursionGuard);
        }
        return 0;
    }

    private int doReadPrevSafely(int id, TIntHashSet recursionGuard) throws IOException {
        recursionGuard.add(id);
        int prev = this.myStorage.getPrevRecord(id);
        if (!recursionGuard.add(prev)) {
            throw new IOException("Recursive records found");
        }
        return prev;
    }
}

