/*
 * Decompiled with CFR 0.152.
 */
package com.android.tools.perflib.vmtrace;

import com.android.annotations.NonNull;
import com.android.ddmlib.ByteBufferUtil;
import com.android.tools.perflib.vmtrace.Call;
import com.android.tools.perflib.vmtrace.MethodInfo;
import com.android.tools.perflib.vmtrace.MethodProfileData;
import com.android.tools.perflib.vmtrace.ThreadInfo;
import com.android.tools.perflib.vmtrace.TraceAction;
import com.android.tools.perflib.vmtrace.VmTraceData;
import com.google.common.base.Charsets;
import com.google.common.collect.Maps;
import com.google.common.io.Closeables;
import com.google.common.primitives.UnsignedInts;
import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Map;
import java.util.Set;

public class VmTraceParser {
    private static final int TRACE_MAGIC = 1464814675;
    private static final String HEADER_SECTION_VERSION = "*version";
    private static final String HEADER_SECTION_THREADS = "*threads";
    private static final String HEADER_SECTION_METHODS = "*methods";
    private static final String HEADER_END = "*end";
    private static final String KEY_CLOCK = "clock";
    private static final String KEY_DATA_OVERFLOW = "data-file-overflow";
    private static final String KEY_VM = "vm";
    private final File mTraceFile;
    private final VmTraceData.Builder mTraceDataBuilder;
    private VmTraceData mTraceData;
    static final int PARSE_VERSION = 0;
    static final int PARSE_THREADS = 1;
    static final int PARSE_METHODS = 2;
    static final int PARSE_OPTIONS = 4;

    public VmTraceParser(File traceFile) {
        if (!traceFile.exists()) {
            throw new IllegalArgumentException("Trace file " + traceFile.getAbsolutePath() + " does not exist.");
        }
        this.mTraceFile = traceFile;
        this.mTraceDataBuilder = new VmTraceData.Builder();
    }

    public void parse() throws IOException {
        long headerLength = this.parseHeader(this.mTraceFile);
        ByteBuffer buffer = ByteBufferUtil.mapFile(this.mTraceFile, headerLength, ByteOrder.LITTLE_ENDIAN);
        this.parseData(buffer);
        this.computeTimingStatistics();
    }

    public VmTraceData getTraceData() {
        if (this.mTraceData == null) {
            this.mTraceData = this.mTraceDataBuilder.build();
        }
        return this.mTraceData;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    long parseHeader(File f) throws IOException {
        long offset = 0L;
        BufferedReader in = null;
        try {
            in = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(f), Charsets.US_ASCII));
            int mode = 0;
            while (true) {
                String line;
                if ((line = in.readLine()) == null) {
                    throw new IOException("Key section does not have an *end marker");
                }
                offset += (long)(line.length() + 1);
                if (line.startsWith("*")) {
                    if (line.equals(HEADER_SECTION_VERSION)) {
                        mode = 0;
                        continue;
                    }
                    if (line.equals(HEADER_SECTION_THREADS)) {
                        mode = 1;
                        continue;
                    }
                    if (line.equals(HEADER_SECTION_METHODS)) {
                        mode = 2;
                        continue;
                    }
                    if (line.equals(HEADER_END)) break;
                }
                switch (mode) {
                    case 0: {
                        this.mTraceDataBuilder.setVersion(Integer.decode(line));
                        mode = 4;
                        break;
                    }
                    case 1: {
                        this.parseThread(line);
                        break;
                    }
                    case 2: {
                        this.parseMethod(line);
                        break;
                    }
                    case 4: {
                        this.parseOption(line);
                    }
                }
            }
            if (in == null) return offset;
        }
        catch (Throwable throwable) {
            if (in == null) throw throwable;
            try {
                Closeables.close(in, (boolean)true);
                throw throwable;
            }
            catch (IOException e) {
                // empty catch block
            }
            throw throwable;
        }
        try {
            Closeables.close((Closeable)in, (boolean)true);
            return offset;
        }
        catch (IOException e) {}
        return offset;
    }

    private void parseOption(String line) {
        String[] tokens = line.split("=");
        if (tokens.length == 2) {
            String key = tokens[0];
            String value = tokens[1];
            if (key.equals(KEY_CLOCK)) {
                if (value.equals("thread-cpu")) {
                    this.mTraceDataBuilder.setVmClockType(VmTraceData.VmClockType.THREAD_CPU);
                } else if (value.equals("wall")) {
                    this.mTraceDataBuilder.setVmClockType(VmTraceData.VmClockType.WALL);
                } else if (value.equals("dual")) {
                    this.mTraceDataBuilder.setVmClockType(VmTraceData.VmClockType.DUAL);
                }
            } else if (key.equals(KEY_DATA_OVERFLOW)) {
                this.mTraceDataBuilder.setDataFileOverflow(Boolean.parseBoolean(value));
            } else if (key.equals(KEY_VM)) {
                this.mTraceDataBuilder.setVm(value);
            } else {
                this.mTraceDataBuilder.setProperty(key, value);
            }
        }
    }

    private void parseThread(String line) {
        int index = line.indexOf(9);
        if (index < 0) {
            return;
        }
        try {
            int id = Integer.decode(line.substring(0, index));
            String name = line.substring(index).trim();
            this.mTraceDataBuilder.addThread(id, name);
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
    }

    void parseMethod(String line) {
        long id;
        String[] tokens = line.split("\t");
        try {
            id = Long.decode(tokens[0]);
        }
        catch (NumberFormatException e) {
            return;
        }
        String className = tokens[1];
        String methodName = null;
        String signature = null;
        String pathname = null;
        int lineNumber = -1;
        if (tokens.length == 6) {
            methodName = tokens[2];
            signature = tokens[3];
            pathname = tokens[4];
            lineNumber = Integer.decode(tokens[5]);
            pathname = this.constructPathname(className, pathname);
        } else if (tokens.length > 2) {
            if (tokens[3].startsWith("(")) {
                methodName = tokens[2];
                signature = tokens[3];
            } else {
                pathname = tokens[2];
                lineNumber = Integer.decode(tokens[3]);
            }
        }
        this.mTraceDataBuilder.addMethod(id, new MethodInfo(id, className, methodName, signature, pathname, lineNumber));
    }

    private String constructPathname(String className, String pathname) {
        int index = className.lastIndexOf(47);
        if (index > 0 && index < className.length() - 1 && pathname.endsWith(".java")) {
            pathname = className.substring(0, index + 1) + pathname;
        }
        return pathname;
    }

    private void parseData(ByteBuffer buffer) {
        int recordSize = this.readDataFileHeader(buffer);
        this.parseMethodTraceData(buffer, recordSize);
    }

    private void parseMethodTraceData(ByteBuffer buffer, int recordSize) {
        int version = this.mTraceDataBuilder.getVersion();
        VmTraceData.VmClockType vmClockType = this.mTraceDataBuilder.getVmClockType();
        while (buffer.hasRemaining()) {
            TraceAction methodAction;
            int threadTime;
            int globalTime;
            int positionStart = buffer.position();
            short threadId = version == 1 ? buffer.get() : buffer.getShort();
            int methodId = buffer.getInt();
            switch (vmClockType) {
                case WALL: {
                    threadTime = globalTime = buffer.getInt();
                    break;
                }
                case DUAL: {
                    threadTime = buffer.getInt();
                    globalTime = buffer.getInt();
                    break;
                }
                default: {
                    globalTime = threadTime = buffer.getInt();
                }
            }
            int positionEnd = buffer.position();
            int bytesRead = positionEnd - positionStart;
            if (bytesRead < recordSize) {
                buffer.position(positionEnd + (recordSize - bytesRead));
            }
            int action = methodId & 3;
            switch (action) {
                case 0: {
                    methodAction = TraceAction.METHOD_ENTER;
                    break;
                }
                case 1: {
                    methodAction = TraceAction.METHOD_EXIT;
                    break;
                }
                case 2: {
                    methodAction = TraceAction.METHOD_EXIT_UNROLL;
                    break;
                }
                default: {
                    throw new RuntimeException("Invalid trace action, expected one of method entry, exit or unroll.");
                }
            }
            this.mTraceDataBuilder.addMethodAction(threadId, UnsignedInts.toLong((int)(methodId &= 0xFFFFFFFC)), methodAction, threadTime, globalTime);
        }
    }

    private int readDataFileHeader(ByteBuffer buffer) {
        int recordSize;
        int magic = buffer.getInt();
        if (magic != 1464814675) {
            String msg = String.format("Error: magic number mismatch; got 0x%x, expected 0x%x\n", magic, 1464814675);
            throw new RuntimeException(msg);
        }
        short version = buffer.getShort();
        if (version != this.mTraceDataBuilder.getVersion()) {
            String msg = String.format("Error: version number mismatch; got %d in data header but %d in options\n", version, this.mTraceData.getVersion());
            throw new RuntimeException(msg);
        }
        if (version < 1 || version > 3) {
            String msg = String.format("Error: unsupported trace version number %d.  Please use a newer version of TraceView to read this file.", version);
            throw new RuntimeException(msg);
        }
        int offsetToData = buffer.getShort() - 16;
        buffer.getLong();
        switch (version) {
            case 1: {
                recordSize = 9;
                break;
            }
            case 2: {
                recordSize = 10;
                break;
            }
            default: {
                recordSize = buffer.getShort();
                offsetToData -= 2;
            }
        }
        while (offsetToData-- > 0) {
            buffer.get();
        }
        return recordSize;
    }

    private void computeTimingStatistics() {
        VmTraceData data = this.getTraceData();
        ProfileDataBuilder builder = new ProfileDataBuilder();
        for (ThreadInfo thread : data.getThreads()) {
            Call c = thread.getTopLevelCall();
            if (c == null) continue;
            builder.computeCallStats(c, null, thread);
        }
        for (Long methodId : builder.getMethodsWithProfileData()) {
            MethodInfo method = data.getMethod(methodId);
            method.setProfileData(builder.getProfileData(methodId));
        }
    }

    private static class ProfileDataBuilder {
        private final Map<Long, MethodProfileData.Builder> mBuilderMap = Maps.newHashMap();

        private ProfileDataBuilder() {
        }

        public void computeCallStats(Call c, Call parent, ThreadInfo thread) {
            long methodId = c.getMethodId();
            MethodProfileData.Builder builder = this.getProfileDataBuilder(methodId);
            builder.addCallTime(c, parent, thread);
            builder.incrementInvocationCount(c, parent, thread);
            if (c.isRecursive()) {
                builder.setRecursive();
            }
            for (Call callee : c.getCallees()) {
                this.computeCallStats(callee, c, thread);
            }
        }

        @NonNull
        private MethodProfileData.Builder getProfileDataBuilder(long methodId) {
            MethodProfileData.Builder builder = this.mBuilderMap.get(methodId);
            if (builder == null) {
                builder = new MethodProfileData.Builder();
                this.mBuilderMap.put(methodId, builder);
            }
            return builder;
        }

        public Set<Long> getMethodsWithProfileData() {
            return this.mBuilderMap.keySet();
        }

        public MethodProfileData getProfileData(Long methodId) {
            return this.mBuilderMap.get(methodId).build();
        }
    }
}

