/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kotlin.reflect.jvm.internal.impl.descriptors.impl

import kotlin.reflect.jvm.internal.impl.builtins.KotlinBuiltIns
import kotlin.reflect.jvm.internal.impl.descriptors.ModuleDescriptor
import kotlin.reflect.jvm.internal.impl.descriptors.ModuleParameters
import kotlin.reflect.jvm.internal.impl.descriptors.PackageFragmentProvider
import kotlin.reflect.jvm.internal.impl.descriptors.PackageViewDescriptor
import kotlin.reflect.jvm.internal.impl.descriptors.annotations.Annotations
import kotlin.reflect.jvm.internal.impl.name.FqName
import kotlin.reflect.jvm.internal.impl.name.Name
import kotlin.reflect.jvm.internal.impl.storage.StorageManager
import java.util.ArrayList
import java.util.LinkedHashSet
import kotlin.properties.Delegates

public class ModuleDescriptorImpl(
        moduleName: Name,
        private val storageManager: StorageManager,
        private val moduleParameters: ModuleParameters
) : DeclarationDescriptorImpl(Annotations.EMPTY, moduleName), ModuleDescriptor, ModuleParameters by moduleParameters {
    init {
        if (!moduleName.isSpecial()) {
            throw IllegalArgumentException("Module name must be special: $moduleName")
        }
    }
    private var isSealed = false

    /*
     * Sealed module cannot have its dependencies modified. Seal the module after you're done configuring it.
     * Module will be sealed automatically as soon as you query its contents.
     */
    public fun seal() {
        if (isSealed) return

        assert(this in dependencies) { "Module $id is not contained in his own dependencies, this is probably a misconfiguration" }
        isSealed = true
    }

    private val dependencies: MutableList<ModuleDescriptorImpl> = ArrayList()
    private var packageFragmentProviderForModuleContent: PackageFragmentProvider? = null

    private val packageFragmentProviderForWholeModuleWithDependencies by Delegates.lazy {
        seal()
        dependencies.forEach {
            dependency ->
            assert(dependency.isInitialized) {
                "Dependency module ${dependency.id} was not initialized by the time contents of dependent module ${this.id} were queried"
            }
        }
        CompositePackageFragmentProvider(dependencies.map {
            it.packageFragmentProviderForModuleContent!!
        })
    }

    public val isInitialized: Boolean
        get() = packageFragmentProviderForModuleContent != null

    public fun addDependencyOnModule(dependency: ModuleDescriptorImpl) {
        assert(!isSealed) { "Can't modify dependencies of sealed module $id" }
        assert(dependency !in dependencies) {
            "Trying to add dependency on module ${dependency.id} a second time for module ${this.id}, this is probably a misconfiguration"
        }
        dependencies.add(dependency)
    }

    private val id: String
        get() = getName().toString()

    /*
     * Call initialize() to set module contents. Uninitialized module cannot be queried for its contents.
     * Initialize() and seal() can be called in any order.
     */
    public fun initialize(providerForModuleContent: PackageFragmentProvider) {
        assert(!isInitialized) { "Attempt to initialize module $id twice" }
        packageFragmentProviderForModuleContent = providerForModuleContent
    }

    override fun getPackage(fqName: FqName): PackageViewDescriptor? {
        val fragments = packageFragmentProviderForWholeModuleWithDependencies.getPackageFragments(fqName)
        return if (!fragments.isEmpty()) PackageViewDescriptorImpl(this, fqName, fragments) else null
    }

    override fun getSubPackagesOf(fqName: FqName, nameFilter: (Name) -> Boolean): Collection<FqName> {
        return packageFragmentProviderForWholeModuleWithDependencies.getSubPackagesOf(fqName, nameFilter)
    }

    private val friendModules = LinkedHashSet<ModuleDescriptor>()

    override fun isFriend(other: ModuleDescriptor) = other == this || other in friendModules

    public fun addFriend(friend: ModuleDescriptorImpl): Unit {
        assert(friend != this) { "Attempt to make module $id a friend to itself" }
        assert(!isSealed) { "Attempt to add friend module ${friend.id} to sealed module $id" }
        friendModules.add(friend)
    }

    override val builtIns: KotlinBuiltIns
        get() = KotlinBuiltIns.getInstance()
}
