/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.openapi.util.io;

import com.intellij.Patches;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.SystemInfo;
import com.intellij.openapi.util.io.FileAttributes;
import com.intellij.openapi.util.io.win32.FileInfo;
import com.intellij.openapi.util.io.win32.IdeaWin32;
import com.intellij.util.ArrayUtil;
import com.intellij.util.BitUtil;
import com.intellij.util.SystemProperties;
import com.intellij.util.containers.ContainerUtil;
import com.sun.jna.Memory;
import com.sun.jna.Native;
import com.sun.jna.Platform;
import com.sun.jna.Pointer;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class FileSystemUtil {
    private static final Logger LOG = Logger.getInstance("#com.intellij.openapi.util.io.FileSystemUtil");
    @NotNull
    private static Mediator ourMediator = FileSystemUtil.getMediator();

    private static Mediator getMediator() {
        boolean forceNio2 = SystemProperties.getBooleanProperty("idea.io.use.nio2", false);
        boolean forceFallback = SystemProperties.getBooleanProperty("idea.io.use.fallback", false);
        Throwable error = null;
        if (!forceNio2 && !forceFallback) {
            if (SystemInfo.isWindows && IdeaWin32.isAvailable()) {
                try {
                    return FileSystemUtil.check(new IdeaWin32MediatorImpl());
                }
                catch (Throwable t) {
                    error = t;
                }
            } else if (SystemInfo.isLinux || SystemInfo.isMac || SystemInfo.isSolaris || SystemInfo.isFreeBSD) {
                try {
                    return FileSystemUtil.check(new JnaUnixMediatorImpl());
                }
                catch (Throwable t) {
                    error = t;
                }
            }
        }
        if (!forceFallback && SystemInfo.isJavaVersionAtLeast("1.7") && !"1.7.0-ea".equals(SystemInfo.JAVA_VERSION)) {
            try {
                return FileSystemUtil.check(new Nio2MediatorImpl());
            }
            catch (Throwable t) {
                error = t;
            }
        }
        if (!forceFallback) {
            LOG.warn("Failed to load filesystem access layer: " + SystemInfo.OS_NAME + ", " + SystemInfo.JAVA_VERSION + ", nio2=" + forceNio2, error);
        }
        return new FallbackMediatorImpl();
    }

    private static Mediator check(Mediator mediator) throws Exception {
        String quickTestPath = SystemInfo.isWindows ? "C:\\" : "/";
        mediator.getAttributes(quickTestPath);
        return mediator;
    }

    private FileSystemUtil() {
    }

    @Nullable
    public static FileAttributes getAttributes(@NotNull String path) {
        if (path == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "com/intellij/openapi/util/io/FileSystemUtil", "getAttributes"));
        }
        try {
            return ourMediator.getAttributes(path);
        }
        catch (Exception e) {
            LOG.warn(e);
            return null;
        }
    }

    @Nullable
    public static FileAttributes getAttributes(@NotNull File file) {
        if (file == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "file", "com/intellij/openapi/util/io/FileSystemUtil", "getAttributes"));
        }
        return FileSystemUtil.getAttributes(file.getPath());
    }

    public static boolean isSymLink(@NotNull String path) {
        if (path == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "com/intellij/openapi/util/io/FileSystemUtil", "isSymLink"));
        }
        if (SystemInfo.areSymLinksSupported) {
            FileAttributes attributes = FileSystemUtil.getAttributes(path);
            return attributes != null && attributes.isSymLink();
        }
        return false;
    }

    public static boolean isSymLink(@NotNull File file) {
        if (file == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "file", "com/intellij/openapi/util/io/FileSystemUtil", "isSymLink"));
        }
        return FileSystemUtil.isSymLink(file.getAbsolutePath());
    }

    public static boolean clonePermissionsToExecute(@NotNull String source, @NotNull String target) {
        if (source == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "source", "com/intellij/openapi/util/io/FileSystemUtil", "clonePermissionsToExecute"));
        }
        if (target == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "target", "com/intellij/openapi/util/io/FileSystemUtil", "clonePermissionsToExecute"));
        }
        try {
            return ourMediator.clonePermissions(source, target, true);
        }
        catch (Exception e) {
            LOG.warn(e);
            return false;
        }
    }

    private static class FallbackMediatorImpl
    extends Mediator {
        private final Object myFileSystem;
        private final Method myGetBooleanAttributes;

        private FallbackMediatorImpl() {
            Method getBooleanAttributes;
            Object fileSystem;
            try {
                Field fs = File.class.getDeclaredField("fs");
                fs.setAccessible(true);
                fileSystem = fs.get(null);
                getBooleanAttributes = fileSystem.getClass().getMethod("getBooleanAttributes", File.class);
                getBooleanAttributes.setAccessible(true);
            }
            catch (Throwable t) {
                fileSystem = null;
                getBooleanAttributes = null;
            }
            this.myFileSystem = fileSystem;
            this.myGetBooleanAttributes = getBooleanAttributes;
        }

        @Override
        protected FileAttributes getAttributes(@NotNull String path) throws Exception {
            if (path == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "com/intellij/openapi/util/io/FileSystemUtil$FallbackMediatorImpl", "getAttributes"));
            }
            File file = new File(path);
            if (this.myFileSystem != null) {
                int flags = (Integer)this.myGetBooleanAttributes.invoke(this.myFileSystem, file);
                if (flags != 0) {
                    boolean isDirectory = BitUtil.isSet(flags, 4);
                    boolean isSpecial = !BitUtil.isSet(flags, 2) && !BitUtil.isSet(flags, 4);
                    boolean isHidden = BitUtil.isSet(flags, 8) && !FallbackMediatorImpl.isWindowsRoot(path);
                    boolean isWritable = SystemInfo.isWindows && isDirectory || file.canWrite();
                    return new FileAttributes(isDirectory, isSpecial, false, isHidden, file.length(), file.lastModified(), isWritable);
                }
            } else if (file.exists()) {
                boolean isDirectory = file.isDirectory();
                boolean isSpecial = !isDirectory && !file.isFile();
                boolean isHidden = file.isHidden() && !FallbackMediatorImpl.isWindowsRoot(path);
                boolean isWritable = SystemInfo.isWindows && isDirectory || file.canWrite();
                return new FileAttributes(isDirectory, isSpecial, false, isHidden, file.length(), file.lastModified(), isWritable);
            }
            return null;
        }

        private static boolean isWindowsRoot(String p) {
            return SystemInfo.isWindows && p.length() >= 2 && p.length() <= 3 && Character.isLetter(p.charAt(0)) && p.charAt(1) == ':';
        }

        @Override
        protected boolean clonePermissions(@NotNull String source, @NotNull String target, boolean onlyPermissionsToExecute) {
            if (source == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "source", "com/intellij/openapi/util/io/FileSystemUtil$FallbackMediatorImpl", "clonePermissions"));
            }
            if (target == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "target", "com/intellij/openapi/util/io/FileSystemUtil$FallbackMediatorImpl", "clonePermissions"));
            }
            if (SystemInfo.isUnix) {
                File srcFile = new File(source);
                File dstFile = new File(target);
                if (!onlyPermissionsToExecute && !dstFile.setWritable(srcFile.canWrite(), true)) {
                    return false;
                }
                return dstFile.setExecutable(srcFile.canExecute(), true);
            }
            return false;
        }
    }

    private static class JnaUnixMediatorImpl
    extends Mediator {
        private static final int[] LINUX_32 = new int[]{16, 44, 72, 24, 28};
        private static final int[] LINUX_64 = new int[]{24, 48, 88, 28, 32};
        private static final int[] LNX_PPC32 = new int[]{16, 48, 80, 24, 28};
        private static final int[] LNX_PPC64 = LINUX_64;
        private static final int[] LNX_ARM32 = LNX_PPC32;
        private static final int[] BSD_32 = new int[]{8, 48, 32, 12, 16};
        private static final int[] BSD_64 = new int[]{8, 72, 40, 12, 16};
        private static final int[] BSD_32_12 = new int[]{24, 96, 64, 28, 32};
        private static final int[] BSD_64_12 = new int[]{24, 112, 64, 28, 32};
        private static final int[] SUN_OS_32 = new int[]{20, 48, 64, 28, 32};
        private static final int[] SUN_OS_64 = new int[]{16, 40, 64, 24, 28};
        private final int[] myOffsets;
        private final int myUid;
        private final int myGid;
        private final boolean myCoarseTs = SystemProperties.getBooleanProperty("idea.io.coarse.ts", false);

        private JnaUnixMediatorImpl() {
            if ("linux-x86".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = LINUX_32;
            } else if ("linux-x86-64".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = LINUX_64;
            } else if ("linux-arm".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = LNX_ARM32;
            } else if ("linux-ppc".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = LNX_PPC32;
            } else if ("linux-ppc64le".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = LNX_PPC64;
            } else if ("darwin".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = BSD_64;
            } else if ("freebsd-x86".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = SystemInfo.isOsVersionAtLeast("12") ? BSD_32_12 : BSD_32;
            } else if ("freebsd-x86-64".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = SystemInfo.isOsVersionAtLeast("12") ? BSD_64_12 : BSD_64;
            } else if ("sunos-x86".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = SUN_OS_32;
            } else if ("sunos-x86-64".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = SUN_OS_64;
            } else {
                throw new IllegalStateException("Unsupported OS/arch: " + SystemInfo.OS_NAME + "/" + SystemInfo.OS_ARCH);
            }
            Native.register(LibC.class, "c");
            Native.register(SystemInfo.isLinux ? LinuxLibC.class : UnixLibC.class, "c");
            this.myUid = LibC.getuid();
            this.myGid = LibC.getgid();
        }

        @Override
        protected FileAttributes getAttributes(@NotNull String path) {
            long mTime1;
            boolean isSymlink;
            int res2;
            if (path == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "com/intellij/openapi/util/io/FileSystemUtil$JnaUnixMediatorImpl", "getAttributes"));
            }
            Memory buffer = new Memory(256L);
            int n = res2 = SystemInfo.isLinux ? LinuxLibC.__lxstat64(1, path, buffer) : UnixLibC.lstat(path, buffer);
            if (res2 != 0) {
                return null;
            }
            int mode = this.getModeFlags(buffer) & 0xFFFF;
            boolean bl = isSymlink = (mode & 0xF000) == 40960;
            if (isSymlink) {
                if (!JnaUnixMediatorImpl.loadFileStatus(path, buffer)) {
                    return FileAttributes.BROKEN_SYMLINK;
                }
                mode = this.getModeFlags(buffer) & 0xFFFF;
            }
            boolean isDirectory = (mode & 0xF000) == 16384;
            boolean isSpecial = !isDirectory && (mode & 0xF000) != 32768;
            long size = buffer.getLong(this.myOffsets[1]);
            long l = mTime1 = SystemInfo.is32Bit ? (long)buffer.getInt(this.myOffsets[2]) : buffer.getLong(this.myOffsets[2]);
            long mTime2 = this.myCoarseTs ? 0L : (SystemInfo.is32Bit ? (long)buffer.getInt(this.myOffsets[2] + 4) : buffer.getLong(this.myOffsets[2] + 8));
            long mTime = mTime1 * 1000L + mTime2 / 1000000L;
            boolean writable = this.ownFile(buffer) ? (mode & 0x92) != 0 : LibC.access(path, 2) == 0;
            return new FileAttributes(isDirectory, isSpecial, isSymlink, false, size, mTime, writable);
        }

        private static boolean loadFileStatus(String path, Memory buffer) {
            return (SystemInfo.isLinux ? LinuxLibC.__xstat64(1, path, buffer) : UnixLibC.stat(path, buffer)) == 0;
        }

        @Override
        protected boolean clonePermissions(@NotNull String source, @NotNull String target, boolean onlyPermissionsToExecute) {
            int permissions;
            if (source == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "source", "com/intellij/openapi/util/io/FileSystemUtil$JnaUnixMediatorImpl", "clonePermissions"));
            }
            if (target == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "target", "com/intellij/openapi/util/io/FileSystemUtil$JnaUnixMediatorImpl", "clonePermissions"));
            }
            Memory buffer = new Memory(256L);
            if (!JnaUnixMediatorImpl.loadFileStatus(source, buffer)) {
                return false;
            }
            int sourcePermissions = this.getModeFlags(buffer) & 0x1FF;
            if (onlyPermissionsToExecute) {
                if (!JnaUnixMediatorImpl.loadFileStatus(target, buffer)) {
                    return false;
                }
                int targetPermissions = this.getModeFlags(buffer) & 0x1FF;
                permissions = targetPermissions & 0xFFFFFFB6 | sourcePermissions & 0x49;
            } else {
                permissions = sourcePermissions;
            }
            return LibC.chmod(target, permissions) == 0;
        }

        private int getModeFlags(Memory buffer) {
            return SystemInfo.isLinux ? buffer.getInt(this.myOffsets[0]) : (int)buffer.getShort(this.myOffsets[0]);
        }

        private boolean ownFile(Memory buffer) {
            return buffer.getInt(this.myOffsets[3]) == this.myUid && buffer.getInt(this.myOffsets[4]) == this.myGid;
        }

        private static class LinuxLibC {
            private LinuxLibC() {
            }

            static native int __lxstat64(int var0, String var1, Pointer var2);

            static native int __xstat64(int var0, String var1, Pointer var2);
        }

        private static class UnixLibC {
            private UnixLibC() {
            }

            static native int lstat(String var0, Pointer var1);

            static native int stat(String var0, Pointer var1);
        }

        private static class LibC {
            private LibC() {
            }

            static native int getuid();

            static native int getgid();

            static native int chmod(String var0, int var1);

            static native int access(String var0, int var1);
        }
    }

    private static class IdeaWin32MediatorImpl
    extends Mediator {
        private IdeaWin32 myInstance = IdeaWin32.getInstance();

        private IdeaWin32MediatorImpl() {
        }

        @Override
        protected FileAttributes getAttributes(@NotNull String path) {
            if (path == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "com/intellij/openapi/util/io/FileSystemUtil$IdeaWin32MediatorImpl", "getAttributes"));
            }
            FileInfo fileInfo = this.myInstance.getInfo(path);
            return fileInfo != null ? fileInfo.toFileAttributes() : null;
        }
    }

    private static class Nio2MediatorImpl
    extends Mediator {
        private final Method myGetPath;
        private final Object myLinkOptions;
        private final Object myNoFollowLinkOptions;
        private final Method myReadAttributes;
        private final Method mySetAttribute;
        private final Method myToRealPath;
        private final Method myToMillis;
        private final Class<?> mySchema;
        private final Method myIsSymbolicLink;
        private final Method myIsDirectory;
        private final Method myIsOther;
        private final Method mySize;
        private final Method myLastModifiedTime;
        private final Method myIsHidden;
        private final Method myIsReadOnly;
        private final Method myPermissions;

        private Nio2MediatorImpl() throws Exception {
            assert (Patches.USE_REFLECTION_TO_ACCESS_JDK7);
            this.myGetPath = Nio2MediatorImpl.accessible(Class.forName("java.nio.file.Paths").getMethod("get", String.class, String[].class));
            Class<?> pathClass = Class.forName("java.nio.file.Path");
            Class<?> filesClass = Class.forName("java.nio.file.Files");
            Class<?> linkOptClass = Class.forName("java.nio.file.LinkOption");
            this.myLinkOptions = Array.newInstance(linkOptClass, 0);
            this.myNoFollowLinkOptions = Array.newInstance(linkOptClass, 1);
            Array.set(this.myNoFollowLinkOptions, 0, linkOptClass.getField("NOFOLLOW_LINKS").get(null));
            Class<?> linkOptArrayClass = this.myLinkOptions.getClass();
            this.myReadAttributes = Nio2MediatorImpl.accessible(filesClass.getMethod("readAttributes", pathClass, Class.class, linkOptArrayClass));
            this.mySetAttribute = Nio2MediatorImpl.accessible(filesClass.getMethod("setAttribute", pathClass, String.class, Object.class, linkOptArrayClass));
            this.myToRealPath = Nio2MediatorImpl.accessible(pathClass.getMethod("toRealPath", linkOptArrayClass));
            this.myToMillis = Nio2MediatorImpl.accessible(Class.forName("java.nio.file.attribute.FileTime").getMethod("toMillis", new Class[0]));
            this.mySchema = Class.forName("java.nio.file.attribute." + (SystemInfo.isWindows ? "DosFileAttributes" : "PosixFileAttributes"));
            this.myIsSymbolicLink = Nio2MediatorImpl.accessible(this.mySchema.getMethod("isSymbolicLink", new Class[0]));
            this.myIsDirectory = Nio2MediatorImpl.accessible(this.mySchema.getMethod("isDirectory", new Class[0]));
            this.myIsOther = Nio2MediatorImpl.accessible(this.mySchema.getMethod("isOther", new Class[0]));
            this.mySize = Nio2MediatorImpl.accessible(this.mySchema.getMethod("size", new Class[0]));
            this.myLastModifiedTime = Nio2MediatorImpl.accessible(this.mySchema.getMethod("lastModifiedTime", new Class[0]));
            if (SystemInfo.isWindows) {
                this.myIsHidden = Nio2MediatorImpl.accessible(this.mySchema.getMethod("isHidden", new Class[0]));
                this.myIsReadOnly = Nio2MediatorImpl.accessible(this.mySchema.getMethod("isReadOnly", new Class[0]));
                this.myPermissions = null;
            } else {
                this.myIsReadOnly = null;
                this.myIsHidden = null;
                this.myPermissions = Nio2MediatorImpl.accessible(this.mySchema.getMethod("permissions", new Class[0]));
            }
        }

        private static Method accessible(Method method) {
            method.setAccessible(true);
            return method;
        }

        @Override
        protected FileAttributes getAttributes(@NotNull String path) throws Exception {
            if (path == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "com/intellij/openapi/util/io/FileSystemUtil$Nio2MediatorImpl", "getAttributes"));
            }
            try {
                boolean isSymbolicLink;
                Object attributes;
                block8: {
                    Object pathObj = this.myGetPath.invoke(null, path, ArrayUtil.EMPTY_STRING_ARRAY);
                    attributes = this.myReadAttributes.invoke(null, pathObj, this.mySchema, this.myNoFollowLinkOptions);
                    boolean bl = isSymbolicLink = (Boolean)this.myIsSymbolicLink.invoke(attributes, new Object[0]) != false || SystemInfo.isWindows && (Boolean)this.myIsOther.invoke(attributes, new Object[0]) != false && (Boolean)this.myIsDirectory.invoke(attributes, new Object[0]) != false;
                    if (isSymbolicLink) {
                        try {
                            attributes = this.myReadAttributes.invoke(null, pathObj, this.mySchema, this.myLinkOptions);
                        }
                        catch (InvocationTargetException e) {
                            Throwable cause = e.getCause();
                            if (cause == null || !"java.nio.file.NoSuchFileException".equals(cause.getClass().getName())) break block8;
                            return FileAttributes.BROKEN_SYMLINK;
                        }
                    }
                }
                boolean isDirectory = (Boolean)this.myIsDirectory.invoke(attributes, new Object[0]);
                boolean isOther = (Boolean)this.myIsOther.invoke(attributes, new Object[0]);
                long size = (Long)this.mySize.invoke(attributes, new Object[0]);
                long lastModified = (Long)this.myToMillis.invoke(this.myLastModifiedTime.invoke(attributes, new Object[0]), new Object[0]);
                if (SystemInfo.isWindows) {
                    boolean isHidden = new File(path).getParent() == null ? false : (Boolean)this.myIsHidden.invoke(attributes, new Object[0]);
                    boolean isWritable = isDirectory || (Boolean)this.myIsReadOnly.invoke(attributes, new Object[0]) == false;
                    return new FileAttributes(isDirectory, isOther, isSymbolicLink, isHidden, size, lastModified, isWritable);
                }
                boolean isWritable = new File(path).canWrite();
                return new FileAttributes(isDirectory, isOther, isSymbolicLink, false, size, lastModified, isWritable);
            }
            catch (InvocationTargetException e) {
                Throwable cause = e.getCause();
                if (cause instanceof IOException || cause != null && "java.nio.file.InvalidPathException".equals(cause.getClass().getName())) {
                    LOG.debug(cause);
                    return null;
                }
                throw e;
            }
        }

        @Override
        protected boolean clonePermissions(@NotNull String source, @NotNull String target, boolean onlyPermissionsToExecute) throws Exception {
            if (source == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "source", "com/intellij/openapi/util/io/FileSystemUtil$Nio2MediatorImpl", "clonePermissions"));
            }
            if (target == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "target", "com/intellij/openapi/util/io/FileSystemUtil$Nio2MediatorImpl", "clonePermissions"));
            }
            if (SystemInfo.isUnix) {
                Object sourcePath = this.myGetPath.invoke(null, source, ArrayUtil.EMPTY_STRING_ARRAY);
                Object targetPath = this.myGetPath.invoke(null, target, ArrayUtil.EMPTY_STRING_ARRAY);
                Collection sourcePermissions = this.getPermissions(sourcePath);
                Collection targetPermissions = this.getPermissions(targetPath);
                if (sourcePermissions != null && targetPermissions != null) {
                    if (onlyPermissionsToExecute) {
                        HashSet permissionsToSet = ContainerUtil.newHashSet();
                        for (Object permission : targetPermissions) {
                            if (permission.toString().endsWith("_EXECUTE")) continue;
                            permissionsToSet.add(permission);
                        }
                        for (Object permission : sourcePermissions) {
                            if (!permission.toString().endsWith("_EXECUTE")) continue;
                            permissionsToSet.add(permission);
                        }
                        this.mySetAttribute.invoke(null, targetPath, "posix:permissions", permissionsToSet, this.myLinkOptions);
                    } else {
                        this.mySetAttribute.invoke(null, targetPath, "posix:permissions", sourcePermissions, this.myLinkOptions);
                    }
                    return true;
                }
            }
            return false;
        }

        private Collection getPermissions(Object sourcePath) throws IllegalAccessException, InvocationTargetException {
            Object attributes = this.myReadAttributes.invoke(null, sourcePath, this.mySchema, this.myLinkOptions);
            return attributes != null ? (Collection)this.myPermissions.invoke(attributes, new Object[0]) : null;
        }
    }

    private static abstract class Mediator {
        private Mediator() {
        }

        @Nullable
        protected abstract FileAttributes getAttributes(@NotNull String var1) throws Exception;

        protected boolean clonePermissions(@NotNull String source, @NotNull String target, boolean onlyPermissionsToExecute) throws Exception {
            if (source == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "source", "com/intellij/openapi/util/io/FileSystemUtil$Mediator", "clonePermissions"));
            }
            if (target == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "target", "com/intellij/openapi/util/io/FileSystemUtil$Mediator", "clonePermissions"));
            }
            return false;
        }
    }
}

