/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.tools.idea.actions;

import static com.intellij.openapi.actionSystem.LangDataKeys.MODULE;
import static com.intellij.openapi.actionSystem.LangDataKeys.MODULE_CONTEXT_ARRAY;

import com.android.ide.common.util.PathString;
import com.android.tools.idea.projectsystem.ProjectSystemUtil;
import com.android.tools.idea.util.FileExtensions;
import com.intellij.openapi.actionSystem.AnAction;
import com.intellij.openapi.actionSystem.AnActionEvent;
import com.intellij.openapi.actionSystem.DataContext;
import com.intellij.openapi.application.WriteAction;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.util.PlatformIcons;
import java.io.IOException;
import org.jetbrains.android.util.AndroidBundle;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Action to create the main Sample Data directory
 */
public class CreateSampleDataDirectory extends AnAction {
  private static final Logger LOG = Logger.getInstance(CreateSampleDataDirectory.class);

  public CreateSampleDataDirectory() {
    super(AndroidBundle.messagePointer("new.sampledata.dir.action.title"),
          AndroidBundle.messagePointer("new.sampledata.dir.action.description"), PlatformIcons.FOLDER_ICON);
  }

  @Nullable
  private static Module getModuleFromSelection(@NotNull DataContext dataContext) {
    Module[] modules = MODULE_CONTEXT_ARRAY.getData(dataContext);

    if (modules != null && modules.length > 0) {
      return modules[0];
    } else {
      return  MODULE.getData(dataContext);
    }
  }

  private static boolean isActionVisibleForModule(@Nullable Module module) {
    if (module == null) return false;

    PathString sampleDataDirPath = ProjectSystemUtil.getModuleSystem(module).getSampleDataDirectory();
    if (sampleDataDirPath == null) return false;

    // Only display if the directory doesn't exist already
    VirtualFile sampleDataDir = FileExtensions.toVirtualFile(sampleDataDirPath);
    return sampleDataDir == null || !sampleDataDir.exists();
  }

  @Override
  public void update(@NotNull AnActionEvent e) {
    Module module = getModuleFromSelection(e.getDataContext());
    e.getPresentation().setEnabledAndVisible(isActionVisibleForModule(module));
  }

  @Override
  public void actionPerformed(@NotNull AnActionEvent e) {
    Module module = getModuleFromSelection(e.getDataContext());
    assert module != null; // Needs to exist or the action wouldn't be visible

    try {
      WriteAction.run(() -> ProjectSystemUtil.getModuleSystem(module).getOrCreateSampleDataDirectory());
    } catch (IOException ex) {
      LOG.warn("Unable to create sample data directory for module " + module.getName(), ex);
    }
  }
}
