/*
 * Copyright (C) 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.tools.idea.actions

import com.android.AndroidProjectTypes
import com.android.tools.idea.model.AndroidModel
import com.android.tools.idea.model.AndroidModuleInfo
import com.android.tools.idea.npw.model.ProjectSyncInvoker.DefaultProjectSyncInvoker
import com.android.tools.idea.npw.model.RenderTemplateModel.Companion.fromFacet
import com.android.tools.idea.npw.project.getModuleTemplates
import com.android.tools.idea.npw.project.getPackageForApplication
import com.android.tools.idea.npw.project.getPackageForPath
import com.android.tools.idea.npw.template.ConfigureTemplateParametersStep
import com.android.tools.idea.npw.template.TemplateResolver
import com.android.tools.idea.ui.wizard.StudioWizardDialogBuilder
import com.android.tools.idea.ui.wizard.WizardUtils
import com.android.tools.idea.ui.wizard.WizardUtils.COMPOSE_MIN_AGP_VERSION
import com.android.tools.idea.wizard.model.ModelWizard
import com.android.tools.idea.wizard.template.Category
import com.android.tools.idea.wizard.template.TemplateConstraint
import com.android.tools.idea.wizard.template.WizardUiContext
import com.intellij.openapi.actionSystem.AnAction
import com.intellij.openapi.actionSystem.AnActionEvent
import com.intellij.openapi.actionSystem.CommonDataKeys
import com.intellij.openapi.actionSystem.DataKey
import com.intellij.openapi.actionSystem.LangDataKeys
import com.intellij.openapi.module.Module
import icons.AndroidIcons
import icons.StudioIcons
import org.jetbrains.android.facet.AndroidFacet
import org.jetbrains.android.refactoring.hasAndroidxProperty
import org.jetbrains.android.refactoring.isAndroidx
import org.jetbrains.android.util.AndroidBundle
import java.io.File

// These categories will be using a new wizard
val NEW_WIZARD_CATEGORIES = setOf(Category.Activity, Category.Google, Category.Automotive, Category.Compose)
@JvmField
val CREATED_FILES = DataKey.create<MutableList<File>>("CreatedFiles")

/**
 * An action to launch a wizard to create a component from a template.
 */
// TODO(qumeric): consider accepting [Template] instead?
data class NewAndroidComponentAction @JvmOverloads constructor(
  private val category: Category,
  private val templateName: String,
  private val minSdkApi: Int,
  private val minBuildSdkApi: Int = minSdkApi,
  private val templateConstraints: Collection<TemplateConstraint> = setOf()
) : AnAction(templateName, AndroidBundle.message("android.wizard.action.new.component", templateName), null) {

  @Deprecated("Please use the main constructor")
  constructor(
    category: String,
    templateName: String,
    minSdkApi: Int
  ): this(Category.values().find { it.name == category }!!, templateName, minSdkApi)

  var shouldOpenFiles = true

  private val isActivityTemplate: Boolean
    get() = NEW_WIZARD_CATEGORIES.contains(category)

  init {
    templatePresentation.icon = if (isActivityTemplate) AndroidIcons.Activity else StudioIcons.Shell.Filetree.ANDROID_FILE
  }

  override fun update(e: AnActionEvent) {
    val module = LangDataKeys.MODULE.getData(e.dataContext) ?: return
    val moduleInfo = AndroidModuleInfo.getInstance(module) ?: return
    val presentation = e.presentation
    presentation.isVisible = true
    // See also com.android.tools.idea.npw.template.ChooseActivityTypeStep#validateTemplate
    val buildSdkVersion = moduleInfo.buildSdkVersion
    when {
      minSdkApi > moduleInfo.minSdkVersion.featureLevel -> {
        presentation.text = AndroidBundle.message("android.wizard.action.requires.minsdk", templateName, minSdkApi)
        presentation.isEnabled = false
      }
      buildSdkVersion != null && minBuildSdkApi > buildSdkVersion.featureLevel -> {
        presentation.text = AndroidBundle.message("android.wizard.action.requires.minbuildsdk", templateName, minBuildSdkApi)
        presentation.isEnabled = false
      }
      templateConstraints.contains(TemplateConstraint.AndroidX) && !useAndroidX(module) -> {
        presentation.text = AndroidBundle.message("android.wizard.action.requires.androidx", templateName)
        presentation.isEnabled = false
      }
      !WizardUtils.hasComposeMinAgpVersion(module.project, category) -> {
        presentation.text = AndroidBundle.message("android.wizard.action.requires.new.agp", templateName, COMPOSE_MIN_AGP_VERSION)
        presentation.isEnabled = false
      }
      else -> {
        val facet = AndroidFacet.getInstance(module)
        val isProjectReady = facet != null && AndroidModel.get(facet) != null &&
                             facet.configuration.projectType != AndroidProjectTypes.PROJECT_TYPE_INSTANTAPP
        presentation.isEnabled = isProjectReady
      }
    }
  }

  override fun actionPerformed(e: AnActionEvent) {
    val module = LangDataKeys.MODULE.getData(e.dataContext) ?: return
    val facet = AndroidFacet.getInstance(module) ?: return
    if (AndroidModel.get(facet) == null) {
      return
    }
    var targetDirectory = CommonDataKeys.VIRTUAL_FILE.getData(e.dataContext)
    // If the user selected a simulated folder entry (eg "Manifests"), there will be no target directory
    if (targetDirectory != null && !targetDirectory.isDirectory) {
      targetDirectory = targetDirectory.parent
      assert(targetDirectory != null)
    }
    val activityDescription = e.presentation.text // e.g. "Empty Activity", "Tabbed Activity"
    // TODO(qumeric): always show all available templates but preselect a good default
    val moduleTemplates = facet.getModuleTemplates(targetDirectory)
    assert(moduleTemplates.isNotEmpty())
    val initialPackageSuggestion =
      if (targetDirectory == null) facet.getPackageForApplication() else facet.getPackageForPath(moduleTemplates, targetDirectory)
    val templateModel = fromFacet(
      facet, initialPackageSuggestion, moduleTemplates[0], "New $activityDescription", DefaultProjectSyncInvoker(),
      shouldOpenFiles
    )
    val newActivity = TemplateResolver.getAllTemplates()
      .filter { WizardUiContext.MenuEntry in it.uiContexts }
      .find { it.name == templateName }

    templateModel.newTemplate = newActivity!!

    val dialogTitle = AndroidBundle.message(
      if (isActivityTemplate) "android.wizard.new.activity.title" else "android.wizard.new.component.title"
    )
    val stepTitle = AndroidBundle.message(
      if (isActivityTemplate) "android.wizard.config.activity.title" else "android.wizard.config.component.title"
    )
    val wizardBuilder = ModelWizard.Builder().apply {
      addStep(ConfigureTemplateParametersStep(templateModel, stepTitle, moduleTemplates))
    }
    StudioWizardDialogBuilder(wizardBuilder.build(), dialogTitle).setProject(module.project).build().show()
    e.dataContext.getData(CREATED_FILES)?.addAll(templateModel.createdFiles)
  }

  companion object {
    private fun useAndroidX(module: Module?) = module != null && module.project.hasAndroidxProperty() && module.project.isAndroidx()
  }
}