/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.tools.idea.common.actions

import com.android.tools.idea.actions.DesignerActions
import com.android.tools.idea.configurations.Configuration
import com.android.tools.idea.configurations.DeviceMenuAction
import com.android.tools.idea.uibuilder.editor.NlActionManager
import com.intellij.openapi.actionSystem.ActionManager
import com.intellij.openapi.actionSystem.AnAction
import com.intellij.openapi.actionSystem.AnActionEvent

abstract class SwitchDeviceAction: AnAction() {
  final override fun update(e: AnActionEvent) {
    val surface = e.getData(NlActionManager.LAYOUT_EDITOR)
    if (surface != null) {
      val config = surface.configurations.firstOrNull()
      if (config != null) {
        e.presentation.isEnabled = true
        return
      }
    }
    e.presentation.isEnabled = false
  }

  override fun actionPerformed(e: AnActionEvent) {
    val surface = e.getRequiredData(NlActionManager.LAYOUT_EDITOR)
    val config = surface.configurations.firstOrNull() ?: return
    switchDevice(config)
  }

  abstract fun switchDevice(config: Configuration)
}


/**
 * Change {@link DesignSurface}'s device list to next one.
 */
class NextDeviceAction private constructor(): SwitchDeviceAction() {

  override fun switchDevice(config: Configuration) {
    val devices = DeviceMenuAction.getSortedDevicesInMenu(config)
    if (devices.isEmpty()) {
      return
    }
    val nextDevice = when (val index = devices.indexOf(config.device)) {
      -1 -> devices.first() // If current device is not in the list, we navigate to first device
      devices.lastIndex -> devices.first()
      else -> devices[index + 1]
    }
    config.setDevice(nextDevice, true)
  }

  companion object {
    @JvmStatic
    fun getInstance(): NextDeviceAction {
      return ActionManager.getInstance().getAction(DesignerActions.ACTION_NEXT_DEVICE) as NextDeviceAction
    }
  }
}

class PreviousDeviceAction private constructor(): SwitchDeviceAction() {

  override fun switchDevice(config: Configuration) {
    val devices = DeviceMenuAction.getSortedDevicesInMenu(config)
    if (devices.isEmpty()) {
      return
    }
    val previousDevice = when (val index = devices.indexOf(config.device)) {
      -1 -> devices.first() // If current device is not in the list, we navigate to first device
      0 -> devices.last()
      else -> devices[index - 1]
    }
    config.setDevice(previousDevice, true)
  }

  companion object {
    @JvmStatic
    fun getInstance(): PreviousDeviceAction {
      return ActionManager.getInstance().getAction(DesignerActions.ACTION_PREVIOUS_DEVICE) as PreviousDeviceAction
    }
  }
}
