/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.tools.idea.dagger

import com.android.annotations.concurrency.WorkerThread
import com.android.tools.idea.dagger.localization.DaggerBundle.message
import com.android.tools.idea.flags.StudioFlags
import com.google.wireless.android.sdk.stats.DaggerEditorEvent
import com.intellij.codeHighlighting.Pass
import com.intellij.codeInsight.daemon.RelatedItemLineMarkerInfo
import com.intellij.codeInsight.daemon.RelatedItemLineMarkerProvider
import com.intellij.codeInsight.navigation.NavigationUtil
import com.intellij.navigation.GotoRelatedItem
import com.intellij.openapi.components.service
import com.intellij.openapi.editor.markup.GutterIconRenderer
import com.intellij.openapi.progress.ProgressManager
import com.intellij.psi.PsiClass
import com.intellij.psi.PsiElement
import com.intellij.psi.PsiField
import com.intellij.psi.PsiIdentifier
import com.intellij.psi.PsiMethod
import com.intellij.psi.PsiParameter
import com.intellij.psi.impl.source.tree.LeafPsiElement
import com.intellij.psi.presentation.java.SymbolPresentationUtil
import com.intellij.psi.util.parentOfType
import com.intellij.ui.awt.RelativePoint
import icons.StudioIcons
import org.jetbrains.kotlin.lexer.KtTokens
import javax.swing.Icon

/**
 * Provides [RelatedItemLineMarkerInfo] for Dagger elements.
 *
 * Adds gutter icon that allows to navigate between Dagger elements.
 */
class DaggerRelatedItemLineMarkerProvider : RelatedItemLineMarkerProvider() {

  private class GotoItemWithAnalyticsTracking(
    fromElement: PsiElement,
    toElement: PsiElement,
    group: String,
    val customNameToDisplay: String? = null
  ) : GotoRelatedItem(toElement, group) {
    private val fromElementType = getTypeForMetrics(fromElement)
    private val toElementType = getTypeForMetrics(toElement)

    override fun navigate() {
      element?.project?.service<DaggerAnalyticsTracker>()
        ?.trackNavigation(DaggerEditorEvent.NavigationMetadata.NavigationContext.CONTEXT_GUTTER, fromElementType, toElementType)
      super.navigate()
    }

    override fun getCustomName() = customNameToDisplay
  }

  @WorkerThread
  override fun collectNavigationMarkers(element: PsiElement, result: MutableCollection<in RelatedItemLineMarkerInfo<*>>) {
    if (!StudioFlags.DAGGER_SUPPORT_ENABLED.get() || !element.project.service<DaggerDependencyChecker>().isDaggerPresent()) return

    ProgressManager.checkCanceled()
    if (!element.canBeLineMarkerProvide) return

    // We provide RelatedItemLineMarkerInfo for PsiIdentifier/KtIdentifier (leaf element), not for PsiField/PsiMethod,
    // that's why we check that `element.parent` is Dagger related element, not `element` itself. See [LineMarkerProvider.getLineMarkerInfo]
    val parent = element.parent
    val (icon, gotoTargets) = when {
      parent.isDaggerConsumer -> getIconAndGoToItemsForConsumer(parent)
      parent.isDaggerProvider -> getIconAndGoToItemsForProvider(parent)
      parent.isDaggerModule -> getIconAndGoToItemsForModule(parent)
      parent.isDaggerComponent -> getIconAndGoToItemsForComponent(parent)
      parent.isDaggerSubcomponent -> getIconAndGoToItemsForSubcomponent(parent)
      else -> return
    }

    if (gotoTargets.isEmpty()) return

    val typeForMetrics = getTypeForMetrics(parent)

    val info = RelatedItemLineMarkerInfo<PsiElement>(
      element,
      element.textRange,
      icon,
      Pass.LINE_MARKERS,
      getTooltipProvider(parent, gotoTargets),
      { mouseEvent, elt ->
        elt.project.service<DaggerAnalyticsTracker>().trackClickOnGutter(typeForMetrics)
        if (gotoTargets.size == 1) {
          gotoTargets.first().navigate()
        }
        else {
          NavigationUtil.getRelatedItemsPopup(gotoTargets, "Go to Related Files").show(RelativePoint(mouseEvent))
        }
      },
      GutterIconRenderer.Alignment.RIGHT,
      gotoTargets
    )
    result.add(info)
  }

  private fun getTooltipProvider(
    targetElement: PsiElement,
    gotoTargets: List<GotoRelatedItem>
  ): (PsiElement) -> String {
    return {
      val fromElementString = SymbolPresentationUtil.getSymbolPresentableText(targetElement)

      if (gotoTargets.size == 1) {
        with(gotoTargets.single()) {
          val toElementString = customName ?: SymbolPresentationUtil.getSymbolPresentableText(element!!)

          when (group) {
            message("modules.included") -> message("navigate.to.included.module", fromElementString, toElementString)
            message("providers") -> message("navigate.to.provider", fromElementString, toElementString)
            message("consumers") -> message("navigate.to.consumer", fromElementString, toElementString)
            message("exposed.by.components") -> message("navigate.to.component.exposes", fromElementString, toElementString)
            message("parent.components") -> message("navigate.to.parent.component", fromElementString, toElementString)
            message("subcomponents") -> message("navigate.to.subcomponent", fromElementString, toElementString)
            message("included.in.components") -> message("navigate.to.component.that.include", fromElementString, toElementString)
            message("included.in.modules") -> message("navigate.to.module.that.include", fromElementString, toElementString)
            else -> error("[Dagger tools] Unknown navigation group: $group")
          }
        }
      }
      else {
        message("dependency.related.files.for", fromElementString)
      }
    }
  }

  private fun getIconAndGoToItemsForSubcomponent(subcomponent: PsiElement): Pair<Icon, List<GotoRelatedItem>> {
    // [subcomponent] is always PsiClass or KtClass or KtObjectDeclaration, see [isDaggerSubcomponent].
    val asPsiClass = subcomponent.toPsiClass()!!

    val parents = getDaggerParentComponentsForSubcomponent(asPsiClass)
      .map { GotoItemWithAnalyticsTracking(subcomponent, it, message("parent.components")) }

    ProgressManager.checkCanceled()
    val modules = getModulesForComponent(asPsiClass).map {
      GotoItemWithAnalyticsTracking(subcomponent, it, message("modules.included"))
    }

    ProgressManager.checkCanceled()
    val subcomponents = getSubcomponents(asPsiClass).map {
      GotoItemWithAnalyticsTracking(subcomponent, it, message("subcomponents"))
    }

    return Pair(StudioIcons.Misc.DEPENDENCY_CONSUMER, parents + modules + subcomponents)
  }

  private fun getIconAndGoToItemsForComponent(component: PsiElement): Pair<Icon, List<GotoRelatedItem>> {
    // component is always PsiClass or KtClass, see [isDaggerComponent].
    val componentAsPsiClass = component.toPsiClass()!!

    val components = getDependantComponentsForComponent(componentAsPsiClass)
      .map { GotoItemWithAnalyticsTracking(component, it, message("parent.components")) }

    ProgressManager.checkCanceled()
    val subcomponents = getSubcomponents(componentAsPsiClass).map {
      GotoItemWithAnalyticsTracking(component, it, message("subcomponents"))
    }

    ProgressManager.checkCanceled()
    val modules = getModulesForComponent(componentAsPsiClass).map {
      GotoItemWithAnalyticsTracking(component, it, message("modules.included"))
    }
    return Pair(StudioIcons.Misc.DEPENDENCY_CONSUMER, components + subcomponents + modules)
  }

  private fun getIconAndGoToItemsForModule(module: PsiElement): Pair<Icon, List<GotoRelatedItem>> {
    // [module] is always PsiClass or KtClass, see [isDaggerModule].
    val gotoTargets = getUsagesForDaggerModule(module.toPsiClass()!!).map {
      val group = if (it.isDaggerComponent) message("included.in.components") else message("included.in.modules")
      GotoItemWithAnalyticsTracking(module, it, group)
    }
    return Pair(StudioIcons.Misc.DEPENDENCY_CONSUMER, gotoTargets)
  }

  private fun getIconAndGoToItemsForProvider(provider: PsiElement): Pair<Icon, List<GotoRelatedItem>> {
    val consumers = getDaggerConsumersFor(provider).map {
      val nameToDisplay = when (it) {
        is PsiField -> it.parentOfType<PsiClass>()?.name
        is PsiParameter -> it.parentOfType<PsiMethod>()?.name
        else -> error("[Dagger editor] invalid consumer type ${it::class}")
      }
      GotoItemWithAnalyticsTracking(provider, it, message("consumers"), nameToDisplay)
    }

    ProgressManager.checkCanceled()
    val components = getDaggerComponentMethodsForProvider(provider).map {
      GotoItemWithAnalyticsTracking(provider, it, message("exposed.by.components"), it.parentOfType<PsiClass>()?.name)
    }

    ProgressManager.checkCanceled()

    return Pair(StudioIcons.Misc.DEPENDENCY_CONSUMER, consumers + components)
  }

  private fun getIconAndGoToItemsForConsumer(consumer: PsiElement) = getProvidersFor(consumer)

  private fun getProvidersFor(consumer: PsiElement): Pair<Icon, List<GotoRelatedItem>> {
    val gotoTargets = getDaggerProvidersFor(consumer).map { GotoItemWithAnalyticsTracking(consumer, it, message("providers")) }
    return Pair(StudioIcons.Misc.DEPENDENCY_PROVIDER, gotoTargets)
  }
}

/**
 * Returns true if element is Java/Kotlin identifier or kotlin "constructor" keyword.
 *
 * Only leaf elements are suitable for ItemLineMarkerInfo (see [LineMarkerProvider]),
 * and we return true for those leaf elements that could define Dagger consumer/provider.
 */
private val PsiElement.canBeLineMarkerProvide: Boolean
  get() {
    return when (this) {
      is PsiIdentifier -> true
      is LeafPsiElement -> this.elementType == KtTokens.CONSTRUCTOR_KEYWORD || this.elementType == KtTokens.IDENTIFIER
      else -> false
    }
  }