/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.tools.idea.gradle.project.sync.errors

import com.android.tools.idea.gradle.project.sync.idea.issues.BuildIssueComposer
import com.android.tools.idea.gradle.project.sync.quickFixes.OpenFileAtLocationQuickFix
import com.intellij.build.FilePosition
import com.intellij.build.issue.BuildIssue
import org.jetbrains.plugins.gradle.issue.GradleIssueChecker
import org.jetbrains.plugins.gradle.issue.GradleIssueData
import org.jetbrains.plugins.gradle.service.execution.GradleExecutionErrorHandler
import org.jetbrains.plugins.gradle.service.execution.GradleExecutionErrorHandler.getErrorLocation
import java.io.File

class GenericIssueChecker: GradleIssueChecker {
  override fun check(issueData: GradleIssueData): BuildIssue? {
    val message = GradleExecutionErrorHandler.getRootCauseAndLocation(issueData.error).first.message ?: return null
    val buildIssueComposer = BuildIssueComposer(message)
    if (message.isNotEmpty()) {
      val lines = message.lines()
      val errLocation = getErrorLocation(lines[lines.size - 1])
      if (errLocation != null) {
        buildIssueComposer.addQuickFix("Open file",
                                       OpenFileAtLocationQuickFix(FilePosition(File(errLocation.first), errLocation.second - 1, -1)))
        return buildIssueComposer.composeBuildIssue()
      }
    }
    if (issueData.filePosition == null) return null
    buildIssueComposer.addQuickFix("Open file", OpenFileAtLocationQuickFix(issueData.filePosition as FilePosition))
    return buildIssueComposer.composeBuildIssue()
  }
}