/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.tools.idea.npw.assetstudio;

import com.android.ide.common.util.PathString;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Common abstraction for images and resource definition generated by
 * {@link IconGenerator#generateIcons(GraphicGeneratorContext, IconGenerator.IconOptions, String)}.
 */
public abstract class GeneratedIcon {
  @NotNull private final String name;
  @Nullable private final PathString outputPath;
  @NotNull private final IconCategory category;
  @Nullable private String errorMessage;

  protected GeneratedIcon(
      @NotNull String name, @Nullable PathString outputPath, @NotNull IconCategory category, @Nullable String errorMessage) {
    this.name = name;
    this.outputPath = outputPath;
    this.category = category;
    this.errorMessage = errorMessage;
  }

  /** The icon category, see {@link IconCategory}. */
  @NotNull
  public final IconCategory getCategory() {
    return category;
  }

  /**
   * The icon name, never null or empty. The meaning of the name depends on the category. For
   * example, the name {@link IconCategory#PREVIEW} is a {@link LauncherIconGenerator.PreviewShape#id}.
   */
  @NotNull
  public final String getName() {
    return name;
  }

  /**
   * The relative path of the icon in the resource folder (e.g. res/mipmap-xdpi/foo.png).
   * May be null for icons that don't need to be generated on disk (e.g. a preview icon).
   */
  @Nullable
  public PathString getOutputPath() {
    return outputPath;
  }

  /**
   * If there was a problem generating the icon, returns an error message describing the problem. Otherwise returns null.
   */
  @Nullable
  public String getErrorMessage() {
    return errorMessage;
  }

  public void setErrorMessage(@Nullable String errorMessage) {
    this.errorMessage = errorMessage;
  }
}
