/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.tools.idea.uibuilder.scene;

import static com.android.SdkConstants.ATTR_SHOW_IN;
import static com.android.SdkConstants.TOOLS_URI;
import static com.android.resources.Density.DEFAULT_DENSITY;
import static com.android.tools.idea.common.surface.SceneView.SQUARE_SHAPE_POLICY;
import static com.intellij.util.ui.update.Update.HIGH_PRIORITY;
import static com.intellij.util.ui.update.Update.LOW_PRIORITY;

import com.android.ide.common.rendering.api.RenderSession;
import com.android.ide.common.rendering.api.ResourceReference;
import com.android.ide.common.rendering.api.ResourceValue;
import com.android.ide.common.rendering.api.SessionParams;
import com.android.ide.common.rendering.api.ViewInfo;
import com.android.tools.idea.AndroidPsiUtils;
import com.android.tools.idea.common.analytics.CommonUsageTracker;
import com.android.tools.idea.common.diagnostics.NlDiagnosticsManager;
import com.android.tools.idea.common.model.AndroidCoordinate;
import com.android.tools.idea.common.model.AndroidDpCoordinate;
import com.android.tools.idea.common.model.Coordinates;
import com.android.tools.idea.common.model.ModelListener;
import com.android.tools.idea.common.model.NlComponent;
import com.android.tools.idea.common.model.NlModel;
import com.android.tools.idea.common.model.SelectionListener;
import com.android.tools.idea.common.model.SelectionModel;
import com.android.tools.idea.common.scene.DefaultSceneManagerHierarchyProvider;
import com.android.tools.idea.common.scene.Scene;
import com.android.tools.idea.common.scene.SceneComponent;
import com.android.tools.idea.common.scene.SceneManager;
import com.android.tools.idea.common.scene.TemporarySceneComponent;
import com.android.tools.idea.common.scene.decorator.SceneDecoratorFactory;
import com.android.tools.idea.common.surface.DesignSurface;
import com.android.tools.idea.common.surface.SceneView;
import com.android.tools.idea.common.type.DesignerEditorFileType;
import com.android.tools.idea.configurations.Configuration;
import com.android.tools.idea.configurations.ConfigurationListener;
import com.android.tools.idea.rendering.Locale;
import com.android.tools.idea.rendering.RenderLogger;
import com.android.tools.idea.rendering.RenderResult;
import com.android.tools.idea.rendering.RenderService;
import com.android.tools.idea.rendering.RenderTask;
import com.android.tools.idea.rendering.imagepool.ImagePool;
import com.android.tools.idea.rendering.parsers.LayoutPullParsers;
import com.android.tools.idea.rendering.parsers.TagSnapshot;
import com.android.tools.idea.res.ResourceNotificationManager;
import com.android.tools.idea.uibuilder.analytics.NlAnalyticsManager;
import com.android.tools.idea.uibuilder.api.ViewEditor;
import com.android.tools.idea.uibuilder.api.ViewHandler;
import com.android.tools.idea.uibuilder.handlers.ViewEditorImpl;
import com.android.tools.idea.uibuilder.handlers.constraint.targets.ConstraintDragDndTarget;
import com.android.tools.idea.uibuilder.menu.NavigationViewSceneView;
import com.android.tools.idea.uibuilder.model.NlComponentHelperKt;
import com.android.tools.idea.uibuilder.scene.decorator.NlSceneDecoratorFactory;
import com.android.tools.idea.uibuilder.surface.NlDesignSurface;
import com.android.tools.idea.uibuilder.surface.SceneMode;
import com.android.tools.idea.uibuilder.surface.ScreenView;
import com.android.tools.idea.uibuilder.surface.ScreenViewLayer;
import com.android.tools.idea.uibuilder.type.MenuFileType;
import com.android.tools.idea.util.ListenerCollection;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.wireless.android.sdk.stats.LayoutEditorRenderResult;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.progress.util.ProgressIndicatorBase;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Disposer;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.xml.XmlTag;
import com.intellij.util.Alarm;
import com.intellij.util.concurrency.AppExecutorUtil;
import com.intellij.util.concurrency.EdtExecutorService;
import com.intellij.util.ui.TimerUtil;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.ui.TimerUtil;
import com.intellij.util.ui.UIUtil;
import com.intellij.util.ui.update.MergingUpdateQueue;
import com.intellij.util.ui.update.Update;
import java.awt.*;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import javax.swing.*;
import org.jetbrains.android.facet.AndroidFacet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.ide.PooledThreadExecutor;

/**
 * {@link SceneManager} that creates a Scene from an NlModel representing a layout using layoutlib.
 */
public class LayoutlibSceneManager extends SceneManager {
  private static final SceneDecoratorFactory DECORATOR_FACTORY = new NlSceneDecoratorFactory();

  @Nullable private SceneView mySecondarySceneView;

  private int myDpi = 0;
  private final SelectionChangeListener mySelectionChangeListener = new SelectionChangeListener();
  private final ModelChangeListener myModelChangeListener = new ModelChangeListener();
  private final ConfigurationListener myConfigurationChangeListener = new ConfigurationChangeListener();
  private final boolean myAreListenersRegistered;
  private final Object myProgressLock = new Object();
  @GuardedBy("myProgressLock")
  private AndroidPreviewProgressIndicator myCurrentIndicator;
  // Protects all accesses to the rendering queue reference
  private final Object myRenderingQueueLock = new Object();
  @GuardedBy("myRenderingQueueLock")
  private MergingUpdateQueue myRenderingQueue;
  private static final int RENDER_DELAY_MS = 10;
  private RenderTask myRenderTask;
  // Protects all accesses to the myRenderTask reference. RenderTask calls to render and layout do not need to be protected
  // since RenderTask is able to handle those safely.
  private final Object myRenderingTaskLock = new Object();
  private ResourceNotificationManager.ResourceVersion myRenderedVersion;
  // Protects all read/write accesses to the myRenderResult reference
  private final ReentrantReadWriteLock myRenderResultLock = new ReentrantReadWriteLock();
  @GuardedBy("myRenderResultLock")
  private RenderResult myRenderResult;
  // Variables to track previous values of the configuration bar for tracking purposes
  private String myPreviousDeviceName;
  private Locale myPreviousLocale;
  private String myPreviousVersion;
  private String myPreviousTheme;
  @AndroidCoordinate private static final int VISUAL_EMPTY_COMPONENT_SIZE = 1;
  private long myElapsedFrameTimeMs = -1;
  private final LinkedList<CompletableFuture<Void>> myRenderFutures = new LinkedList<>();
  private final Semaphore myUpdateHierarchyLock = new Semaphore(1);
  @NotNull private final ViewEditor myViewEditor;
  private final ListenerCollection<RenderListener> myRenderListeners = ListenerCollection.createWithDirectExecutor();
  /**
   * {@code Executor} to run the {@code Runnable} that disposes {@code RenderTask}s. This allows
   * {@code SyncLayoutlibSceneManager} to use a different strategy to dispose the tasks that does not involve using
   * pooled threads.
   */
  @NotNull private final Executor myRenderTaskDisposerExecutor;
  /**
   * True if we are currently in the middle of a render. This attribute is used to prevent listeners from triggering unnecessary renders.
   * If we try to schedule a new render while this is true, we simply re-use the last render in progress.
   */
  private final AtomicBoolean myIsCurrentlyRendering = new AtomicBoolean(false);

  /**
   * If true, the renders using this LayoutlibSceneManager will use transparent backgrounds
   */
  private boolean useTransparentRendering = false;

  /**
   * If true, the renders will use {@link SessionParams.RenderingMode.SHRINK}
   */
  private boolean useShrinkRendering = false;

  /**
   * If true, the scene is interactive
   */
  private boolean isInteractive = false;

  /**
   * If true, the render will paint the system decorations (status and navigation bards)
   */
  private boolean useShowDecorations;

  /**
   * If false, the use of the {@link ImagePool} will be disabled for the scene manager.
   */
  private boolean useImagePool = true;

  /**
   * Value in the range [0f..1f] to set the quality of the rendering, 0 meaning the lowest quality.
   */
  private float quality = 1f;

  /**
   * When true, it render from layoutlib will contain validation results. When false it'll bypass
   * the validation. In order to allow validation, the layoutlib needs to re-inflate.
   */
  public boolean isLayoutValidationEnabled = false;

  /**
   * {@link Consumer} called when setting up the Rendering {@link MergingUpdateQueue} to do additional setup. This can be used for
   * additional setup required for testing.
   */
  @NotNull private final Consumer<MergingUpdateQueue> myRenderingQueueSetup;

  /**
   * When true, this will force the current {@link RenderTask} to be disposed and re-created on the next render. This will also
   * re-inflate the model.
   */
  private final AtomicBoolean myForceInflate = new AtomicBoolean(false);

  private final AtomicBoolean isDisposed = new AtomicBoolean(false);

  protected static LayoutEditorRenderResult.Trigger getTriggerFromChangeType(@Nullable NlModel.ChangeType changeType) {
    if (changeType == null) {
      return null;
    }

    switch (changeType) {
      case RESOURCE_EDIT:
      case RESOURCE_CHANGED:
        return LayoutEditorRenderResult.Trigger.RESOURCE_CHANGE;
      case EDIT:
      case ADD_COMPONENTS:
      case DELETE:
      case DND_COMMIT:
      case DND_END:
      case DROP:
      case RESIZE_END:
      case RESIZE_COMMIT:
        return LayoutEditorRenderResult.Trigger.EDIT;
      case BUILD:
        return LayoutEditorRenderResult.Trigger.BUILD;
      case CONFIGURATION_CHANGE:
      case UPDATE_HIERARCHY:
        break;
    }

    return null;
  }

  /**
   * Creates a new LayoutlibSceneManager.
   *
   * @param model                      the {@link NlModel} to be rendered by this {@link LayoutlibSceneManager}.
   * @param designSurface              the {@link DesignSurface} user to present the result of the renders.
   * @param renderTaskDisposerExecutor {@link Executor} to be used for running the slow {@link #dispose()} calls.
   * @param renderingQueueSetup        {@link Consumer} of {@link MergingUpdateQueue} to run additional setup on the queue used to handle render
   *                                   requests.
   * @param sceneComponentProvider     a {@link SceneManager.SceneComponentHierarchyProvider providing the mapping from {@link NlComponent} to
   *                                   {@link SceneComponent}s.
   */
  protected LayoutlibSceneManager(@NotNull NlModel model,
                                  @NotNull DesignSurface designSurface,
                                  @NotNull Executor renderTaskDisposerExecutor,
                                  @NotNull Consumer<MergingUpdateQueue> renderingQueueSetup,
                                  @NotNull SceneComponentHierarchyProvider sceneComponentProvider) {
    super(model, designSurface, false, sceneComponentProvider);
    myRenderTaskDisposerExecutor = renderTaskDisposerExecutor;
    myRenderingQueueSetup = renderingQueueSetup;
    createSceneView();
    updateTrackingConfiguration();

    getDesignSurface().getSelectionModel().addListener(mySelectionChangeListener);

    Scene scene = getScene();

    myViewEditor = new ViewEditorImpl(model, scene);

    model.getConfiguration().addListener(myConfigurationChangeListener);

    List<NlComponent> components = model.getComponents();
    if (!components.isEmpty()) {
      NlComponent rootComponent = components.get(0).getRoot();
      boolean previous = getScene().isAnimated();
      scene.setAnimated(false);
      List<SceneComponent> hierarchy = sceneComponentProvider.createHierarchy(this, rootComponent);
      SceneComponent root = hierarchy.isEmpty() ? null : hierarchy.get(0);
      updateFromComponent(root, new HashSet<>());
      scene.setRoot(root);
      updateTargets();
      scene.setAnimated(previous);
    }

    model.addListener(myModelChangeListener);
    myAreListenersRegistered = true;

    // let's make sure the selection is correct
    scene.selectionChanged(getDesignSurface().getSelectionModel(), getDesignSurface().getSelectionModel().getSelection());
  }

  /**
   * Creates a new LayoutlibSceneManager with the default settings for running render requests.
   * See {@link LayoutlibSceneManager#LayoutlibSceneManager(NlModel, DesignSurface, Executor, Consumer)}
   *
   * @param model                  the {@link NlModel} to be rendered by this {@link LayoutlibSceneManager}.
   * @param designSurface          the {@link DesignSurface} user to present the result of the renders.
   * @param sceneComponentProvider a {@link SceneManager.SceneComponentHierarchyProvider providing the mapping from {@link NlComponent} to
   *                               {@link SceneComponent}s.
   */
  public LayoutlibSceneManager(@NotNull NlModel model,
                               @NotNull DesignSurface designSurface,
                               @NotNull SceneComponentHierarchyProvider sceneComponentProvider) {
    this(model, designSurface, AppExecutorUtil.getAppExecutorService(), queue -> {}, sceneComponentProvider);
  }

  /**
   * Creates a new LayoutlibSceneManager with the default settings for running render requests.
   * See {@link LayoutlibSceneManager#LayoutlibSceneManager(NlModel, DesignSurface, Executor, Consumer)}
   *
   * @param model the {@link NlModel} to be rendered by this {@link LayoutlibSceneManager}.
   * @param designSurface the {@link DesignSurface} user to present the result of the renders.
   */
  public LayoutlibSceneManager(@NotNull NlModel model, @NotNull DesignSurface designSurface) {
    this(model, designSurface, AppExecutorUtil.getAppExecutorService(), queue -> {}, new LayoutlibSceneManagerHierarchyProvider());
  }

  @NotNull
  public ViewEditor getViewEditor() {
    return myViewEditor;
  }

  @Override
  @NotNull
  public TemporarySceneComponent createTemporaryComponent(@NotNull NlComponent component) {
    Scene scene = getScene();

    assert scene.getRoot() != null;

    TemporarySceneComponent tempComponent = new TemporarySceneComponent(getScene(), component);
    tempComponent.setTargetProvider(sceneComponent -> ImmutableList.of(new ConstraintDragDndTarget()));
    scene.setAnimated(false);
    scene.getRoot().addChild(tempComponent);
    syncFromNlComponent(tempComponent);
    scene.setAnimated(true);

    return tempComponent;
  }

  @Override
  @NotNull
  public SceneDecoratorFactory getSceneDecoratorFactory() {
    return DECORATOR_FACTORY;
  }

  /**
   * In the layout editor, Scene uses {@link AndroidDpCoordinate}s whereas rendering is done in (zoomed and offset)
   * {@link AndroidCoordinate}s. The scaling factor between them is the ratio of the screen density to the standard density (160).
   */
  @Override
  public float getSceneScalingFactor() {
    return getModel().getConfiguration().getDensity().getDpiValue() / (float)DEFAULT_DENSITY;
  }

  @Override
  public void dispose() {
    if (isDisposed.getAndSet(true)) {
      return;
    }

    try {
      if (myAreListenersRegistered) {
        NlModel model = getModel();
        getDesignSurface().getSelectionModel().removeListener(mySelectionChangeListener);
        model.getConfiguration().removeListener(myConfigurationChangeListener);
        model.removeListener(myModelChangeListener);
      }
      myRenderListeners.clear();

      stopProgressIndicator();
    }
    finally {
      super.dispose();
      if (ApplicationManager.getApplication().isReadAccessAllowed()) {
        // dispose is called by the project close using the read lock. Invoke the render task dispose later without the lock.
        myRenderTaskDisposerExecutor.execute(this::disposeRenderTask);
      }
      else {
        disposeRenderTask();
      }
    }
  }

  private void disposeRenderTask() {
    RenderTask renderTask;
    synchronized (myRenderingTaskLock) {
      renderTask = myRenderTask;
      myRenderTask = null;
    }
    if (renderTask != null) {
      try {
        renderTask.dispose();
      } catch (Throwable t) {
        Logger.getInstance(LayoutlibSceneManager.class).warn(t);
      }
    }
    myRenderResultLock.writeLock().lock();
    try {
      if (myRenderResult != null) {
        myRenderResult.dispose();
      }
      myRenderResult = null;
    }
    finally {
      myRenderResultLock.writeLock().unlock();
    }
  }

  private void stopProgressIndicator() {
    synchronized (myProgressLock) {
      if (myCurrentIndicator != null) {
        myCurrentIndicator.stop();
        myCurrentIndicator = null;
      }
    }
  }


  @NotNull
  @Override
  protected NlDesignSurface getDesignSurface() {
    return (NlDesignSurface) super.getDesignSurface();
  }

  @NotNull
  @Override
  protected SceneView doCreateSceneView() {
    NlModel model = getModel();

    DesignerEditorFileType type = model.getType();

    if (type == MenuFileType.INSTANCE) {
      return createSceneViewsForMenu();
    }

    SceneMode mode = getDesignSurface().getSceneMode();

    SceneView primarySceneView = mode.createPrimarySceneView(getDesignSurface(), this);
    mySecondarySceneView = mode.createSecondarySceneView(getDesignSurface(), this);

    getDesignSurface().updateErrorDisplay();

    return primarySceneView;
  }

  @NotNull
  @Override
  public List<SceneView> getSceneViews() {
    ImmutableList.Builder<SceneView> builder = ImmutableList.<SceneView>builder()
      .addAll(super.getSceneViews());

    if (mySecondarySceneView != null) {
      builder.add(mySecondarySceneView);
    }

    return builder.build();
  }

  private SceneView createSceneViewsForMenu() {
    NlModel model = getModel();
    XmlTag tag = model.getFile().getRootTag();
    SceneView sceneView;

    // TODO See if there's a better way to trigger the NavigationViewSceneView. Perhaps examine the view objects?
    if (tag != null && Objects.equals(tag.getAttributeValue(ATTR_SHOW_IN, TOOLS_URI), NavigationViewSceneView.SHOW_IN_ATTRIBUTE_VALUE)) {
      sceneView = ScreenView.newBuilder(getDesignSurface(), this)
        .withLayersProvider((sv) -> ImmutableList.of(new ScreenViewLayer(sv)))
        .withContentSizePolicy(NavigationViewSceneView.CONTENT_SIZE_POLICY)
        .withShapePolicy(SQUARE_SHAPE_POLICY)
        .build();
    }
    else {
      sceneView = ScreenView.newBuilder(getDesignSurface(), this).build();
    }

    getDesignSurface().updateErrorDisplay();
    return sceneView;
  }

  @Nullable
  public SceneView getSecondarySceneView() {
    return mySecondarySceneView;
  }

  public void updateTargets() {
    SceneComponent root = getScene().getRoot();
    if (root != null) {
      updateTargetProviders(root);
      root.updateTargets();
    }
  }

  private static void updateTargetProviders(@NotNull SceneComponent component) {
    ViewHandler handler = NlComponentHelperKt.getViewHandler(component.getNlComponent());
    component.setTargetProvider(handler);

    for (SceneComponent child : component.getChildren()) {
      updateTargetProviders(child);
    }
  }

  private class ModelChangeListener implements ModelListener {
    @Override
    public void modelDerivedDataChanged(@NotNull NlModel model) {
      NlDesignSurface surface = getDesignSurface();
      // TODO: this is the right behavior, but seems to unveil repaint issues. Turning it off for now.
      if (false && surface.getSceneMode() == SceneMode.BLUEPRINT) {
        requestLayout(true);
      }
      else {
        requestRender(getTriggerFromChangeType(model.getLastChangeType()))
          .thenRunAsync(() ->
            // Selection change listener should run in UI thread not in the layoublib rendering thread. This avoids race condition.
            mySelectionChangeListener.selectionChanged(surface.getSelectionModel(), surface.getSelectionModel().getSelection())
          , EdtExecutorService.getInstance());
      }
    }

    @Override
    public void modelChanged(@NotNull NlModel model) {
      requestModelUpdate();
      ApplicationManager.getApplication().invokeLater(() -> {
        if (!isDisposed.get()) {
          mySelectionChangeListener
            .selectionChanged(getDesignSurface().getSelectionModel(), getDesignSurface().getSelectionModel().getSelection());
        }
      });
    }

    @Override
    public void modelChangedOnLayout(@NotNull NlModel model, boolean animate) {
      UIUtil.invokeLaterIfNeeded(() -> {
        if (!Disposer.isDisposed(LayoutlibSceneManager.this)) {
          boolean previous = getScene().isAnimated();
          getScene().setAnimated(animate);
          update();
          getScene().setAnimated(previous);
        }
      });
    }

    @Override
    public void modelActivated(@NotNull NlModel model) {
      ResourceNotificationManager manager = ResourceNotificationManager.getInstance(getModel().getProject());
      ResourceNotificationManager.ResourceVersion version =
        manager.getCurrentVersion(getModel().getFacet(), getModel().getFile(), getModel().getConfiguration());
      if (!version.equals(myRenderedVersion)) {
        requestModelUpdate();
        model.updateTheme();
      }
      else {
        requestLayoutAndRender(false);
      }
    }

    @Override
    public void modelDeactivated(@NotNull NlModel model) {
      synchronized (myRenderingQueueLock) {
        if (myRenderingQueue != null) {
          myRenderingQueue.cancelAllUpdates();
        }
      }
      disposeRenderTask();
    }

    @Override
    public void modelLiveUpdate(@NotNull NlModel model, boolean animate) {
      requestLayoutAndRender(animate);
    }
  }

  private class SelectionChangeListener implements SelectionListener {
    @Override
    public void selectionChanged(@NotNull SelectionModel model, @NotNull List<NlComponent> selection) {
      updateTargets();
      Scene scene = getScene();
      scene.needsRebuildList();
      scene.repaint();
    }
  }

  /**
   * Adds a new render request to the queue.
   * @param trigger build trigger for reporting purposes
   * @return {@link CompletableFuture} that will be completed once the render has been done.
   */
  @NotNull
  private CompletableFuture<Void> requestRender(@Nullable LayoutEditorRenderResult.Trigger trigger) {
    if (isDisposed.get()) {
      Logger.getInstance(LayoutlibSceneManager.class).warn("requestRender after LayoutlibSceneManager has been disposed");
      return CompletableFuture.completedFuture(null);
    }

    CompletableFuture<Void> callback = new CompletableFuture<>();
    synchronized (myRenderFutures) {
      myRenderFutures.add(callback);
    }

    if (myIsCurrentlyRendering.get()) {
      return callback;
    }

    // This update is low priority so the model updates take precedence
    getRenderingQueue().queue(new Update("model.render", LOW_PRIORITY) {
      @Override
      public void run() {
        render(trigger);
      }

      @Override
      public boolean canEat(Update update) {
        return this.equals(update);
      }
    });

    return callback;
  }

  private class ConfigurationChangeListener implements ConfigurationListener {
    @Override
    public boolean changed(int flags) {
      if ((flags & CFG_DEVICE) != 0) {
        int newDpi = getModel().getConfiguration().getDensity().getDpiValue();
        if (myDpi != newDpi) {
          myDpi = newDpi;
          // Update from the model to update the dpi
          LayoutlibSceneManager.this.update();
        }
      }
      return true;
    }
  }

  @Override
  @NotNull
  public CompletableFuture<Void> requestRender() {
    return requestRender(getTriggerFromChangeType(getModel().getLastChangeType()));
  }

  /**
   * Similar to {@link #requestRender()} but it will be logged as a user initiated action. This is
   * not exposed at SceneManager level since it only makes sense for the Layout editor.
   */
  @NotNull
  public CompletableFuture<Void> requestUserInitiatedRender() {
    forceReinflate();
    return requestRender(LayoutEditorRenderResult.Trigger.USER);
  }

  @Override
  @NotNull
  public CompletableFuture<Void> requestLayoutAndRender(boolean animate) {
    // Don't render if we're just showing the blueprint
    if (getDesignSurface().getSceneMode() == SceneMode.BLUEPRINT) {
      return requestLayout(animate);
    }

    if (getDesignSurface().isRenderingSynchronously()) {
      return render(getTriggerFromChangeType(getModel().getLastChangeType())).thenRun(() -> notifyListenersModelLayoutComplete(animate));
    } else {
      return doRequestLayoutAndRender(animate);
    }
  }

  @NotNull
  CompletableFuture<Void> doRequestLayoutAndRender(boolean animate) {
    return requestRender(getTriggerFromChangeType(getModel().getLastChangeType()))
      .whenCompleteAsync((result, ex) -> notifyListenersModelLayoutComplete(animate), AppExecutorUtil.getAppExecutorService());
  }

  /**
   * Asynchronously inflates the model and updates the view hierarchy
   */
  protected void requestModelUpdate() {
    if (isDisposed.get()) {
      return;
    }
    synchronized (myProgressLock) {
      if (myCurrentIndicator == null) {
        myCurrentIndicator = new AndroidPreviewProgressIndicator();
        myCurrentIndicator.start();
      }
    }

    getRenderingQueue().queue(new Update("model.update", HIGH_PRIORITY) {
      @Override
      public void run() {
        NlModel model = getModel();
        Project project = model.getModule().getProject();
        if (!project.isOpen()) {
          return;
        }
        DumbService.getInstance(project).runWhenSmart(() -> {
          if (model.getVirtualFile().isValid() && !model.getFacet().isDisposed()) {
            updateModel()
              .whenComplete((result, ex) -> stopProgressIndicator());
          }
          else {
            stopProgressIndicator();
          }
        });
      }

      @Override
      public boolean canEat(Update update) {
        return equals(update);
      }
    });
  }

  @NotNull
  private MergingUpdateQueue getRenderingQueue() {
    synchronized (myRenderingQueueLock) {
      if (myRenderingQueue == null) {
        myRenderingQueue = new MergingUpdateQueue("android.layout.rendering", RENDER_DELAY_MS, true, null, this, null,
                                                  Alarm.ThreadToUse.POOLED_THREAD);
        myRenderingQueue.setRestartTimerOnAdd(true);
        // Run any additional setup for the rendering queue
        myRenderingQueueSetup.accept(myRenderingQueue);
      }
      return myRenderingQueue;
    }
  }

  /**
   * Whether we should render just the viewport
   */
  private static boolean ourRenderViewPort;

  public static void setRenderViewPort(boolean state) {
    ourRenderViewPort = state;
  }

  public static boolean isRenderViewPort() {
    return ourRenderViewPort;
  }

  public void setTransparentRendering(boolean enabled) {
    useTransparentRendering = enabled;
  }

  public void setShrinkRendering(boolean enabled) {
    useShrinkRendering = enabled;
  }

  public void setShowDecorations(boolean enabled) {
    if (useShowDecorations != enabled) {
      useShowDecorations = enabled;
      forceReinflate(); // Showing decorations changes the XML content of the render so requires re-inflation
    }
  }

  public boolean isShowingDecorations() {
    return useShowDecorations;
  }

  public void setUseImagePool(boolean enabled) {
    useImagePool = enabled;
  }

  public void setQuality(float quality) {
    this.quality = quality;
  }

  @Override
  @NotNull
  public CompletableFuture<Void> requestLayout(boolean animate) {
    if (isDisposed.get()) {
      Logger.getInstance(LayoutlibSceneManager.class).warn("requestLayout after LayoutlibSceneManager has been disposed");
    }

    synchronized (myRenderingTaskLock) {
      if (myRenderTask == null) {
        return CompletableFuture.completedFuture(null);
      }
      return myRenderTask.layout()
        .thenAccept(result -> {
          if (result != null && !isDisposed.get()) {
            updateHierarchy(result);
            notifyListenersModelLayoutComplete(animate);
          }
        });
    }
  }

  /**
   * Request a layout pass
   *
   * @param animate if true, the resulting layout should be animated
   */
  @Override
  public void layout(boolean animate) {
    try {
      requestLayout(animate).get(2, TimeUnit.SECONDS);
    }
    catch (InterruptedException | ExecutionException | TimeoutException e) {
      Logger.getInstance(LayoutlibSceneManager.class).warn("Unable to run layout()", e);
    }
  }

  @Nullable
  public RenderResult getRenderResult() {
    myRenderResultLock.readLock().lock();
    try {
      return myRenderResult;
    }
    finally {
      myRenderResultLock.readLock().unlock();
    }
  }

  @Override
  @NotNull
  public Map<Object, Map<ResourceReference, ResourceValue>> getDefaultProperties() {
    myRenderResultLock.readLock().lock();
    try {
      if (myRenderResult == null) {
        return Collections.emptyMap();
      }
      return myRenderResult.getDefaultProperties();
    }
    finally {
      myRenderResultLock.readLock().unlock();
    }
  }

  @Override
  @NotNull
  public Map<Object, String> getDefaultStyles() {
    myRenderResultLock.readLock().lock();
    try {
      if (myRenderResult == null) {
        return Collections.emptyMap();
      }
      return myRenderResult.getDefaultStyles();
    }
    finally {
      myRenderResultLock.readLock().unlock();
    }
  }

  private void updateHierarchy(@Nullable RenderResult result) {
    try {
      myUpdateHierarchyLock.acquire();
      try {
        if (result == null || !result.getRenderResult().isSuccess()) {
          updateHierarchy(Collections.emptyList(), getModel());
        }
        else {
          updateHierarchy(getRootViews(result), getModel());
        }
      } finally {
        myUpdateHierarchyLock.release();
      }
      getModel().checkStructure();
    }
    catch (InterruptedException ignored) {
    }
  }

  @NotNull
  private List<ViewInfo> getRootViews(@NotNull RenderResult result) {
    return getModel().getType() == MenuFileType.INSTANCE ? result.getSystemRootViews() : result.getRootViews();
  }

  @VisibleForTesting
  public static void updateHierarchy(@NotNull XmlTag rootTag, @NotNull List<ViewInfo> rootViews, @NotNull NlModel model) {
    model.syncWithPsi(rootTag, ContainerUtil.map(rootViews, ViewInfoTagSnapshotNode::new));
    updateBounds(rootViews, model);
  }

  @VisibleForTesting
  public static void updateHierarchy(@NotNull List<ViewInfo> rootViews, @NotNull NlModel model) {
    XmlTag root = getRootTag(model);
    if (root != null) {
      updateHierarchy(root, rootViews, model);
    }
  }

  // Get the root tag of the xml file associated with the specified model.
  // Since this code may be called on a non UI thread be extra careful about expired objects.
  @Nullable
  private static XmlTag getRootTag(@NotNull NlModel model) {
    if (Disposer.isDisposed(model)) {
      return null;
    }
    return AndroidPsiUtils.getRootTagSafely(model.getFile());
  }

  /**
   * Synchronously inflates the model and updates the view hierarchy
   *
   * @param force forces the model to be re-inflated even if a previous version was already inflated
   * @returns whether the model was inflated in this call or not
   */
  private CompletableFuture<Boolean> inflate(boolean force) {
    long startInflateTimeMs = System.currentTimeMillis();
    Configuration configuration = getModel().getConfiguration();

    Project project = getModel().getProject();
    if (project.isDisposed() || isDisposed.get()) {
      return CompletableFuture.completedFuture(false);
    }

    ResourceNotificationManager resourceNotificationManager = ResourceNotificationManager.getInstance(project);

    // Some types of files must be saved to disk first, because layoutlib doesn't
    // delegate XML parsers for non-layout files (meaning layoutlib will read the
    // disk contents, so we have to push any edits to disk before rendering)
    LayoutPullParsers.saveFileIfNecessary(getModel().getFile());

    synchronized (myRenderingTaskLock) {
      if (myRenderTask != null && !force) {
        // No need to inflate
        return CompletableFuture.completedFuture(false);
      }
    }

    // Record the current version we're rendering from; we'll use that in #activate to make sure we're picking up any
    // external changes
    AndroidFacet facet = getModel().getFacet();
    myRenderedVersion = resourceNotificationManager.getCurrentVersion(facet, getModel().getFile(), configuration);

    RenderService renderService = RenderService.getInstance(getModel().getProject());
    RenderLogger logger = renderService.createLogger(facet);
    RenderService.RenderTaskBuilder renderTaskBuilder = renderService.taskBuilder(facet, configuration)
      .withPsiFile(getModel().getFile())
      .withLayoutValidation(isLayoutValidationEnabled)
      .withLogger(logger);
    return setupRenderTaskBuilder(renderTaskBuilder).build()
      .thenCompose(newTask -> {
        if (newTask != null) {
          newTask.getLayoutlibCallback()
            .setAdaptiveIconMaskPath(getDesignSurface().getAdaptiveIconShape().getPathDescription());
          return newTask.inflate().whenComplete((result, exception) -> {
            if (exception != null) {
              Logger.getInstance(LayoutlibSceneManager.class).warn(exception);
            }

            // If the result is not valid, we do not need the task. Also if the project was already disposed
            // while we were creating the task, avoid adding it.
            if (getModel().getModule().isDisposed() || result == null || !result.getRenderResult().isSuccess() || isDisposed.get()) {
              newTask.dispose();
            }
            else {
              // Update myRenderTask with the new task
              synchronized (myRenderingTaskLock) {
                if (myRenderTask != null && !myRenderTask.isDisposed()) {
                  try {
                    myRenderTask.dispose();
                  } catch (Throwable t) {
                    Logger.getInstance(LayoutlibSceneManager.class).warn(t);
                  }
                }
                myRenderTask = newTask;
              }
            }
          })
            .thenApply(result -> {
              if (result != null) {
                CommonUsageTracker.Companion.getInstance(getDesignSurface()).logRenderResult(null, result, System.currentTimeMillis() - startInflateTimeMs, true);
                return result;
              } else {
                return RenderResult.createBlank(getModel().getFile());
              }
            })
            .thenApply(result -> {
              if (project.isDisposed()) {
                return false;
              }

              updateHierarchy(result);
              myRenderResultLock.writeLock().lock();
              try {
                updateCachedRenderResult(result);
              }
              finally {
                myRenderResultLock.writeLock().unlock();
              }

              return true;
            });
        }
        else {
          synchronized (myRenderingTaskLock) {
            if (myRenderTask != null && !myRenderTask.isDisposed()) {
              try {
                myRenderTask.dispose();
              } catch (Throwable t) {
                Logger.getInstance(LayoutlibSceneManager.class).warn(t);
              }
              myRenderTask = null;
            }
          }
          RenderResult result = RenderResult.createRenderTaskErrorResult(getModel().getFile(), logger);
          myRenderResultLock.writeLock().lock();
          try {
            updateCachedRenderResult(result);
          }
          finally {
            myRenderResultLock.writeLock().unlock();
          }
        }

        return CompletableFuture.completedFuture(false);
      });
  }

  @GuardedBy("myRenderResultLock")
  private void updateCachedRenderResult(RenderResult result) {
    if (myRenderResult != null && myRenderResult != result) {
      myRenderResult.dispose();
    }
    myRenderResult = result;
  }

  @VisibleForTesting
  @NotNull
  protected RenderService.RenderTaskBuilder setupRenderTaskBuilder(@NotNull RenderService.RenderTaskBuilder taskBuilder) {
    if (!useImagePool) {
      taskBuilder.disableImagePool();
    }

    if (quality < 1f) {
      taskBuilder.withDownscaleFactor(quality);
    }

    if (!useShowDecorations) {
      taskBuilder.disableDecorations();
    }

    if (useShrinkRendering) {
      taskBuilder.withRenderingMode(SessionParams.RenderingMode.SHRINK);
    }

    if (useTransparentRendering) {
      taskBuilder.useTransparentBackground();
    }

    if (!getDesignSurface().getPreviewWithToolsVisibilityAndPosition()) {
      taskBuilder.disableToolsVisibilityAndPosition();
    }

    // If two compose previews share the same ClassLoader they share the same compose framework. This way they share the state. In the
    // interactive preview we would like to control the state of the framework and preview. Shared state makes control impossible.
    // Therefore, for interactive (currently only compose) preview we want to create a dedicated ClassLoader so that the preview has its own
    // compose framework. Having a dedicated ClassLoader also allows for clearing resources right after the preview no longer used. We could
    // apply this approach to static previews as well but it might have negative impact if there are many of them, so applying to the
    // interactive previews only.
    if (isInteractive) {
      taskBuilder.usePrivateClassLoader();
    }

    return taskBuilder;
  }

  /**
   * Asynchronously update the model. This will inflate the layout and notify the listeners using
   * {@link ModelListener#modelDerivedDataChanged(NlModel)}.
   */
  protected CompletableFuture<Void> updateModel() {
    if (isDisposed.get()) {
      return CompletableFuture.completedFuture(null);
    }
    return inflate(true)
      .whenCompleteAsync((result, exception) -> notifyListenersModelUpdateComplete(), AppExecutorUtil.getAppExecutorService())
      .thenApply(result -> null);
  }

  protected void notifyListenersModelLayoutComplete(boolean animate) {
    getModel().notifyListenersModelChangedOnLayout(animate);
  }

  protected void notifyListenersModelUpdateComplete() {
    getModel().notifyListenersModelDerivedDataChanged();
  }

  private void logConfigurationChange(@NotNull DesignSurface surface) {
    Configuration configuration = getModel().getConfiguration();

    if (getModel().getConfigurationModificationCount() != configuration.getModificationCount()) {
      // usage tracking (we only pay attention to individual changes where only one item is affected since those are likely to be triggered
      // by the user
      NlAnalyticsManager analyticsManager = ((NlDesignSurface)surface).getAnalyticsManager();
      if (!StringUtil.equals(configuration.getTheme(), myPreviousTheme)) {
        myPreviousTheme = configuration.getTheme();
        analyticsManager.trackThemeChange();
      }
      else if (configuration.getTarget() != null && !StringUtil.equals(configuration.getTarget().getVersionName(), myPreviousVersion)) {
        myPreviousVersion = configuration.getTarget().getVersionName();
        analyticsManager.trackApiLevelChange();
      }
      else if (!configuration.getLocale().equals(myPreviousLocale)) {
        myPreviousLocale = configuration.getLocale();
        analyticsManager.trackLanguageChange();
      }
      else if (configuration.getDevice() != null && !StringUtil.equals(configuration.getDevice().getDisplayName(), myPreviousDeviceName)) {
        myPreviousDeviceName = configuration.getDevice().getDisplayName();
        analyticsManager.trackDeviceChange();
      }
    }
  }

  /**
   * Renders the current model asynchronously. Once the render is complete, the render callbacks will be called.
   * <p/>
   * If the layout hasn't been inflated before, this call will inflate the layout before rendering.
   */
  @NotNull
  protected CompletableFuture<RenderResult> render(@Nullable LayoutEditorRenderResult.Trigger trigger) {
    if (isDisposed.get()) {
      return CompletableFuture.completedFuture(null);
    }

    myIsCurrentlyRendering.set(true);
    try {
      DesignSurface surface = getDesignSurface();
      logConfigurationChange(surface);
      getModel().resetLastChange();

      long renderStartTimeMs = System.currentTimeMillis();
      return renderImpl(trigger)
        .thenApply(result -> {
          if (result == null) {
            completeRender();
            return null;
          }

          myRenderResultLock.writeLock().lock();
          try {
            updateCachedRenderResult(result);
            // TODO(nro): this may not be ideal -- forcing direct results immediately
            if (!isDisposed.get()) {
              update();
            }
            // Downgrade the write lock to read lock
            myRenderResultLock.readLock().lock();
          }
          finally {
            myRenderResultLock.writeLock().unlock();
          }
          try {
            long renderTimeMs = System.currentTimeMillis() - renderStartTimeMs;
            NlDiagnosticsManager.getWriteInstance(surface).recordRender(renderTimeMs,
                                                                        myRenderResult.getRenderedImage().getWidth() * myRenderResult.getRenderedImage().getHeight() * 4L);
          }
          finally {
            myRenderResultLock.readLock().unlock();
          }

          UIUtil.invokeLaterIfNeeded(() -> {
            if (!isDisposed.get()) {
              update();
            }
          });
          fireRenderListeners();
          completeRender();

          return result;
        });
    }
    catch (Throwable e) {
      if (!getModel().getFacet().isDisposed()) {
        completeRender();
        throw e;
      }
    }
    completeRender();
    return CompletableFuture.completedFuture(null);
  }

  /**
   * Completes all the futures created by {@link #requestRender()} and signals the current render as finished by
   * setting {@link #myIsCurrentlyRendering} to false.
   */
  private void completeRender() {
    ImmutableList<CompletableFuture<Void>> callbacks;
    synchronized (myRenderFutures) {
      callbacks = ImmutableList.copyOf(myRenderFutures);
      myRenderFutures.clear();
    }
    callbacks.forEach(callback -> callback.complete(null));
    myIsCurrentlyRendering.set(false);
  }

  @NotNull
  private CompletableFuture<RenderResult> renderImpl(@Nullable LayoutEditorRenderResult.Trigger trigger) {
    return inflate(myForceInflate.getAndSet(false))
      .whenCompleteAsync((result, ex) -> {
        if (ex != null) {
          Logger.getInstance(LayoutlibSceneManager.class).warn(ex);
        }
        if (result) {
          notifyListenersModelUpdateComplete();
        }
      }, PooledThreadExecutor.INSTANCE)
      .thenCompose(inflated -> {
        long elapsedFrameTimeMs = myElapsedFrameTimeMs;

        synchronized (myRenderingTaskLock) {
          if (myRenderTask == null) {
            getDesignSurface().updateErrorDisplay();
            return CompletableFuture.completedFuture(null);
          }
          long startRenderTimeMs = System.currentTimeMillis();
          if (elapsedFrameTimeMs != -1) {
            myRenderTask.setElapsedFrameTimeNanos(TimeUnit.MILLISECONDS.toNanos(elapsedFrameTimeMs));
          }
          return myRenderTask.render().thenApply(result -> {
            // When the layout was inflated in this same call, we do not have to update the hierarchy again
            if (result != null && !inflated) {
              updateHierarchy(result);
            }
            if (result != null) {
              CommonUsageTracker.Companion.getInstance(getDesignSurface()).logRenderResult(trigger, result, System.currentTimeMillis() - startRenderTimeMs, false);
            }
            return result;
          });
        }
      });
  }

  public void setElapsedFrameTimeMs(long ms) {
    myElapsedFrameTimeMs = ms;
  }

  /**
   * Updates the saved values that are used to log user changes to the configuration toolbar.
   */
  private void updateTrackingConfiguration() {
    Configuration configuration = getModel().getConfiguration();
    myPreviousDeviceName = configuration.getCachedDevice() != null ? configuration.getCachedDevice().getDisplayName() : null;
    myPreviousVersion = configuration.getTarget() != null ? configuration.getTarget().getVersionName() : null;
    myPreviousLocale = configuration.getLocale();
    myPreviousTheme = configuration.getTheme();
  }

  private class AndroidPreviewProgressIndicator extends ProgressIndicatorBase {
    private final Object myLock = new Object();

    @Override
    public void start() {
      super.start();
      UIUtil.invokeLaterIfNeeded(() -> {
        final Timer timer = TimerUtil.createNamedTimer("Android rendering progress timer", 0, event -> {
          synchronized (myLock) {
            if (isRunning()) {
              getDesignSurface().registerIndicator(this);
            }
          }
        });
        timer.setRepeats(false);
        timer.start();
      });
    }

    @Override
    public void stop() {
      synchronized (myLock) {
        super.stop();
        ApplicationManager.getApplication().invokeLater(() -> getDesignSurface().unregisterIndicator(this));
      }
    }
  }

  /**
   * A TagSnapshot tree that mirrors the ViewInfo tree.
   */
  private static class ViewInfoTagSnapshotNode implements NlModel.TagSnapshotTreeNode {

    private final ViewInfo myViewInfo;

    private ViewInfoTagSnapshotNode(ViewInfo info) {
      myViewInfo = info;
    }

    @Nullable
    @Override
    public TagSnapshot getTagSnapshot() {
      Object result = myViewInfo.getCookie();
      return result instanceof TagSnapshot ? (TagSnapshot)result : null;
    }

    @NotNull
    @Override
    public List<NlModel.TagSnapshotTreeNode> getChildren() {
      return ContainerUtil.map(myViewInfo.getChildren(), ViewInfoTagSnapshotNode::new);
    }
  }

  /**
   * Default {@link SceneManager.SceneComponentHierarchyProvider} for {@link LayoutlibSceneManager}.
   * It provides the functionality to sync the {@link NlComponent} hierarchy and the data from Layoutlib to {@link SceneComponent}.
   */
  protected static class LayoutlibSceneManagerHierarchyProvider extends DefaultSceneManagerHierarchyProvider {
    @Override
    public void syncFromNlComponent(@NotNull SceneComponent sceneComponent) {
      super.syncFromNlComponent(sceneComponent);
      NlComponent component = sceneComponent.getNlComponent();
      boolean animate = sceneComponent.getScene().isAnimated() && !sceneComponent.hasNoDimension();
      SceneManager manager = sceneComponent.getScene().getSceneManager();
      if (animate) {
        long time = System.currentTimeMillis();
        sceneComponent.setPositionTarget(Coordinates.pxToDp(manager, NlComponentHelperKt.getX(component)),
                                         Coordinates.pxToDp(manager, NlComponentHelperKt.getY(component)),
                                         time);
        sceneComponent.setSizeTarget(Coordinates.pxToDp(manager, NlComponentHelperKt.getW(component)),
                                     Coordinates.pxToDp(manager, NlComponentHelperKt.getH(component)),
                                     time);
      }
      else {
        sceneComponent.setPosition(Coordinates.pxToDp(manager, NlComponentHelperKt.getX(component)),
                                   Coordinates.pxToDp(manager, NlComponentHelperKt.getY(component)));
        sceneComponent.setSize(Coordinates.pxToDp(manager, NlComponentHelperKt.getW(component)),
                               Coordinates.pxToDp(manager, NlComponentHelperKt.getH(component)));
      }
    }
  }

  private static void clearDerivedData(@NotNull NlComponent component) {
    NlComponentHelperKt.setBounds(component, 0, 0, -1, -1); // -1: not initialized
    NlComponentHelperKt.setViewInfo(component, null);
  }

  // TODO: we shouldn't be going back in and modifying NlComponents here
  private static void updateBounds(@NotNull List<ViewInfo> rootViews, @NotNull NlModel model) {
    model.flattenComponents().forEach(LayoutlibSceneManager::clearDerivedData);
    Map<TagSnapshot, NlComponent> snapshotToComponent =
      model.flattenComponents().collect(Collectors.toMap(NlComponent::getSnapshot, Function.identity(), (n1, n2) -> n1));
    Map<XmlTag, NlComponent> tagToComponent =
      model.flattenComponents().collect(Collectors.toMap(NlComponent::getTagDeprecated, Function.identity()));

    // Update the bounds. This is based on the ViewInfo instances.
    for (ViewInfo view : rootViews) {
      updateBounds(view, 0, 0, snapshotToComponent, tagToComponent);
    }

    ImmutableList<NlComponent> components = model.getComponents();
    if (!rootViews.isEmpty() && !components.isEmpty()) {
      // Finally, fix up bounds: ensure that all components not found in the view
      // info hierarchy inherit position from parent
      fixBounds(components.get(0));
    }
  }

  private static void fixBounds(@NotNull NlComponent root) {
    boolean computeBounds = false;
    if (NlComponentHelperKt.getW(root) == -1 && NlComponentHelperKt.getH(root) == -1) { // -1: not initialized
      computeBounds = true;

      // Look at parent instead
      NlComponent parent = root.getParent();
      if (parent != null && NlComponentHelperKt.getW(parent) >= 0) {
        NlComponentHelperKt.setBounds(root, NlComponentHelperKt.getX(parent), NlComponentHelperKt.getY(parent), 0, 0);
      }
    }

    List<NlComponent> children = root.getChildren();
    if (!children.isEmpty()) {
      for (NlComponent child : children) {
        fixBounds(child);
      }

      if (computeBounds) {
        Rectangle rectangle = new Rectangle(NlComponentHelperKt.getX(root), NlComponentHelperKt.getY(root), NlComponentHelperKt.getW(root),
                                            NlComponentHelperKt.getH(root));
        // Grow bounds to include child bounds
        for (NlComponent child : children) {
          rectangle = rectangle.union(new Rectangle(NlComponentHelperKt.getX(child), NlComponentHelperKt.getY(child),
                                                    NlComponentHelperKt.getW(child), NlComponentHelperKt.getH(child)));
        }

        NlComponentHelperKt.setBounds(root, rectangle.x, rectangle.y, rectangle.width, rectangle.height);
      }
    }
  }

  private static void updateBounds(@NotNull ViewInfo view,
                                   @AndroidCoordinate int parentX,
                                   @AndroidCoordinate int parentY,
                                   Map<TagSnapshot, NlComponent> snapshotToComponent,
                                   Map<XmlTag, NlComponent> tagToComponent) {
    ViewInfo bounds = RenderService.getSafeBounds(view);
    Object cookie = view.getCookie();
    NlComponent component;
    if (cookie != null) {
      if (cookie instanceof TagSnapshot) {
        TagSnapshot snapshot = (TagSnapshot)cookie;
        component = snapshotToComponent.get(snapshot);
        if (component == null) {
          component = tagToComponent.get(snapshot.tag);
        }
        if (component != null && NlComponentHelperKt.getViewInfo(component) == null) {
          NlComponentHelperKt.setViewInfo(component, view);
          int left = parentX + bounds.getLeft();
          int top = parentY + bounds.getTop();
          int width = bounds.getRight() - bounds.getLeft();
          int height = bounds.getBottom() - bounds.getTop();

          NlComponentHelperKt.setBounds(component, left, top, Math.max(width, VISUAL_EMPTY_COMPONENT_SIZE),
                                        Math.max(height, VISUAL_EMPTY_COMPONENT_SIZE));
        }
      }
    }
    parentX += bounds.getLeft();
    parentY += bounds.getTop();

    for (ViewInfo child : view.getChildren()) {
      updateBounds(child, parentX, parentY, snapshotToComponent, tagToComponent);
    }
  }

  protected void fireRenderListeners() {
    myRenderListeners.forEach(RenderListener::onRenderCompleted);
  }

  public void addRenderListener(@NotNull RenderListener listener) {
    if (isDisposed.get()) {
      Logger.getInstance(LayoutlibSceneManager.class).warn("addRenderListener after LayoutlibSceneManager has been disposed");
    }

    myRenderListeners.add(listener);
  }

  public void removeRenderListener(@NotNull RenderListener listener) {
    myRenderListeners.remove(listener);
  }

  /**
   * This invalidates the current {@link RenderTask}. Next render call will be force to re-inflate the model.
   */
  public void forceReinflate() {
    myForceInflate.set(true);
  }

  /**
   * Triggers execution of the Handler and frame callbacks in the layoutlib
   * @return a boolean future that is completed when callbacks are executed that is true if there are more callbacks to execute
   */
  @NotNull
  public CompletableFuture<Boolean> executeCallbacks() {
    if (isDisposed.get()) {
      Logger.getInstance(LayoutlibSceneManager.class).warn("executeCallbacks after LayoutlibSceneManager has been disposed");
    }

    synchronized (myRenderingTaskLock) {
      if (myRenderTask == null) {
        return CompletableFuture.completedFuture(false);
      }
      return myRenderTask.executeCallbacks();
    }
  }

  /**
   * Informs layoutlib that there was a (mouse) touch event detected of a particular type at a particular point
   * @param type type of a touch event
   * @param x horizontal android coordinate of the detected touch event
   * @param y vertical android coordinate of the detected touch event
   * @return a future that is completed when layoutlib handled the touch event
   */
  @NotNull
  public CompletableFuture<Void> triggerTouchEvent(
    @NotNull RenderSession.TouchEventType type, @AndroidCoordinate int x, @AndroidCoordinate int y) {
    if (isDisposed.get()) {
      Logger.getInstance(LayoutlibSceneManager.class).warn("executeCallbacks after LayoutlibSceneManager has been disposed");
    }

    synchronized (myRenderingTaskLock) {
      if (myRenderTask == null) {
        return CompletableFuture.completedFuture(null);
      }
      return myRenderTask.triggerTouchEvent(type, x, y);
    }
  }

  /**
   * Sets interactive mode of the scene.
   * @param interactive true if the scene is interactive, false otherwise.
   */
  public void setInteractive(boolean interactive) {
    isInteractive = interactive;
    getSceneViews().forEach(sv -> sv.setAnimated(isInteractive));
  }
}
