// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.codeInspection.i18n.inconsistentResourceBundle;

import com.intellij.codeInspection.*;
import com.intellij.codeInspection.ui.MultipleCheckboxOptionsPanel;
import com.intellij.codeInspection.ui.OptionAccessor;
import com.intellij.lang.properties.PropertiesUtil;
import com.intellij.lang.properties.ResourceBundle;
import com.intellij.lang.properties.psi.PropertiesFile;
import com.intellij.openapi.util.*;
import com.intellij.psi.PsiFile;
import com.intellij.util.containers.BidirectionalMap;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.FactoryMap;
import org.jdom.Element;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.TestOnly;

import javax.swing.*;
import java.util.*;

public final class InconsistentResourceBundleInspection extends GlobalSimpleInspectionTool {
  private static final Key<Set<ResourceBundle>> VISITED_BUNDLES_KEY = Key.create("VISITED_BUNDLES_KEY");

  private final NotNullLazyValue<InconsistentResourceBundleInspectionProvider[]> myInspectionProviders = NotNullLazyValue.lazy(() -> {
    return new InconsistentResourceBundleInspectionProvider[]{
      new PropertiesKeysConsistencyInspectionProvider(),
      new DuplicatedPropertiesInspectionProvider(),
      new MissingTranslationsInspectionProvider(),
      new PropertiesPlaceholdersInspectionProvider(),
      new InconsistentPropertiesEndsInspectionProvider(),
    };
  });
  private final Map<String, Boolean> mySettings = new LinkedHashMap<>();


  @Override
  @NotNull
  public String getShortName() {
    return "InconsistentResourceBundle";
  }

  @Override
  public @NotNull JComponent createOptionsPanel() {
    final MultipleCheckboxOptionsPanel panel = new MultipleCheckboxOptionsPanel(new OptionAccessor() {
      @Override
      public boolean getOption(String optionName) {
        return isProviderEnabled(optionName);
      }

      @Override
      public void setOption(String optionName, boolean optionValue) {
        if (optionValue) {
          mySettings.remove(optionName);
        }
        else {
          mySettings.put(optionName, false);
        }
      }
    });
    for (final InconsistentResourceBundleInspectionProvider provider : myInspectionProviders.getValue()) {
      panel.addCheckbox(provider.getPresentableName(), provider.getName());
    }
    return panel;
  }

  @Override
  public void writeSettings(final @NotNull Element node) throws WriteExternalException {
    for (final Map.Entry<String, Boolean> e : mySettings.entrySet()) {
      JDOMExternalizerUtil.writeField(node, e.getKey(), Boolean.toString(e.getValue()));
    }
  }

  @Override
  public void readSettings(final @NotNull Element node) throws InvalidDataException {
    mySettings.clear();
    for (final Element e : node.getChildren()) {
      final String name = e.getAttributeValue("name");
      final boolean value = Boolean.parseBoolean(e.getAttributeValue("value"));
      mySettings.put(name, value);
    }
  }

  @Override
  public void inspectionStarted(@NotNull InspectionManager manager,
                                @NotNull GlobalInspectionContext globalContext,
                                @NotNull ProblemDescriptionsProcessor problemDescriptionsProcessor) {
    globalContext.putUserData(VISITED_BUNDLES_KEY, ContainerUtil.newConcurrentSet());
  }

  @Override
  public void checkFile(@NotNull PsiFile file,
                        @NotNull InspectionManager manager,
                        @NotNull ProblemsHolder problemsHolder,
                        @NotNull GlobalInspectionContext globalContext,
                        @NotNull ProblemDescriptionsProcessor problemDescriptionsProcessor) {
    Set<ResourceBundle> visitedBundles = globalContext.getUserData(VISITED_BUNDLES_KEY);
    if (!(file instanceof PropertiesFile)) return;
    final PropertiesFile propertiesFile = (PropertiesFile)file;
    ResourceBundle resourceBundle = propertiesFile.getResourceBundle();
    assert visitedBundles != null;
    if (!visitedBundles.add(resourceBundle)) return;
    List<PropertiesFile> files = resourceBundle.getPropertiesFiles();
    if (files.size() < 2) return;
    BidirectionalMap<PropertiesFile, PropertiesFile> parents = new BidirectionalMap<>();
    for (PropertiesFile f : files) {
      PropertiesFile parent = PropertiesUtil.getParent(f, files);
      if (parent != null) {
        parents.put(f, parent);
      }
    }
    final Map<PropertiesFile, Map<String, String>> propertiesFilesNamesMaps = FactoryMap.create(key -> key.getNamesMap());
    Map<PropertiesFile, Set<String>> keysUpToParent = new HashMap<>();
    for (PropertiesFile f : files) {
      Set<String> keys = new HashSet<>(propertiesFilesNamesMaps.get(f).keySet());
      PropertiesFile parent = parents.get(f);
      while (parent != null) {
        keys.addAll(propertiesFilesNamesMaps.get(parent).keySet());
        parent = parents.get(parent);
      }
      keysUpToParent.put(f, keys);
    }
    for (final InconsistentResourceBundleInspectionProvider provider : myInspectionProviders.getValue()) {
      if (isProviderEnabled(provider.getName())) {
        provider.check(parents, files, keysUpToParent, propertiesFilesNamesMaps, manager, globalContext.getRefManager(),
                       problemDescriptionsProcessor);
      }
    }
  }

  private boolean isProviderEnabled(final String providerName) {
    return ContainerUtil.getOrElse(mySettings, providerName, true);
  }

  @SafeVarargs
  @TestOnly
  public final void enableProviders(final Class<? extends InconsistentResourceBundleInspectionProvider>... providerClasses) {
    Set<Class<? extends InconsistentResourceBundleInspectionProvider>> providersToEnable = ContainerUtil.newHashSet(providerClasses);
    for (InconsistentResourceBundleInspectionProvider inspectionProvider : myInspectionProviders.getValue()) {
      if (providersToEnable.contains(inspectionProvider.getClass())) {
        mySettings.put(inspectionProvider.getName(), true);
      }
    }
  }

  @TestOnly
  public void disableAllProviders() {
    for (InconsistentResourceBundleInspectionProvider inspectionProvider : myInspectionProviders.getValue()) {
      mySettings.put(inspectionProvider.getName(), false);
    }
  }

  @TestOnly
  public void clearSettings() {
    mySettings.clear();
  }
}