// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.compiler.backwardRefs;

import com.intellij.compiler.CompilerDirectHierarchyInfo;
import com.intellij.compiler.CompilerReferenceService;
import com.intellij.compiler.backwardRefs.view.CompilerReferenceFindUsagesTestInfo;
import com.intellij.compiler.backwardRefs.view.CompilerReferenceHierarchyTestInfo;
import com.intellij.compiler.backwardRefs.view.DirtyScopeTestInfo;
import com.intellij.compiler.server.BuildManager;
import com.intellij.lang.injection.InjectedLanguageManager;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.application.ReadAction;
import com.intellij.openapi.compiler.CompileScope;
import com.intellij.openapi.compiler.CompilerManager;
import com.intellij.openapi.diagnostic.ControlFlowException;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.fileEditor.FileDocumentManager;
import com.intellij.openapi.fileTypes.FileType;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.progress.ProcessCanceledException;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ProjectFileIndex;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.openapi.roots.impl.LibraryScopeCache;
import com.intellij.openapi.util.Disposer;
import com.intellij.openapi.util.ModificationTracker;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.openapi.vfs.VirtualFileWithId;
import com.intellij.openapi.vfs.newvfs.ManagingFS;
import com.intellij.psi.PsiDocumentManager;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiNamedElement;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.SearchScope;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.PsiModificationTracker;
import com.intellij.psi.util.PsiUtilCore;
import com.intellij.util.containers.CollectionFactory;
import com.intellij.util.containers.ConcurrentFactoryMap;
import com.intellij.util.indexing.StorageException;
import com.intellij.util.messages.MessageBusConnection;
import it.unimi.dsi.fastutil.ints.IntCollection;
import it.unimi.dsi.fastutil.ints.IntIterator;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;
import org.jetbrains.jps.backwardRefs.CompilerRef;
import org.jetbrains.jps.backwardRefs.index.CompilerReferenceIndex;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

import static com.intellij.psi.search.GlobalSearchScope.getScopeRestrictedByFileTypes;
import static com.intellij.psi.search.GlobalSearchScope.notScope;

public abstract class CompilerReferenceServiceBase<Reader extends CompilerReferenceReader<?>> implements CompilerReferenceService, ModificationTracker {
  private static final Logger LOG = Logger.getInstance(CompilerReferenceServiceBase.class);

  private final Set<FileType> myFileTypes;
  private final DirtyScopeHolder myDirtyScopeHolder;
  private final ProjectFileIndex myProjectFileIndex;
  private final LongAdder myCompilationCount = new LongAdder();
  protected final ReentrantReadWriteLock myLock = new ReentrantReadWriteLock();
  protected final Lock myReadDataLock = myLock.readLock();
  private final Lock myOpenCloseLock = myLock.writeLock();
  protected final Project myProject;
  private final CompilerReferenceReaderFactory<? extends Reader> myReaderFactory;
  // index build start/finish callbacks are not ordered, so "build1 started" -> "build2 started" -> "build1 finished" -> "build2 finished" is expected sequence
  private int myActiveBuilds = 0;

  protected volatile Reader myReader;

  public CompilerReferenceServiceBase(Project project,
                                      CompilerReferenceReaderFactory<? extends Reader> readerFactory,
                                      BiConsumer<? super MessageBusConnection, ? super Set<String>> compilationAffectedModulesSubscription) {
    myProject = project;
    myReaderFactory = readerFactory;
    myProjectFileIndex = ProjectRootManager.getInstance(project).getFileIndex();
    myFileTypes = LanguageCompilerRefAdapter.EP_NAME.getExtensionList().stream().flatMap(a -> a.getFileTypes().stream()).collect(Collectors.toSet());
    myDirtyScopeHolder = new DirtyScopeHolder(this, FileDocumentManager.getInstance(), PsiDocumentManager.getInstance(project), compilationAffectedModulesSubscription);

    if (!CompilerReferenceService.isEnabled()) {
      LOG.error("CompilerReferenceService is disabled, but service was requested");
      return;
    }

    myDirtyScopeHolder.installVFSListener(project);
    if (!ApplicationManager.getApplication().isUnitTestMode()) {
      CompilerManager compilerManager = CompilerManager.getInstance(project);
      boolean isUpToDate;
      File buildDir = BuildManager.getInstance().getProjectSystemDirectory(project);

      boolean validIndexExists = buildDir != null
                                 && CompilerReferenceIndex.exists(buildDir)
                                 && !CompilerReferenceIndex.versionDiffers(buildDir, myReaderFactory.expectedIndexVersion());

      if (validIndexExists) {
        CompileScope projectCompileScope = compilerManager.createProjectCompileScope(project);
        isUpToDate = compilerManager.isUpToDate(projectCompileScope);
      }
      else {
        isUpToDate = false;
      }
      executeOnBuildThread(() -> {
        if (isUpToDate) {
          openReaderIfNeeded(IndexOpenReason.UP_TO_DATE_CACHE);
        }
        else {
          markAsOutdated(validIndexExists);
        }
      });
    }

    Disposer.register(project, () -> closeReaderIfNeeded(IndexCloseReason.PROJECT_CLOSED));
  }

  @Nullable
  @Override
  public GlobalSearchScope getScopeWithoutCodeReferences(@NotNull PsiElement element) {
    if (!isServiceEnabledFor(element)) return null;

    try {
      return CachedValuesManager.getCachedValue(element,
                                                () -> CachedValueProvider.Result.create(buildScopeWithoutReferences(getReferentFileIds(element)),
                                                  PsiModificationTracker.MODIFICATION_COUNT,
                                                  this));
    }
    catch (RuntimeException e1) {
      return onException(e1, "scope without code references");
    }
  }

  @Nullable
  @Override
  public GlobalSearchScope getScopeWithoutImplicitToStringCodeReferences(@NotNull PsiElement aClass) {
    if (!isServiceEnabledFor(aClass)) return null;

    try {
      return CachedValuesManager.getCachedValue(aClass,
                                                () -> CachedValueProvider.Result.create(
                                                  buildScopeWithoutReferences(getReferentFileIdsViaImplicitToString(aClass)),
                                                  PsiModificationTracker.MODIFICATION_COUNT,
                                                  this));
    }
    catch (RuntimeException e1) {
      return onException(e1, "scope without implicit toString references");
    }
  }

  @Nullable
  @Override
  public CompilerDirectHierarchyInfo getDirectInheritors(@NotNull PsiNamedElement aClass,
                                                         @NotNull GlobalSearchScope searchScope,
                                                         @NotNull FileType searchFileType) {
    return getHierarchyInfo(aClass, searchScope, searchFileType, CompilerHierarchySearchType.DIRECT_INHERITOR);
  }

  @Nullable
  @Override
  public CompilerDirectHierarchyInfo getFunExpressions(@NotNull PsiNamedElement functionalInterface,
                                                       @NotNull GlobalSearchScope searchScope,
                                                       @NotNull FileType searchFileType) {
    return getHierarchyInfo(functionalInterface, searchScope, searchFileType, CompilerHierarchySearchType.FUNCTIONAL_EXPRESSION);
  }

  @Nullable
  @Override
  public Integer getCompileTimeOccurrenceCount(@NotNull PsiElement element, boolean isConstructorSuggestion) {
    if (!isServiceEnabledFor(element)) return null;
    try {
      return CachedValuesManager.getCachedValue(element,
                                                () -> CachedValueProvider.Result.create(ConcurrentFactoryMap.createMap(
                                                  (Boolean constructorSuggestion) -> calculateOccurrenceCount(element,
                                                                                                    constructorSuggestion.booleanValue())),
                                                                                        PsiModificationTracker.MODIFICATION_COUNT,
                                                                                        this)).get(Boolean.valueOf(isConstructorSuggestion));
    }
    catch (RuntimeException e) {
      return onException(e, "weighting for completion");
    }
  }

  private Integer calculateOccurrenceCount(@NotNull PsiElement element, boolean isConstructorSuggestion) {
    LanguageCompilerRefAdapter adapter = null;
    if (isConstructorSuggestion) {
      adapter = ReadAction.compute(() -> LanguageCompilerRefAdapter.findAdapter(element));
      if (adapter == null || !adapter.isClass(element)) {
        return null;
      }
    }
    final CompilerElementInfo searchElementInfo = asCompilerElements(element, false, false);
    if (searchElementInfo == null) return null;

    if (!myReadDataLock.tryLock()) return null;
    try {
      if (myReader == null) return null;
      try {
        if (isConstructorSuggestion) {
          int constructorOccurrences = 0;
          for (PsiElement constructor : adapter.getInstantiableConstructors(element)) {
            final CompilerRef constructorRef = adapter.asCompilerRef(constructor, myReader.getNameEnumerator());
            if (constructorRef != null) {
              constructorOccurrences += myReader.getOccurrenceCount(constructorRef);
            }
          }
          final Integer anonymousCount = myReader.getAnonymousCount((CompilerRef.CompilerClassHierarchyElementDef)searchElementInfo.searchElements[0], searchElementInfo.place == ElementPlace.SRC);
          return anonymousCount == null ? constructorOccurrences : (constructorOccurrences + anonymousCount);
        } else {
          return myReader.getOccurrenceCount(searchElementInfo.searchElements[0]);
        }
      }
      catch (IOException e) {
        throw new RuntimeException(e);
      }
    } finally {
      myReadDataLock.unlock();
    }
  }

  @Nullable
  private CompilerHierarchyInfoImpl getHierarchyInfo(@NotNull PsiNamedElement aClass,
                                                     @NotNull GlobalSearchScope searchScope,
                                                     @NotNull FileType searchFileType,
                                                     @NotNull CompilerHierarchySearchType searchType) {
    if (!isServiceEnabledFor(aClass) || searchScope == LibraryScopeCache.getInstance(myProject).getLibrariesOnlyScope()) return null;

    try {
      Map<VirtualFile, SearchId[]> candidatesPerFile = ReadAction.compute(() -> {
        if (myProject.isDisposed()) throw new ProcessCanceledException();
        return CachedValuesManager.getCachedValue(aClass, () -> CachedValueProvider.Result.create(
          ConcurrentFactoryMap.createMap((HierarchySearchKey key) -> calculateDirectInheritors(aClass,
                                                                                               key.mySearchFileType,
                                                                                               key.mySearchType)),
          PsiModificationTracker.MODIFICATION_COUNT, this)).get(new HierarchySearchKey(searchType, searchFileType));
      });

      if (candidatesPerFile == null) return null;
      GlobalSearchScope dirtyScope = myDirtyScopeHolder.getDirtyScope();
      if (ElementPlace.LIB == ReadAction.compute(() -> ElementPlace.get(aClass.getContainingFile().getVirtualFile(), myProjectFileIndex))) {
        dirtyScope = dirtyScope.union(LibraryScopeCache.getInstance(myProject).getLibrariesOnlyScope());
      }
      return new CompilerHierarchyInfoImpl(candidatesPerFile, aClass, dirtyScope, searchScope, myProject, searchFileType, searchType);
    }
    catch (ProcessCanceledException e) {
      throw e;
    }
    catch (RuntimeException e) {
      return onException(e, "hierarchy");
    }
  }

  private boolean isServiceEnabledFor(PsiElement element) {
    if (!isActive()) return false;
    PsiFile file = ReadAction.compute(() -> element.getContainingFile());
    return file != null && !InjectedLanguageManager.getInstance(myProject).isInjectedFragment(file);
  }

  @Override
  public boolean isActive() {
    return myReader != null && CompilerReferenceService.isEnabled();
  }

  private Map<VirtualFile, SearchId[]> calculateDirectInheritors(@NotNull PsiElement aClass,
                                                                 @NotNull FileType searchFileType,
                                                                 @NotNull CompilerHierarchySearchType searchType) {
    SearchScope scope = aClass.getUseScope();
    if (!(scope instanceof GlobalSearchScope)) return null;
    final CompilerElementInfo searchElementInfo = asCompilerElements(aClass, false, true);
    if (searchElementInfo == null) return null;
    CompilerRef searchElement = searchElementInfo.searchElements[0];

    if (!myReadDataLock.tryLock()) return null;
    try {
      if (myReader == null) return null;
      try {
        return myReader.getDirectInheritors(searchElement, ((GlobalSearchScope)scope), myDirtyScopeHolder.getDirtyScope(), searchFileType, searchType);
      }
      catch (StorageException e) {
        throw new RuntimeException(e);
      }
    } finally {
      myReadDataLock.unlock();
    }
  }

  @Nullable
  private GlobalSearchScope buildScopeWithoutReferences(@Nullable IntSet referentFileIds) {
    if (referentFileIds == null) return null;

    return getScopeRestrictedByFileTypes(new ScopeWithoutReferencesOnCompilation(referentFileIds, myProjectFileIndex).intersectWith(notScope(
      myDirtyScopeHolder.getDirtyScope())),
                                         myFileTypes.toArray(FileType.EMPTY_ARRAY));
  }

  @Nullable
  private IntSet getReferentFileIds(@NotNull PsiElement element) {
    return getReferentFileIds(element, true, (ref, elementPlace) -> myReader.findReferentFileIds(ref, elementPlace == ElementPlace.SRC));
  }

  @Nullable
  private IntSet getReferentFileIdsViaImplicitToString(@NotNull PsiElement element) {
    return getReferentFileIds(element, false, (ref, elementPlace) -> myReader.findFileIdsWithImplicitToString(ref));
  }

  @Nullable
  private IntSet getReferentFileIds(@NotNull PsiElement element,
                                    boolean buildHierarchyForLibraryElements,
                                    @NotNull ReferentFileSearcher referentFileSearcher) {
    final CompilerElementInfo compilerElementInfo = asCompilerElements(element, buildHierarchyForLibraryElements, true);
    if (compilerElementInfo == null) return null;

    if (!myReadDataLock.tryLock()) return null;
    try {
      if (myReader == null) return null;
      IntSet referentFileIds = new IntOpenHashSet();
      for (CompilerRef ref : compilerElementInfo.searchElements) {
        try {
          IntSet referents = referentFileSearcher.findReferentFiles(ref, compilerElementInfo.place);
          if (referents == null) {
            return null;
          }
          referentFileIds.addAll(referents);
        }
        catch (StorageException e) {
          throw new RuntimeException(e);
        }
      }
      return referentFileIds;

    }
    finally {
      myReadDataLock.unlock();
    }
  }

  @Nullable
  private CompilerElementInfo asCompilerElements(@NotNull PsiElement psiElement,
                                                 boolean buildHierarchyForLibraryElements,
                                                 boolean checkNotDirty) {
    if (!myReadDataLock.tryLock()) return null;
    try {
      if (myReader == null) return null;
      VirtualFile file = PsiUtilCore.getVirtualFile(psiElement);
      if (file == null) return null;
      ElementPlace place = ElementPlace.get(file, myProjectFileIndex);
      if (checkNotDirty) {
        if (place == null || (place == ElementPlace.SRC && myDirtyScopeHolder.contains(file))) {
          return null;
        }
      }
      final LanguageCompilerRefAdapter adapter = LanguageCompilerRefAdapter.findAdapter(file);
      if (adapter == null) return null;
      final CompilerRef ref = adapter.asCompilerRef(psiElement, myReader.getNameEnumerator());
      if (ref == null) return null;
      if (place == ElementPlace.LIB && buildHierarchyForLibraryElements) {
        final List<CompilerRef> elements = adapter.getHierarchyRestrictedToLibraryScope(ref,
                                                                                        psiElement,
                                                                                        myReader.getNameEnumerator(),
                                                                                        LibraryScopeCache.getInstance(myProject)
                                                                                       .getLibrariesOnlyScope());
        final CompilerRef[] fullHierarchy = new CompilerRef[elements.size() + 1];
        fullHierarchy[0] = ref;
        int i = 1;
        for (CompilerRef element : elements) {
          fullHierarchy[i++] = element;
        }
        return new CompilerElementInfo(place, fullHierarchy);
      }
      else {
        return new CompilerElementInfo(place, ref);
      }
    }
    catch (IOException e) {
      throw new RuntimeException(e);
    } finally {
      myReadDataLock.unlock();
    }
  }

  protected void closeReaderIfNeeded(IndexCloseReason reason) {
    myOpenCloseLock.lock();
    try {
      if (reason == IndexCloseReason.COMPILATION_STARTED) {
        myActiveBuilds++;
        myDirtyScopeHolder.compilerActivityStarted();
      }
      if (myReader != null) {
        myReader.close(reason == IndexCloseReason.AN_EXCEPTION);
        myReader = null;
      }
    } finally {
      myOpenCloseLock.unlock();
    }
  }

  protected void openReaderIfNeeded(IndexOpenReason reason) {
    myCompilationCount.increment();
    myOpenCloseLock.lock();
    try {
      try {
        switch (reason) {
          case UP_TO_DATE_CACHE:
            myDirtyScopeHolder.upToDateChecked(true);
            break;
          case COMPILATION_FINISHED:
            myDirtyScopeHolder.compilerActivityFinished();
        }
      }
      catch (RuntimeException e) {
        --myActiveBuilds;
        throw e;
      }
      if ((--myActiveBuilds == 0) && myProject.isOpen()) {
        myReader = myReaderFactory.create(myProject);
        LOG.info("backward reference index reader " + (myReader == null ? "doesn't exist" : "is opened"));
      }
    }
    finally {
      myOpenCloseLock.unlock();
    }
  }

  private void markAsOutdated(boolean decrementBuildCount) {
    myOpenCloseLock.lock();
    try {
      if (decrementBuildCount) {
        --myActiveBuilds;
      }
      myDirtyScopeHolder.upToDateChecked(false);
    } finally {
      myOpenCloseLock.unlock();
    }
  }

  public ProjectFileIndex getFileIndex() {
    return myProjectFileIndex;
  }

  public Set<FileType> getFileTypes() {
    return myFileTypes;
  }

  public Project getProject() {
    return myProject;
  }

  protected static void executeOnBuildThread(@NotNull Runnable compilationFinished) {
    if (ApplicationManager.getApplication().isUnitTestMode()) {
      compilationFinished.run();
    }
    else {
      BuildManager.getInstance().runCommand(compilationFinished);
    }
  }

  protected enum ElementPlace {
    SRC, LIB;

    public static ElementPlace get(VirtualFile file, ProjectFileIndex index) {
      if (file == null) return null;
      return index.isInSourceContent(file) ? SRC : (index.isInLibrary(file) ? LIB : null);
    }
  }

  protected static final class ScopeWithoutReferencesOnCompilation extends GlobalSearchScope {
    private final IntSet myReferentIds;
    private final ProjectFileIndex myIndex;

    public ScopeWithoutReferencesOnCompilation(IntSet ids, ProjectFileIndex index) {
      myReferentIds = ids;
      myIndex = index;
    }

    @Override
    public boolean contains(@NotNull VirtualFile file) {
      return file instanceof VirtualFileWithId && myIndex.isInSourceContent(file) && !myReferentIds.contains(((VirtualFileWithId)file).getId());
    }

    @Override
    public boolean isSearchInModuleContent(@NotNull Module aModule) {
      return true;
    }

    @Override
    public boolean isSearchInLibraries() {
      return false;
    }
  }

  @Override
  public long getModificationCount() {
    return myCompilationCount.longValue();
  }

  protected static final class CompilerElementInfo {
    public final ElementPlace place;
    public final CompilerRef[] searchElements;

    public CompilerElementInfo(ElementPlace place, CompilerRef... searchElements) {
      this.place = place;
      this.searchElements = searchElements;
    }
  }

  protected static final class HierarchySearchKey {
    private final CompilerHierarchySearchType mySearchType;
    private final FileType mySearchFileType;

    public HierarchySearchKey(CompilerHierarchySearchType searchType, FileType searchFileType) {
      mySearchType = searchType;
      mySearchFileType = searchFileType;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;

      HierarchySearchKey key = (HierarchySearchKey)o;
      return mySearchType == key.mySearchType && mySearchFileType == key.mySearchFileType;
    }

    @Override
    public int hashCode() {
      return 31 * mySearchType.hashCode() + mySearchFileType.hashCode();
    }
  }

  @TestOnly
  @Nullable
  public Set<VirtualFile> getReferentFiles(@NotNull PsiElement element) {
    ManagingFS managingFS = ManagingFS.getInstance();
    IntSet ids = getReferentFileIds(element);
    if (ids == null) {
      return null;
    }
    Set<VirtualFile> fileSet = CollectionFactory.createSmallMemoryFootprintSet(ids.size());
    for (IntIterator iterator = ids.iterator(); iterator.hasNext(); ) {
      VirtualFile vFile = managingFS.findFileById(iterator.nextInt());
      assert vFile != null;
      fileSet.add(vFile);
    }
    return fileSet;
  }

  // should not be used in production code
  @NotNull
  DirtyScopeHolder getDirtyScopeHolder() {
    return myDirtyScopeHolder;
  }

  @Nullable
  public CompilerReferenceFindUsagesTestInfo getTestFindUsages(@NotNull PsiElement element) {
    if (!myReadDataLock.tryLock()) return null;
    try {
      IntSet referentFileIds = getReferentFileIds(element);
      DirtyScopeTestInfo dirtyScopeInfo = myDirtyScopeHolder.getState();
      return new CompilerReferenceFindUsagesTestInfo(referentFileIds, dirtyScopeInfo);
    }
    finally {
      myReadDataLock.unlock();
    }
  }

  @Nullable
  public CompilerReferenceHierarchyTestInfo getTestHierarchy(@NotNull PsiNamedElement element, @NotNull GlobalSearchScope scope, @NotNull FileType fileType) {
    if (!myReadDataLock.tryLock()) return null;
    try {
      final CompilerHierarchyInfoImpl hierarchyInfo = getHierarchyInfo(element, scope, fileType, CompilerHierarchySearchType.DIRECT_INHERITOR);
      final DirtyScopeTestInfo dirtyScopeInfo = myDirtyScopeHolder.getState();
      return new CompilerReferenceHierarchyTestInfo(hierarchyInfo, dirtyScopeInfo);
    }
    finally {
      myReadDataLock.unlock();
    }
  }

  @Nullable
  public CompilerReferenceHierarchyTestInfo getTestFunExpressions(@NotNull PsiNamedElement element, @NotNull GlobalSearchScope scope, @NotNull FileType fileType) {
    if (!myReadDataLock.tryLock()) return null;
    try {
      final CompilerHierarchyInfoImpl hierarchyInfo = getHierarchyInfo(element, scope, fileType, CompilerHierarchySearchType.FUNCTIONAL_EXPRESSION);
      final DirtyScopeTestInfo dirtyScopeInfo = myDirtyScopeHolder.getState();
      return new CompilerReferenceHierarchyTestInfo(hierarchyInfo, dirtyScopeInfo);
    }
    finally {
      myReadDataLock.unlock();
    }
  }

  @Nullable
  protected <T> T onException(@NotNull Exception e, @NotNull String actionName) {
    if (e instanceof ControlFlowException) {
      throw (RuntimeException)e;
    }

    LOG.error("an exception during " + actionName + " calculation", e);
    Throwable unwrapped = e instanceof RuntimeException ? e.getCause() : e;
    if (requireIndexRebuild(unwrapped)) {
      closeReaderIfNeeded(IndexCloseReason.AN_EXCEPTION);
    }
    return null;
  }

  @NotNull
  protected static IntSet intersection(@NotNull IntSet set1, @NotNull IntCollection set2) {
    if (set1.isEmpty()) {
      return set1;
    }

    IntSet result = ((IntOpenHashSet)set1).clone();
    result.retainAll(set2);
    return result;
  }

  private static boolean requireIndexRebuild(@Nullable Throwable exception) {
    return exception instanceof StorageException || exception instanceof IOException;
  }

  protected enum IndexCloseReason {
    AN_EXCEPTION,
    COMPILATION_STARTED,
    PROJECT_CLOSED
  }

  protected enum IndexOpenReason {
    COMPILATION_FINISHED,
    UP_TO_DATE_CACHE
  }

  @FunctionalInterface
  protected interface ReferentFileSearcher {
    @Nullable
    IntSet findReferentFiles(@NotNull CompilerRef ref, @NotNull ElementPlace place) throws StorageException;
  }
}
