// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.compiler.server.impl

import com.intellij.compiler.server.BuildProcessParametersProvider
import com.intellij.compiler.server.CompileServerPlugin
import com.intellij.ide.plugins.IdeaPluginDescriptor
import com.intellij.ide.plugins.PluginManagerCore
import com.intellij.openapi.Disposable
import com.intellij.openapi.application.PluginPathManager
import com.intellij.openapi.diagnostic.Logger
import com.intellij.openapi.project.Project
import com.intellij.openapi.util.io.FileUtil
import com.intellij.openapi.util.io.FileUtilRt
import com.intellij.openapi.util.text.StringUtil
import com.intellij.util.PathUtil
import com.intellij.util.io.URLUtil
import com.intellij.util.text.VersionComparatorUtil
import org.jetbrains.annotations.ApiStatus
import org.jetbrains.annotations.TestOnly
import org.jetbrains.jps.cmdline.ClasspathBootstrap
import java.io.File
import java.io.IOException
import java.nio.file.Files
import java.nio.file.Path
import java.util.*
import java.util.jar.Attributes
import java.util.jar.JarFile

class BuildProcessClasspathManager(parentDisposable: Disposable) {
  @Volatile
  private var compileServerPluginsClasspath: List<String>? = null

  private val lastClasspathLock = Any()
  private var lastRawClasspath: List<String>? = null
  private var lastFilteredClasspath: List<String>? = null

  init {
    CompileServerPlugin.EP_NAME.addChangeListener({ compileServerPluginsClasspath = null }, parentDisposable)
  }

  fun getBuildProcessClasspath(project: Project): List<String> {
    val rawClasspath = computeRawBuildProcessClasspath(project)
    synchronized(lastClasspathLock) {
      if (rawClasspath != lastRawClasspath) {
        lastRawClasspath = rawClasspath
        lastFilteredClasspath = filterOutOlderVersions(rawClasspath)
        if (LOG.isDebugEnabled && lastRawClasspath != lastFilteredClasspath) {
          LOG.debug("older versions of libraries were removed from classpath:")
          LOG.debug("original classpath: $lastRawClasspath")
          LOG.debug("actual classpath: $lastFilteredClasspath")
        }
      }
      return lastFilteredClasspath!!
    }
  }

  private fun computeRawBuildProcessClasspath(project: Project): List<String> {
    return ClasspathBootstrap.getBuildProcessApplicationClasspath() + getBuildProcessPluginsClasspath(project)
  }

  /**
   * For internal use only, use [getBuildProcessClasspath] to get full classpath instead.
   */
  @ApiStatus.Internal
  fun getBuildProcessPluginsClasspath(project: Project): List<String> {
    val dynamicClasspath = BuildProcessParametersProvider.EP_NAME.getExtensions(project).flatMapTo(ArrayList()) { it.classPath }
    return if (dynamicClasspath.isEmpty()) {
      staticClasspath
    }
    else {
      dynamicClasspath.addAll(staticClasspath)
      dynamicClasspath
    }
  }

  private val staticClasspath: List<String>
    get() {
      val classpath = compileServerPluginsClasspath ?: computeCompileServerPluginsClasspath()
      if (compileServerPluginsClasspath == null) {
        compileServerPluginsClasspath = classpath
      }
      return classpath
    }

  companion object {
    private val LOG = Logger.getInstance(BuildProcessClasspathManager::class.java)

    private fun findClassesRoot(relativePath: String, plugin: IdeaPluginDescriptor, baseFile: Path): String? {
      val jarFile = baseFile.resolve("lib/$relativePath")
      if (Files.exists(jarFile)) {
        return jarFile.toString()
      }

      // ... 'plugin run configuration': all module outputs are copied to 'classes' folder
      val classesDir = baseFile.resolve("classes")
      if (Files.isDirectory(classesDir)) {
        return classesDir.toString()
      }

      // development mode
      if (PluginManagerCore.isRunningFromSources()) {
        // ... try "out/classes/production/<module-name>", assuming that JAR name was automatically generated from module name
        val fileName = FileUtilRt.getNameWithoutExtension(PathUtil.getFileName(relativePath))
        val moduleName = OLD_TO_NEW_MODULE_NAME[fileName] ?:
                         //try restoring module name from JAR name automatically generated by BaseLayout.convertModuleNameToFileName
                         "intellij." + fileName.replace('-', '.')
        var baseOutputDir = baseFile.parent
        if (baseOutputDir.fileName.toString() == "test") {
          baseOutputDir = baseOutputDir.parent.resolve("production")
        }
        val moduleDir = baseOutputDir.resolve(moduleName)
        if (Files.isDirectory(moduleDir)) {
          return moduleDir.toString()
        }
        // ... try "<plugin-dir>/lib/<jar-name>", assuming that <jar-name> is a module library committed to VCS
        val pluginDir = getPluginDir(plugin)
        if (pluginDir != null) {
          val libraryFile = File(pluginDir, "lib/" + PathUtil.getFileName(relativePath))
          if (libraryFile.exists()) {
            return libraryFile.path
          }
        }
        // ... look for <jar-name> on the classpath, assuming that <jar-name> is an external (read: Maven) library
        try {
          val urls = BuildProcessClasspathManager::class.java.classLoader.getResources(JarFile.MANIFEST_NAME).asSequence()
          val jarPath = urls.mapNotNull { URLUtil.splitJarUrl(it.file)?.first }.firstOrNull { PathUtil.getFileName(it) == relativePath }
          if (jarPath != null) {
            return jarPath
          }
        }
        catch (ignored: IOException) {
        }
      }
      LOG.error("Cannot add '" + relativePath + "' from '" + plugin.name + ' ' + plugin.version + "'" + " to compiler classpath")
      return null
    }

    private fun computeCompileServerPluginsClasspath(): List<String> {
      val classpath = ArrayList<String>()
      for (serverPlugin in CompileServerPlugin.EP_NAME.extensions) {
        val pluginId = serverPlugin.pluginDescriptor.pluginId
        val plugin = PluginManagerCore.getPlugin(pluginId)
        LOG.assertTrue(plugin != null, pluginId)
        val baseFile = plugin!!.pluginPath
        if (Files.isRegularFile(baseFile)) {
          classpath.add(baseFile.toString())
        }
        else {
          StringUtil.split(serverPlugin.classpath, ";").mapNotNullTo(classpath) { findClassesRoot(it, plugin, baseFile)}
        }
      }
      return classpath
    }

    private fun getPluginDir(plugin: IdeaPluginDescriptor): File? {
      val pluginDirName = StringUtil.getShortName(plugin.pluginId.idString)
      val extraDir = System.getProperty("idea.external.build.development.plugins.dir")
      if (extraDir != null) {
        val extraDirFile = File(extraDir, pluginDirName)
        if (extraDirFile.isDirectory) {
          return extraDirFile
        }
      }
      var pluginHome = PluginPathManager.getPluginHome(pluginDirName)
      if (!pluginHome.isDirectory && StringUtil.isCapitalized(pluginDirName)) {
        pluginHome = PluginPathManager.getPluginHome(StringUtil.decapitalize(pluginDirName))
      }
      return if (pluginHome.isDirectory) pluginHome else null
    }

    private fun filterOutOlderVersions(classpath: List<String>): List<String> {
      data class JarInfo(val path: String, val title: String, val version: String)

      fun readTitleAndVersion(path: String): JarInfo? {
        val file = File(path)
        if (!file.isFile || !FileUtil.extensionEquals(file.name, "jar")) return null
        JarFile(file).use {
          val attributes = it.manifest?.mainAttributes ?: return null
          val title = attributes.getValue(Attributes.Name.IMPLEMENTATION_TITLE) ?: return null
          val version = attributes.getValue(Attributes.Name.IMPLEMENTATION_VERSION) ?: return null
          return JarInfo(path, title, version)
        }
      }

      val jarInfos = classpath.mapNotNull(::readTitleAndVersion)
      val titleToInfo = jarInfos.groupBy { it.title }
      val pathToInfo = jarInfos.associateBy { it.path }
      return classpath.filter { path ->
        val pathInfo = pathToInfo[path] ?: return@filter true
        val sameTitle = titleToInfo[pathInfo.title]
        if (sameTitle == null || sameTitle.size <= 1) return@filter true
        sameTitle.all { VersionComparatorUtil.compare(it.version, pathInfo.version) <= 0 }
      }
    }

    @JvmStatic @TestOnly
    fun filterOutOlderVersionsForTests(classpath: List<String>): List<String> = filterOutOlderVersions(classpath)

    @JvmStatic
    fun getLauncherClasspath(project: Project): List<String> {
      return BuildProcessParametersProvider.EP_NAME.getExtensions(project).flatMap { it.launcherClassPath }
    }

    //todo[nik] this is a temporary compatibility fix; we should update plugin layout so JAR names correspond to module names instead.
    private val OLD_TO_NEW_MODULE_NAME = mapOf(
      "android-jps-plugin" to "intellij.android.jpsBuildPlugin.jps",
      "android-jps-model" to "intellij.android.jps.model",
      "build-common" to "intellij.android.buildCommon",
      "sdk-common" to "android.sdktools.sdk-common",
      "sdklib" to "android.sdktools.sdklib",
      "layoutlib-api" to "android.sdktools.layoutlib-api",
      "repository" to "android.sdktools.repository",
      "manifest-merger" to "android.sdktools.manifest-merger",
    )
  }
}