// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.history.core;

import com.intellij.util.io.PagePool;
import com.intellij.util.io.storage.AbstractRecordsTable;
import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.nio.file.Path;

public final class LocalHistoryRecordsTable extends AbstractRecordsTable {
  private static final int VERSION = 4;

  private static final int LAST_ID_OFFSET = DEFAULT_HEADER_SIZE;
  private static final int FIRST_RECORD_OFFSET = LAST_ID_OFFSET + 8;
  private static final int LAST_RECORD_OFFSET = FIRST_RECORD_OFFSET + 4;
  private static final int FS_TIMESTAMP_OFFSET = LAST_RECORD_OFFSET + 4;
  private static final int HEADER_SIZE = FS_TIMESTAMP_OFFSET + 8;

  private static final int PREV_RECORD_OFFSET = DEFAULT_RECORD_SIZE;
  private static final int NEXT_RECORD_OFFSET = PREV_RECORD_OFFSET + 4;
  private static final int TIMESTAMP_OFFSET = NEXT_RECORD_OFFSET + 4;
  private static final int RECORD_SIZE = TIMESTAMP_OFFSET + 8;

  private static final byte[] ZEROS = new byte[RECORD_SIZE];

  public LocalHistoryRecordsTable(@NotNull Path storageFilePath, @NotNull PagePool pool) throws IOException {
    super(storageFilePath, pool);
  }

  @Override
  protected int getHeaderSize() {
    return HEADER_SIZE;
  }

  @Override
  protected int getRecordSize() {
    return RECORD_SIZE;
  }

  @Override
  protected int getImplVersion() {
    return VERSION;
  }

  @Override
  protected byte[] getZeros() {
    return ZEROS;
  }

  public long getLastId() {
    return myStorage.getLong(LAST_ID_OFFSET);
  }

  public void setLastId(long lastId) {
    markDirty();
    myStorage.putLong(LAST_ID_OFFSET, lastId);
  }

  public void setFirstRecord(int record) {
    markDirty();
    myStorage.putInt(FIRST_RECORD_OFFSET, record);
  }

  public int getFirstRecord() {
    return myStorage.getInt(FIRST_RECORD_OFFSET);
  }

  public void setLastRecord(int record) {
    markDirty();
    myStorage.putInt(LAST_RECORD_OFFSET, record);
  }

  public int getLastRecord() {
    return myStorage.getInt(LAST_RECORD_OFFSET);
  }

  public void setFSTimestamp(long timestamp) {
    markDirty();
    myStorage.putLong(FS_TIMESTAMP_OFFSET, timestamp);
  }

  public long getFSTimestamp() {
    return myStorage.getLong(FS_TIMESTAMP_OFFSET);
  }

  public void setPrevRecord(int record, int prevRecord) {
    markDirty();
    myStorage.putInt(getOffset(record, PREV_RECORD_OFFSET), prevRecord);
  }

  public int getPrevRecord(int record) {
    return myStorage.getInt(getOffset(record, PREV_RECORD_OFFSET));
  }

  public void setNextRecord(int record, int nextRecord) {
    markDirty();
    myStorage.putInt(getOffset(record, NEXT_RECORD_OFFSET), nextRecord);
  }

  public int getNextRecord(int record) {
    return myStorage.getInt(getOffset(record, NEXT_RECORD_OFFSET));
  }

  public void setTimestamp(int record, long timestamp) {
    markDirty();
    myStorage.putLong(getOffset(record, TIMESTAMP_OFFSET), timestamp);
  }

  public long getTimestamp(int record) {
    return myStorage.getLong(getOffset(record, TIMESTAMP_OFFSET));
  }
}

