// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.internal.statistic.collectors.fus.ui;

import com.intellij.internal.statistic.beans.MetricEvent;
import com.intellij.internal.statistic.eventLog.EventLogGroup;
import com.intellij.internal.statistic.eventLog.events.EventFields;
import com.intellij.internal.statistic.eventLog.events.EventId2;
import com.intellij.internal.statistic.service.fus.collectors.ApplicationUsagesCollector;
import com.intellij.openapi.editor.colors.EditorColorsManager;
import com.intellij.openapi.editor.colors.EditorColorsScheme;
import com.intellij.openapi.editor.colors.impl.AbstractColorsScheme;
import com.intellij.openapi.options.Scheme;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.ui.ColorUtil;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;

import java.util.HashSet;
import java.util.Set;

public class EditorColorSchemesUsagesCollector extends ApplicationUsagesCollector {

  private final static int CURR_VERSION = 3;

  public static final String SCHEME_NAME_OTHER = "Other";
  public final static String[] KNOWN_NAMES = {
    "Default",
    "Darcula",
    "Obsidian",
    "Visual Studio",
    "Solarized",
    "Wombat",
    "Monkai",
    "XCode",
    "Sublime",
    "Oblivion",
    "Zenburn",
    "Cobalt",
    "Netbeans",
    "Eclipse",
    "Aptana",
    "Flash Builder",
    "IdeaLight",
    "High contrast",
    "ReSharper",
    "Rider",
    "IntelliJ Light",
    SCHEME_NAME_OTHER
  };

  private static final EventLogGroup GROUP = new EventLogGroup("ui.editor.color.schemes", CURR_VERSION);
  private static final EventId2<String, Boolean> COLOR_SCHEME =
    GROUP.registerEvent(
      "enabled.color.scheme",
      EventFields.String("scheme", ContainerUtil.newArrayList(KNOWN_NAMES)),
      EventFields.Boolean("is_dark")
    );

  @Override
  public EventLogGroup getGroup() {
    return GROUP;
  }

  @NotNull
  @Override
  public Set<MetricEvent> getMetrics() {
    EditorColorsScheme currentScheme = EditorColorsManager.getInstance().getGlobalScheme();
    Set<MetricEvent> usages = new HashSet<>();
    if (currentScheme instanceof AbstractColorsScheme) {
      String schemeName = currentScheme.getName();
      if (schemeName.startsWith(Scheme.EDITABLE_COPY_PREFIX)) {
        EditorColorsScheme original = ((AbstractColorsScheme)currentScheme).getOriginal();
        if (original != null) {
          schemeName = original.getName();
        }
      }
      String scheme = getKnownSchemeName(schemeName);
      boolean isDark = ColorUtil.isDark(currentScheme.getDefaultBackground());
      usages.add(COLOR_SCHEME.metric(scheme, isDark));
    }
    return usages;
  }

  @NotNull
  private static String getKnownSchemeName(@NonNls @NotNull String schemeName) {
    for (@NonNls String knownName : KNOWN_NAMES) {
      if (StringUtil.toUpperCase(schemeName).contains(StringUtil.toUpperCase(knownName))) {
        return knownName;
      }
    }
    return SCHEME_NAME_OTHER;
  }
}
