// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.jarRepository.settings;

import com.intellij.ide.JavaUiBundle;
import com.intellij.openapi.ui.DialogBuilder;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.ui.CheckboxTree;
import com.intellij.ui.CheckboxTreeBase;
import com.intellij.ui.CheckedTreeNode;
import com.intellij.ui.SimpleTextAttributes;
import com.intellij.ui.components.JBScrollPane;
import com.intellij.util.ui.tree.TreeUtil;
import org.eclipse.aether.artifact.Artifact;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.idea.maven.aether.ArtifactDependencyNode;

import javax.swing.*;
import java.util.*;

class DependencyExclusionEditor {
  private final CheckboxTree myDependenciesTree;
  private final CheckedTreeNode myRootNode;
  private final JPanel myMainPanel;

  DependencyExclusionEditor(ArtifactDependencyNode root, JPanel parentComponent) {
    myMainPanel = parentComponent;
    myRootNode = createDependencyTreeNode(root);
    CheckboxTreeBase.CheckPolicy policy = new CheckboxTreeBase.CheckPolicy(false, true, true, false);
    myDependenciesTree = new CheckboxTree(new CheckboxTree.CheckboxTreeCellRenderer() {
      {
        myIgnoreInheritance = true;
      }

      @Override
      public void customizeRenderer(JTree tree,
                                    Object value,
                                    boolean selected,
                                    boolean expanded,
                                    boolean leaf,
                                    int row,
                                    boolean hasFocus) {
        if (!(value instanceof CheckedTreeNode)) return;

        Object userObject = ((CheckedTreeNode)value).getUserObject();
        if (!(userObject instanceof Artifact)) return;

        Artifact artifact = (Artifact)userObject;
        @NlsSafe final String groupArtifactFragment = artifact.getGroupId() + ":" + artifact.getArtifactId();
        getTextRenderer().append(groupArtifactFragment, SimpleTextAttributes.REGULAR_ATTRIBUTES, true);
        @NlsSafe final String versionFragment = ":" + artifact.getVersion();
        getTextRenderer().append(versionFragment, SimpleTextAttributes.GRAYED_ATTRIBUTES, true);
      }
    }, myRootNode, policy);
    myDependenciesTree.setRootVisible(false);
  }

  @Nullable
  public Set<String> selectExcludedDependencies(List<String> excludedDependencies) {
    uncheckExcludedNodes(myRootNode, new HashSet<>(excludedDependencies), false);
    TreeUtil.expandAll(myDependenciesTree);
    DialogBuilder dialogBuilder =
      new DialogBuilder(myMainPanel)
        .title(JavaUiBundle.message("dialog.title.include.transitive.dependencies"))
        .centerPanel(new JBScrollPane(myDependenciesTree));
    dialogBuilder.setPreferredFocusComponent(myDependenciesTree);

    if (dialogBuilder.showAndGet()) {
      return collectUncheckedNodes(myRootNode, new LinkedHashSet<>());
    }
    return null;
  }

  private static void uncheckExcludedNodes(CheckedTreeNode node, Set<String> excluded, boolean parentIsExcluded) {
    boolean isExcluded = parentIsExcluded || excluded.contains(getGroupAndArtifactId(node));
    node.setChecked(!isExcluded);
    Enumeration children = node.children();
    while (children.hasMoreElements()) {
      Object child = children.nextElement();
      uncheckExcludedNodes((CheckedTreeNode)child, excluded, isExcluded);
    }
  }

  private static Set<String> collectUncheckedNodes(CheckedTreeNode node, Set<String> result) {
    if (node.isChecked()) {
      Enumeration children = node.children();
      while (children.hasMoreElements()) {
        Object child = children.nextElement();
        collectUncheckedNodes((CheckedTreeNode)child, result);
      }
    }
    else {
      result.add(getGroupAndArtifactId(node));
    }
    return result;
  }

  @NotNull
  private static String getGroupAndArtifactId(CheckedTreeNode node) {
    Artifact artifact = (Artifact)node.getUserObject();
    return artifact.getGroupId() + ":" + artifact.getArtifactId();
  }

  @NotNull
  private static CheckedTreeNode createDependencyTreeNode(ArtifactDependencyNode node) {
    CheckedTreeNode treeNode = new CheckedTreeNode(node.getArtifact());
    for (ArtifactDependencyNode dependency : node.getDependencies()) {
      treeNode.add(createDependencyTreeNode(dependency));
    }
    return treeNode;
  }
}
