/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.junit4;

import com.intellij.rt.execution.junit.ComparisonFailureData;
import com.intellij.rt.execution.testFrameworks.AbstractExpectedPatterns;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

public class ExpectedPatterns extends AbstractExpectedPatterns {
  private static final List<Pattern> PATTERNS = new ArrayList<Pattern>();

  private static final String[] PATTERN_STRINGS = new String[]{
    "\nexpected: is \"(.*)\"\n\\s*got: \"(.*)\"\n",
    "\nexpected: is \"(.*)\"\n\\s*but:\\s+was \"(.*)\"",
    "\nexpected: (.*)\n\\s*got: (.*)",
    "expected same:<(.*)> was not:<(.*)>",
    "\nexpected: \"(.*)\"\n\\s*but: was \"(.*)\"",
    "expected: (.*)\n\\s*but: was (.*)",
    "expected: (.*)\\s+but was: (.*)",
    "expecting:\\s+<(.*)> to be equal to:\\s+<(.*)>\\s+but was not"
  };

  private static final String MESSAGE_LENGTH_FOR_PATTERN_MATCHING = "idea.junit.message.length.threshold";
  private static final String JUNIT_FRAMEWORK_COMPARISON_NAME = "junit.framework.ComparisonFailure";
  private static final String ORG_JUNIT_COMPARISON_NAME = "org.junit.ComparisonFailure";

  static {
    registerPatterns(PATTERN_STRINGS, PATTERNS);
  }

  public static ComparisonFailureData createExceptionNotification(String message) {
    return createExceptionNotification(message, PATTERNS);
  }

  public static ComparisonFailureData createExceptionNotification(Throwable assertion) {
    if (isComparisonFailure(assertion)) {
      return ComparisonFailureData.create(assertion);
    }
    try {
      final Throwable cause = assertion.getCause();
      if (isComparisonFailure(cause)) {
        return ComparisonFailureData.create(cause);
      }
    }
    catch (Throwable ignore) {
    }

    final String message = assertion.getMessage();
    if (message != null  && acceptedByThreshold(message.length())) {
      try {

        return createExceptionNotification(message);
      }
      catch (Throwable ignored) {}
    }
    return null;
  }

  private static boolean isComparisonFailure(Throwable throwable) {
    if (throwable == null) return false;
    return isComparisonFailure(throwable.getClass());
  }

  private static boolean isComparisonFailure(Class<?> aClass) {
    if (aClass == null) return false;
    final String throwableClassName = aClass.getName();
    if (throwableClassName.equals(JUNIT_FRAMEWORK_COMPARISON_NAME) ||
        throwableClassName.equals(ORG_JUNIT_COMPARISON_NAME) ||
        throwableClassName.equals(ComparisonFailureData.OPENTEST4J_ASSERTION)) {
      return true;
    }
    return isComparisonFailure(aClass.getSuperclass());
  }


  private static boolean acceptedByThreshold(int messageLength) {
    int threshold = 10000;
    try {
      final String property = System.getProperty(MESSAGE_LENGTH_FOR_PATTERN_MATCHING);
      if (property != null) {
        try {
          threshold = Integer.parseInt(property);
        }
        catch (NumberFormatException ignore) {}
      }
    }
    catch (SecurityException ignored) {}
    return messageLength < threshold;
  }
}
