/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.openapi.diff.impl.patch;

import com.intellij.openapi.vcs.FileStatus;
import org.jetbrains.annotations.Nullable;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author yole
 */
public class TextFilePatch extends FilePatch {
  private final Charset myCharset;
  @Nullable private final String myLineSeparator;
  private final List<PatchHunk> myHunks;
  @Nullable private FileStatus myFileStatus;

  public TextFilePatch(@Nullable Charset charset) {
    this(charset, null);
  }

  public TextFilePatch(@Nullable Charset charset, @Nullable String lineSeparator) {
    myCharset = charset;
    myLineSeparator = lineSeparator;
    myHunks = new ArrayList<>();
  }

  public TextFilePatch pathsOnlyCopy() {
    return new TextFilePatch(this);
  }

  private TextFilePatch(final TextFilePatch patch) {
    myCharset = patch.myCharset;
    setBeforeVersionId(patch.getBeforeVersionId());
    setAfterVersionId(patch.getAfterVersionId());
    setBeforeName(patch.getBeforeName());
    setAfterName(patch.getAfterName());
    myHunks = patch.myHunks;
    myLineSeparator = patch.getLineSeparator();
    setNewFileMode(patch.getNewFileMode());
    setFileStatus(patch.myFileStatus);
  }

  public void addHunk(final PatchHunk hunk) {
    myHunks.add(hunk);
  }

  public List<PatchHunk> getHunks() {
    return Collections.unmodifiableList(myHunks);
  }

  public boolean hasNoModifiedContent() {
    return myHunks.isEmpty();
  }

  @Override
  public boolean isNewFile() {
    return myFileStatus == FileStatus.ADDED || (myHunks.size() == 1 && myHunks.get(0).isNewContent());
  }

  public String getSingleHunkPatchText() {
    if (myHunks.isEmpty()) return "";     // file can be empty, only status changed
    assert myHunks.size() == 1;
    return myHunks.get(0).getText();
  }

  @Override
  public boolean isDeletedFile() {
    return myFileStatus == FileStatus.DELETED || (myHunks.size() == 1 && myHunks.get(0).isDeletedContent());
  }

  @Nullable
  public Charset getCharset() {
    return myCharset;
  }

  @Nullable
  public String getLineSeparator() {
    return myLineSeparator;
  }

  public void setFileStatus(@Nullable FileStatus fileStatus) {
    myFileStatus = fileStatus;
  }
}
