// Copyright 2000-2021 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.sh.codeStyle;

import com.intellij.application.options.CodeStyleAbstractPanel;
import com.intellij.icons.AllIcons;
import com.intellij.ide.highlighter.HighlighterFactory;
import com.intellij.openapi.editor.colors.EditorColorsScheme;
import com.intellij.openapi.editor.highlighter.EditorHighlighter;
import com.intellij.openapi.fileChooser.FileChooserDescriptorFactory;
import com.intellij.openapi.fileTypes.FileType;
import com.intellij.openapi.fileTypes.SyntaxHighlighter;
import com.intellij.openapi.fileTypes.SyntaxHighlighterFactory;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.project.ProjectUtil;
import com.intellij.openapi.ui.TextFieldWithBrowseButton;
import com.intellij.psi.codeStyle.CodeStyleConstraints;
import com.intellij.psi.codeStyle.CodeStyleSettings;
import com.intellij.psi.codeStyle.CommonCodeStyleSettings;
import com.intellij.sh.ShBundle;
import com.intellij.sh.ShFileType;
import com.intellij.sh.ShLanguage;
import com.intellij.sh.formatter.ShShfmtFormatterUtil;
import com.intellij.sh.settings.ShSettings;
import com.intellij.ui.DocumentAdapter;
import com.intellij.ui.JBColor;
import com.intellij.ui.components.ActionLink;
import com.intellij.ui.components.fields.IntegerField;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import javax.swing.event.DocumentEvent;

public class ShCodeStylePanel extends CodeStyleAbstractPanel {
  private JPanel myPanel;
  private JPanel myRightPanel;
  private JPanel myWarningPanel;

  private JCheckBox myTabCharacter;
  private IntegerField myIndentField;
  private IntegerField myTabField;
  private JLabel myWarningLabel;
  private JLabel myErrorLabel;

  private JCheckBox myBinaryOpsStartLine;
  private JCheckBox mySwitchCasesIndented;
  private JCheckBox myRedirectFollowedBySpace;
  private JCheckBox myKeepColumnAlignmentPadding;
  private JCheckBox myMinifyProgram;
  private JCheckBox myUnixLineSeparator;

  @SuppressWarnings("unused")
  private ActionLink myShfmtDownloadLink;
  private TextFieldWithBrowseButton myShfmtPathSelector;

  ShCodeStylePanel(CodeStyleSettings currentSettings, CodeStyleSettings settings) {
    super(ShLanguage.INSTANCE, currentSettings, settings);
    installPreviewPanel(myRightPanel);

    Project project = ProjectUtil.guessCurrentProject(getPanel());
    myShfmtPathSelector.addBrowseFolderListener(ShBundle.message("sh.code.style.choose.path"), "", project, FileChooserDescriptorFactory.createSingleFileDescriptor());
    myShfmtPathSelector.getTextField().getDocument().addDocumentListener(new DocumentAdapter() {
      @Override
      protected void textChanged(@NotNull DocumentEvent documentEvent) {
        myWarningPanel.setVisible(!ShShfmtFormatterUtil.isValidPath(myShfmtPathSelector.getText()));
      }
    });

    myWarningLabel.setIcon(AllIcons.General.Warning);
    myErrorLabel.setForeground(JBColor.RED);

    myBinaryOpsStartLine.setText(ShBundle.message("sh.code.style.binary.ops.like.and.may.start.a.line"));
    mySwitchCasesIndented.setText(ShBundle.message("sh.code.style.switch.cases.will.be.indented"));
    myRedirectFollowedBySpace.setText(ShBundle.message("sh.code.style.redirect.operators.will.be.followed.by.a.space"));
    myKeepColumnAlignmentPadding.setText(ShBundle.message("sh.code.style.keep.column.alignment.padding"));
    myMinifyProgram.setText(ShBundle.message("sh.code.style.minify.program.to.reduce.its.size"));
    myUnixLineSeparator.setText(ShBundle.message("sh.code.style.unix.line.separator"));

    addPanelToWatch(myPanel);
  }

  private void createUIComponents() {
    myIndentField = new IntegerField(null, CodeStyleConstraints.MIN_INDENT_SIZE, CodeStyleConstraints.MAX_INDENT_SIZE);
    myTabField = new IntegerField(null, CodeStyleConstraints.MIN_TAB_SIZE, CodeStyleConstraints.MAX_TAB_SIZE);
    myShfmtDownloadLink = new ActionLink(ShBundle.message("sh.code.style.download.link"), e -> {
        ShShfmtFormatterUtil.download(ProjectUtil.guessCurrentProject(getPanel()),
                                      () -> myShfmtPathSelector.setText(ShSettings.getShfmtPath()),
                                      () -> myErrorLabel.setVisible(true));
    });
  }

  @Override
  protected int getRightMargin() {
    return 0;
  }

  @Nullable
  @Override
  protected EditorHighlighter createHighlighter(EditorColorsScheme scheme) {
    SyntaxHighlighter highlighter = SyntaxHighlighterFactory.getSyntaxHighlighter(ShLanguage.INSTANCE, null, null);
    return HighlighterFactory.createHighlighter(highlighter, scheme);
  }

  @NotNull
  @Override
  protected FileType getFileType() {
    return ShFileType.INSTANCE;
  }

  @Nullable
  @Override
  protected String getPreviewText() {
    return GENERAL_CODE_SAMPLE;
  }

  @Override
  public void apply(CodeStyleSettings settings) {
    CommonCodeStyleSettings.IndentOptions indentOptions = settings.getLanguageIndentOptions(ShLanguage.INSTANCE);
    indentOptions.INDENT_SIZE = myIndentField.getValue();
    indentOptions.TAB_SIZE = myTabField.getValue();
    indentOptions.USE_TAB_CHARACTER = myTabCharacter.isSelected();

    ShCodeStyleSettings shSettings = settings.getCustomSettings(ShCodeStyleSettings.class);
    shSettings.BINARY_OPS_START_LINE = myBinaryOpsStartLine.isSelected();
    shSettings.SWITCH_CASES_INDENTED = mySwitchCasesIndented.isSelected();
    shSettings.REDIRECT_FOLLOWED_BY_SPACE = myRedirectFollowedBySpace.isSelected();
    shSettings.KEEP_COLUMN_ALIGNMENT_PADDING = myKeepColumnAlignmentPadding.isSelected();
    shSettings.MINIFY_PROGRAM = myMinifyProgram.isSelected();
    shSettings.USE_UNIX_LINE_SEPARATOR = myUnixLineSeparator.isSelected();
    ShSettings.setShfmtPath(myShfmtPathSelector.getText());
    myWarningPanel.setVisible(!ShShfmtFormatterUtil.isValidPath(myShfmtPathSelector.getText()));
    myErrorLabel.setVisible(false);
  }

  @Override
  public boolean isModified(CodeStyleSettings settings) {
    CommonCodeStyleSettings.IndentOptions indentOptions = settings.getLanguageIndentOptions(ShLanguage.INSTANCE);
    ShCodeStyleSettings shSettings = settings.getCustomSettings(ShCodeStyleSettings.class);

    return isFieldModified(myBinaryOpsStartLine, shSettings.BINARY_OPS_START_LINE)
        || isFieldModified(mySwitchCasesIndented, shSettings.SWITCH_CASES_INDENTED)
        || isFieldModified(myRedirectFollowedBySpace, shSettings.REDIRECT_FOLLOWED_BY_SPACE)
        || isFieldModified(myKeepColumnAlignmentPadding, shSettings.KEEP_COLUMN_ALIGNMENT_PADDING)
        || isFieldModified(myMinifyProgram, shSettings.MINIFY_PROGRAM)
        || isFieldModified(myUnixLineSeparator, shSettings.USE_UNIX_LINE_SEPARATOR)
        || isFieldModified(myTabCharacter, indentOptions.USE_TAB_CHARACTER)
        || isFieldModified(myIndentField, indentOptions.INDENT_SIZE)
        || isFieldModified(myTabField, indentOptions.TAB_SIZE)
        || isFieldModified(myShfmtPathSelector, ShSettings.getShfmtPath());
  }

  @Nullable
  @Override
  public JComponent getPanel() {
    return myPanel;
  }

  @Override
  protected void resetImpl(CodeStyleSettings settings) {
    CommonCodeStyleSettings.IndentOptions indentOptions = settings.getLanguageIndentOptions(ShLanguage.INSTANCE);
    myIndentField.setValue(indentOptions.INDENT_SIZE);
    myTabField.setValue(indentOptions.TAB_SIZE);
    myTabCharacter.setSelected(indentOptions.USE_TAB_CHARACTER);

    ShCodeStyleSettings shSettings = settings.getCustomSettings(ShCodeStyleSettings.class);
    myBinaryOpsStartLine.setSelected(shSettings.BINARY_OPS_START_LINE);
    mySwitchCasesIndented.setSelected(shSettings.SWITCH_CASES_INDENTED);
    myRedirectFollowedBySpace.setSelected(shSettings.REDIRECT_FOLLOWED_BY_SPACE);
    myKeepColumnAlignmentPadding.setSelected(shSettings.KEEP_COLUMN_ALIGNMENT_PADDING);
    myMinifyProgram.setSelected(shSettings.MINIFY_PROGRAM);
    myUnixLineSeparator.setSelected(shSettings.USE_UNIX_LINE_SEPARATOR);
    myShfmtPathSelector.setText(ShSettings.getShfmtPath());
    myWarningPanel.setVisible(!ShShfmtFormatterUtil.isValidPath(ShSettings.getShfmtPath()));
    myErrorLabel.setVisible(false);
  }

  private static boolean isFieldModified(@NotNull JCheckBox checkBox, boolean value) {
    return checkBox.isSelected() != value;
  }

  private static boolean isFieldModified(@NotNull IntegerField textField, int value) {
    return textField.getValue() != value;
  }

  private static boolean isFieldModified(@NotNull TextFieldWithBrowseButton browseButton, String value) {
    return !browseButton.getText().equals(value);
  }

  @NonNls private static final String GENERAL_CODE_SAMPLE = "#!/usr/bin/env sh\n" +
                                                            "\n" +
                                                            "function foo() {\n" +
                                                            "  if [ -x $file ]; then\n" +
                                                            "    myArray=(item1 item2 item3)\n" +
                                                            "  elif [ $file1 -nt $file2 ]; then\n" +
                                                            "    unset myArray\n" +
                                                            "  else\n" +
                                                            "    echo \"Usage: $0 file ...\"\n" +
                                                            "  fi\n" +
                                                            "}\n" +
                                                            "\n" +
                                                            "for (( i = 0; i < 5; i++ )); do\n" +
                                                            "  read -p r\n" +
                                                            "  print -n $r\n" +
                                                            "  wait $!\n" +
                                                            "done\n";
}
