// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.terminal;

import com.intellij.execution.filters.Filter;
import com.intellij.execution.filters.HyperlinkInfo;
import com.intellij.openapi.Disposable;
import com.intellij.openapi.actionSystem.DataKey;
import com.intellij.openapi.actionSystem.DataProvider;
import com.intellij.openapi.application.ReadAction;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.progress.ProcessCanceledException;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Disposer;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.wm.ToolWindow;
import com.intellij.ui.SearchTextField;
import com.intellij.ui.components.JBScrollBar;
import com.intellij.ui.components.JBScrollPane;
import com.intellij.util.LineSeparator;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.ui.JBSwingUtilities;
import com.intellij.util.ui.RegionPainter;
import com.jediterm.terminal.SubstringFinder;
import com.jediterm.terminal.TerminalColor;
import com.jediterm.terminal.TtyConnector;
import com.jediterm.terminal.model.SelectionUtil;
import com.jediterm.terminal.model.StyleState;
import com.jediterm.terminal.model.TerminalSelection;
import com.jediterm.terminal.model.TerminalTextBuffer;
import com.jediterm.terminal.model.hyperlinks.LinkInfo;
import com.jediterm.terminal.model.hyperlinks.LinkResult;
import com.jediterm.terminal.model.hyperlinks.LinkResultItem;
import com.jediterm.terminal.ui.JediTermWidget;
import com.jediterm.terminal.ui.TerminalAction;
import com.jediterm.terminal.ui.TerminalPanel;
import com.jediterm.terminal.ui.settings.SettingsProvider;
import com.jediterm.terminal.util.Pair;
import org.jetbrains.annotations.Nls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import javax.swing.event.DocumentListener;
import java.awt.*;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.List;

public class JBTerminalWidget extends JediTermWidget implements Disposable, DataProvider {
  public static final DataKey<String> SELECTED_TEXT_DATA_KEY = DataKey.create(JBTerminalWidget.class.getName() + " selected text");
  public static final DataKey<JBTerminalWidget> TERMINAL_DATA_KEY = DataKey.create(JBTerminalWidget.class.getName());
  private static final Logger LOG = Logger.getInstance(JBTerminalWidget.class);

  private final CompositeFilterWrapper myCompositeFilterWrapper;
  private JBTerminalWidgetListener myListener;

  public JBTerminalWidget(@NotNull Project project,
                          @NotNull JBTerminalSystemSettingsProviderBase settingsProvider,
                          @NotNull Disposable parent) {
    this(project, 80, 24, settingsProvider, null, parent);
  }

  public JBTerminalWidget(@NotNull Project project,
                          int columns,
                          int lines,
                          @NotNull JBTerminalSystemSettingsProviderBase settingsProvider,
                          @Nullable TerminalExecutionConsole console,
                          @NotNull Disposable parent) {
    super(columns, lines, settingsProvider);
    myCompositeFilterWrapper = new CompositeFilterWrapper(project, console, this);
    addHyperlinkFilter(line -> runFilters(project, line));
    setName("terminal");
    Disposer.register(parent, this);
    setFocusTraversalPolicy(new DefaultFocusTraversalPolicy() {
      @Override
      public Component getDefaultComponent(Container aContainer) {
        return getTerminalPanel();
      }
    });
  }

  @Nullable
  private LinkResult runFilters(@NotNull Project project, @NotNull String line) {
    Filter.Result r = ReadAction.compute(() -> {
      if (project.isDisposed()) {
        return null;
      }
      try {
        return myCompositeFilterWrapper.getCompositeFilter().applyFilter(line, line.length());
      }
      catch (ProcessCanceledException e) {
        if (LOG.isDebugEnabled()) {
          LOG.debug("Skipping running filters on " + line, e);
        }
        return null;
      }
    });
    if (r != null) {
      return new LinkResult(ContainerUtil.mapNotNull(r.getResultItems(), item -> convertResultItem(project, item)));
    }
    return null;
  }

  @Nullable
  private static LinkResultItem convertResultItem(@NotNull Project project, @NotNull Filter.ResultItem item) {
    HyperlinkInfo info = item.getHyperlinkInfo();
    if (info != null) {
      return new LinkResultItem(item.getHighlightStartOffset(), item.getHighlightEndOffset(),
                                new LinkInfo(() -> info.navigate(project)));
    }
    return null;
  }

  public JBTerminalWidgetListener getListener() {
    return myListener;
  }

  public void setListener(JBTerminalWidgetListener listener) {
    myListener = listener;
  }

  @Override
  protected JBTerminalPanel createTerminalPanel(@NotNull SettingsProvider settingsProvider,
                                                @NotNull StyleState styleState,
                                                @NotNull TerminalTextBuffer textBuffer) {
    JBTerminalPanel panel = new JBTerminalPanel((JBTerminalSystemSettingsProviderBase)settingsProvider, textBuffer, styleState);

    Disposer.register(this, panel);
    return panel;
  }

  @Override
  public @NotNull JBTerminalPanel getTerminalPanel() {
    return (JBTerminalPanel)super.getTerminalPanel();
  }

  @Override
  protected Graphics getComponentGraphics(Graphics graphics) {
    return JBSwingUtilities.runGlobalCGTransform(this, super.getComponentGraphics(graphics));
  }

  @Override
  protected JScrollBar createScrollBar() {
    JBScrollBar bar = new JBScrollBar() {
      @Override
      public Color getBackground() {
        return myTerminalPanel.getBackground();
      }
    };
    bar.setOpaque(true);
    bar.putClientProperty(JBScrollPane.Alignment.class, JBScrollPane.Alignment.RIGHT);
    bar.putClientProperty(JBScrollBar.TRACK, (RegionPainter<Object>)(g, x, y, width, height, object) -> {
      SubstringFinder.FindResult result = myTerminalPanel.getFindResult();
      if (result != null) {
        int modelHeight = bar.getModel().getMaximum() - bar.getModel().getMinimum();

        TerminalColor backgroundColor = mySettingsProvider.getFoundPatternColor().getBackground();
        if (backgroundColor != null) {
          g.setColor(mySettingsProvider.getTerminalColorPalette().getBackground(backgroundColor));
        }
        int anchorHeight = Math.max(2, height / modelHeight);
        for (SubstringFinder.FindResult.FindItem r : result.getItems()) {
          int where = height * r.getStart().y / modelHeight;
          g.fillRect(x, y + where, width, anchorHeight);
        }
      }
    });
    return bar;
  }

  @Override
  public List<TerminalAction> getActions() {
    List<TerminalAction> actions = super.getActions();
    if (isInTerminalToolWindow()) {
      JBTerminalSystemSettingsProviderBase settingsProvider = getSettingsProvider();
      actions.add(new TerminalAction(settingsProvider.getNewSessionActionPresentation(), input -> {
        myListener.onNewSession();
        return true;
      }).withMnemonicKey(KeyEvent.VK_T).withEnabledSupplier(() -> myListener != null));
      actions.add(new TerminalAction(settingsProvider.getCloseSessionActionPresentation(), input -> {
        myListener.onSessionClosed();
        return true;
      }).withMnemonicKey(KeyEvent.VK_T).withEnabledSupplier(() -> myListener != null));

      actions.add(TerminalSplitAction.create(true, myListener).withMnemonicKey(KeyEvent.VK_V).separatorBefore(true));
      actions.add(TerminalSplitAction.create(false, myListener).withMnemonicKey(KeyEvent.VK_H));
      if (myListener != null && myListener.isGotoNextSplitTerminalAvailable()) {
        actions.add(settingsProvider.getGotoNextSplitTerminalAction(myListener, true));
        actions.add(settingsProvider.getGotoNextSplitTerminalAction(myListener, false));
      }
      actions.add(new TerminalAction(settingsProvider.getPreviousTabActionPresentation(), input -> {
        myListener.onPreviousTabSelected();
        return true;
      }).withMnemonicKey(KeyEvent.VK_T).withEnabledSupplier(() -> myListener != null));
      actions.add(new TerminalAction(settingsProvider.getNextTabActionPresentation(), input -> {
        myListener.onNextTabSelected();
        return true;
      }).withMnemonicKey(KeyEvent.VK_T).withEnabledSupplier(() -> myListener != null));
      actions.add(new TerminalAction(settingsProvider.getMoveTabRightActionPresentation(), input -> {
        myListener.moveTabRight();
        return true;
      }).withMnemonicKey(KeyEvent.VK_R).withEnabledSupplier(() -> myListener != null && myListener.canMoveTabRight()));
      actions.add(new TerminalAction(settingsProvider.getMoveTabLeftActionPresentation(), input -> {
        myListener.moveTabLeft();
        return true;
      }).withMnemonicKey(KeyEvent.VK_L).withEnabledSupplier(() -> myListener != null && myListener.canMoveTabLeft()));
      actions.add(new TerminalAction(settingsProvider.getShowTabsActionPresentation(), input -> {
        myListener.showTabs();
        return true;
      }).withMnemonicKey(KeyEvent.VK_T).withEnabledSupplier(() -> myListener != null));
    }
    return actions;
  }

  private boolean isInTerminalToolWindow() {
    return isTerminalToolWindow(getTerminalPanel().getContextToolWindow());
  }

  static boolean isTerminalToolWindow(@Nullable ToolWindow toolWindow) {
    return toolWindow != null && "Terminal".equals(toolWindow.getId()); // TerminalToolWindowFactory.TOOL_WINDOW_ID is not visible here
  }

  @Override
  public void dispose() {
    close();
  }

  @Override
  protected SearchComponent createSearchComponent() {
    return new SearchComponent() {
      private final SearchTextField myTextField = new SearchTextField(false);

      @Override
      public String getText() {
        return myTextField.getText();
      }

      @Override
      public boolean ignoreCase() {
        return false;
      }

      @Override
      public JComponent getComponent() {
        myTextField.setOpaque(false);
        return myTextField;
      }

      @Override
      public void addDocumentChangeListener(DocumentListener listener) {
        myTextField.addDocumentListener(listener);
      }

      @Override
      public void addKeyListener(KeyListener listener) {
        myTextField.addKeyboardListener(listener);
      }

      @Override
      public void addIgnoreCaseListener(ItemListener listener) {

      }

      @Override
      public void onResultUpdated(SubstringFinder.FindResult result) {
      }

      @Override
      public void nextFindResultItem(SubstringFinder.FindResult.FindItem item) {
      }

      @Override
      public void prevFindResultItem(SubstringFinder.FindResult.FindItem item) {
      }
    };
  }

  public void addMessageFilter(@NotNull Filter filter) {
    myCompositeFilterWrapper.addFilter(filter);
  }

  public void start(TtyConnector connector) {
    setTtyConnector(connector);
    start();
  }

  public @NotNull JBTerminalSystemSettingsProviderBase getSettingsProvider() {
    return (JBTerminalSystemSettingsProviderBase)mySettingsProvider;
  }

  public void moveDisposable(@NotNull Disposable newParent) {
    Disposer.register(newParent, this);
  }

  public void notifyStarted() {
    if (myListener != null) {
      myListener.onTerminalStarted();
    }
  }

  @Nullable
  @Override
  public Object getData(@NotNull String dataId) {
    if (SELECTED_TEXT_DATA_KEY.is(dataId)) {
      return getSelectedText();
    }
    if (TERMINAL_DATA_KEY.is(dataId)) {
      return this;
    }
    return null;
  }

  @Nullable
  private String getSelectedText() {
    TerminalPanel terminalPanel = getTerminalPanel();
    TerminalSelection selection = terminalPanel.getSelection();
    if (selection != null) {
      Pair<Point, Point> points = selection.pointsForRun(terminalPanel.getColumnCount());
      if (points.first != null && points.second != null) {
        TerminalTextBuffer buffer = terminalPanel.getTerminalTextBuffer();
        buffer.lock();
        try {
          return SelectionUtil.getSelectionText(points.first, points.second, buffer);
        }
        finally {
          buffer.unlock();
        }
      }
    }
    return null;
  }

  public void writePlainMessage(@NotNull @Nls String message) {
    String str = StringUtil.convertLineSeparators(message, LineSeparator.LF.getSeparatorString());
    List<String> lines = StringUtil.split(str, LineSeparator.LF.getSeparatorString(), true, false);
    boolean first = true;
    for (String line : lines) {
      if (!first) {
        myTerminal.carriageReturn();
        myTerminal.newLine();
      }
      myTerminal.writeCharacters(line);
      first = false;
    }
  }
}
