// Copyright 2000-2021 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package org.intellij.plugins.intelliLang.inject;

import com.intellij.lang.Language;
import com.intellij.lang.injection.MultiHostRegistrar;
import com.intellij.lang.injection.general.Injection;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.Trinity;
import com.intellij.psi.*;
import com.intellij.psi.impl.source.tree.injected.FallbackInjectionPerformer;
import org.intellij.plugins.intelliLang.inject.config.BaseInjection;
import org.jetbrains.annotations.NotNull;

import java.util.Collections;
import java.util.List;

final class DefaultLanguageInjectionPerformer implements FallbackInjectionPerformer {
  @Override
  public boolean isPrimary() {
    return false;
  }

  @Override
  public boolean performInjection(@NotNull MultiHostRegistrar registrar,
                                  @NotNull Injection injection,
                                  @NotNull PsiElement context) {
    if (!(context instanceof PsiLanguageInjectionHost) || !((PsiLanguageInjectionHost)context).isValidHost()) {
      return false;
    }

    PsiLanguageInjectionHost host = (PsiLanguageInjectionHost)context;

    PsiFile containingFile = context.getContainingFile();
    InjectedLanguage injectedLanguage = InjectedLanguage.create(injection.getInjectedLanguageId(),
                                                                injection.getPrefix(),
                                                                injection.getSuffix(), false);

    String injectionSupportId = injection.getSupportId();
    LanguageInjectionSupport support = injectionSupportId != null ? InjectorUtils.findInjectionSupport(injectionSupportId) : null;

    Language language = injectedLanguage.getLanguage();
    final ElementManipulator<PsiLanguageInjectionHost> manipulator = ElementManipulators.getManipulator(host);
    if (language == null || manipulator == null) {
      if (injection instanceof BaseInjection) {
        return InjectorUtils.registerInjectionSimple(host, (BaseInjection)injection, support, registrar);
      }
      return false;
    }

    List<Trinity<PsiLanguageInjectionHost, InjectedLanguage, TextRange>> trinities =
      Collections.singletonList(Trinity.create(host, injectedLanguage, manipulator.getRangeInElement(host)));
    InjectorUtils.registerInjection(language, trinities, containingFile, registrar);
    if (support != null) {
      InjectorUtils.registerSupport(support, false, context, language);
    }
    return true;
  }
}
