/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jetbrains.idea.maven.navigator;

import com.intellij.openapi.project.Project;
import com.intellij.openapi.ui.DialogWrapper;
import com.intellij.openapi.util.NlsContexts;
import com.intellij.ui.ScrollPaneFactory;
import com.intellij.ui.treeStructure.SimpleNode;
import com.intellij.ui.treeStructure.SimpleNodeVisitor;
import com.intellij.ui.treeStructure.SimpleTree;
import com.intellij.util.ui.JBUI;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.idea.maven.project.MavenProjectsManager;
import org.jetbrains.idea.maven.tasks.MavenShortcutsManager;
import org.jetbrains.idea.maven.tasks.MavenTasksManager;

import javax.swing.*;
import javax.swing.tree.TreeSelectionModel;

public class SelectFromMavenProjectsDialog extends DialogWrapper {
  private final SimpleTree myTree;
  private final NodeSelector mySelector;

  public SelectFromMavenProjectsDialog(Project project,
                                       @NlsContexts.DialogTitle String title,
                                       final Class<? extends MavenProjectsStructure.MavenSimpleNode> nodeClass) {
    this(project, title, nodeClass, null);
  }

  public SelectFromMavenProjectsDialog(Project project,
                                       @NlsContexts.DialogTitle String title,
                                       final Class<? extends MavenProjectsStructure.MavenSimpleNode> nodeClass,
                                       @Nullable NodeSelector selector) {
    super(project, false);
    mySelector = selector;
    setTitle(title);

    myTree = new SimpleTree();
    myTree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);

    MavenProjectsStructure treeStructure = new MavenProjectsStructure(project,
                                                                      MavenProjectsManager.getInstance(project),
                                                                      MavenTasksManager.getInstance(project),
                                                                      MavenShortcutsManager.getInstance(project),
                                                                      MavenProjectsNavigator.getInstance(project),
                                                                      myTree) {
      @Override
      protected Class<? extends MavenSimpleNode>[] getVisibleNodesClasses() {
        //noinspection unchecked
        return new Class[]{nodeClass};
      }

      @Override
      protected boolean showDescriptions() {
        return false;
      }

      @Override
      protected boolean showOnlyBasicPhases() {
        return false;
      }
    };
    treeStructure.update();

    if (mySelector != null) {
      final SimpleNode[] selection = new SimpleNode[]{null};
      treeStructure.accept(new SimpleNodeVisitor() {
        @Override
        public boolean accept(SimpleNode each) {
          if (!mySelector.shouldSelect(each)) return false;
          selection[0] = each;
          return true;
        }
      });
      if (selection[0] != null) {
        treeStructure.select(selection[0]);
      }
    }

    init();
  }

  protected SimpleNode getSelectedNode() {
    return myTree.getNodeFor(myTree.getSelectionPath());
  }

  @Override
  @Nullable
  protected JComponent createCenterPanel() {
    final JScrollPane pane = ScrollPaneFactory.createScrollPane(myTree);
    pane.setPreferredSize(JBUI.size(320, 400));
    return pane;
  }

  protected interface NodeSelector {
    boolean shouldSelect(SimpleNode node);
  }
}
