// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package org.jetbrains.jps.model.java;

import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.jps.model.ex.JpsElementBase;

import java.util.Objects;

public class JavaSourceRootProperties extends JpsElementBase<JavaSourceRootProperties> {
  private String myPackagePrefix = "";
  private boolean myForGeneratedSources;

  @ApiStatus.Internal
  public JavaSourceRootProperties(@NotNull String packagePrefix, boolean forGeneratedSources) {
    myPackagePrefix = packagePrefix;
    myForGeneratedSources = forGeneratedSources;
  }

  @NotNull
  public String getPackagePrefix() {
    return myPackagePrefix;
  }

  @NotNull
  @Override
  public JavaSourceRootProperties createCopy() {
    return new JavaSourceRootProperties(myPackagePrefix, myForGeneratedSources);
  }

  /**
   * Returns {@code true} if the source root contain only sources which were automatically generated by some program. Classes from such
   * sources roots can be used in the IDE, they will be compiled with the other sources, etc, however IDE will warn users if they try
   * editing them manually.
   */
  public boolean isForGeneratedSources() {
    return myForGeneratedSources;
  }

  public void setPackagePrefix(@NotNull String packagePrefix) {
    if (!Objects.equals(myPackagePrefix, packagePrefix)) {
      myPackagePrefix = packagePrefix;
      fireElementChanged();
    }
  }

  public void setForGeneratedSources(boolean forGeneratedSources) {
    if (myForGeneratedSources != forGeneratedSources) {
      myForGeneratedSources = forGeneratedSources;
      fireElementChanged();
    }
  }

  @Override
  public void applyChanges(@NotNull JavaSourceRootProperties modified) {
    setPackagePrefix(modified.myPackagePrefix);
    setForGeneratedSources(modified.myForGeneratedSources);
  }
}
