/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jetbrains.plugins.groovy.refactoring.rename;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiManager;
import com.intellij.psi.PsiMember;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.plugins.groovy.transformations.impl.DefaultRenameHelper;

public interface GrRenameHelper {

  ExtensionPointName<GrRenameHelper> EP_NAME = ExtensionPointName.create("org.intellij.groovy.renameHelper");

  /**
   * GrReferenceExpression may resolve to member generated by some AST Transformation from original member.
   * <p>
   * Example:
   * <pre>
   *   class A {
   *     def foo
   *   }
   *   new A().getFoo()
   * </pre>
   * {@code getFoo} references accessor method generated from original <i>foo</i> field.
   * Suppose we are renaming the field to <i>bar</i>.
   * Then {@code member} will be the referenced accessor method and {@code newOriginalName} is the new name of the field, i.e. <i>bar</i>.
   * Given this information {@link DefaultRenameHelper} returns the name of the new accessor
   * that will be generated after rename, i.e. <i>getBar</i>.
   *
   * @param member          currently resolved member.
   * @param newOriginalName new name of original member.
   * @return name of new member which will be generated after original member changes its name to {@code newOriginalName}.
   * @see DefaultRenameHelper
   */
  @Nullable
  default String getNewMemberName(@NotNull PsiMember member, @NotNull String newOriginalName) {
    return null;
  }

  /**
   * @param manager
   * @param before element to which the reference resolved before rename
   * @param after  element to which the reference resolves after rename
   */
  default boolean isQualificationNeeded(@NotNull PsiManager manager, @NotNull PsiElement before, @NotNull PsiElement after) {
    return false;
  }
}
