/*
 * Decompiled with CFR 0.152.
 */
package org.junit.platform.commons.util;

import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.junit.platform.commons.meta.API;
import org.junit.platform.commons.util.BlacklistedExceptions;
import org.junit.platform.commons.util.ClassFileVisitor;
import org.junit.platform.commons.util.CloseablePath;
import org.junit.platform.commons.util.PreconditionViolationException;
import org.junit.platform.commons.util.Preconditions;
import org.junit.platform.commons.util.StringUtils;

@API(value=API.Usage.Internal)
class ClasspathScanner {
    private static final Logger LOG = Logger.getLogger(ClasspathScanner.class.getName());
    private static final String DEFAULT_PACKAGE_NAME = "";
    private static final char CLASSPATH_RESOURCE_PATH_SEPARATOR = '/';
    private static final char PACKAGE_SEPARATOR_CHAR = '.';
    private static final String PACKAGE_SEPARATOR_STRING = String.valueOf('.');
    private static final String MALFORMED_CLASS_NAME_ERROR_MESSAGE = "Malformed class name";
    private final Supplier<ClassLoader> classLoaderSupplier;
    private final BiFunction<String, ClassLoader, Optional<Class<?>>> loadClass;

    ClasspathScanner(Supplier<ClassLoader> classLoaderSupplier, BiFunction<String, ClassLoader, Optional<Class<?>>> loadClass) {
        this.classLoaderSupplier = classLoaderSupplier;
        this.loadClass = loadClass;
    }

    boolean isPackage(String packageName) {
        ClasspathScanner.assertPackageNameIsPlausible(packageName);
        try {
            return packageName.isEmpty() || this.getClassLoader().getResources(ClasspathScanner.packagePath(packageName.trim())).hasMoreElements();
        }
        catch (Exception ex) {
            return false;
        }
    }

    List<Class<?>> scanForClassesInPackage(String basePackageName, Predicate<Class<?>> classFilter, Predicate<String> classNameFilter) {
        ClasspathScanner.assertPackageNameIsPlausible(basePackageName);
        Preconditions.notNull(classFilter, "classFilter must not be null");
        Preconditions.notNull(classNameFilter, "classNameFilter must not be null");
        basePackageName = basePackageName.trim();
        return this.findClassesForUris(this.getRootUrisForPackage(basePackageName), basePackageName, classFilter, classNameFilter);
    }

    List<Class<?>> scanForClassesInClasspathRoot(URI root, Predicate<Class<?>> classFilter, Predicate<String> classNameFilter) {
        Preconditions.notNull(root, "root must not be null");
        Preconditions.notNull(classFilter, "classFilter must not be null");
        Preconditions.notNull(classNameFilter, "classNameFilter must not be null");
        return this.findClassesForUri(root, DEFAULT_PACKAGE_NAME, classFilter, classNameFilter);
    }

    private List<Class<?>> findClassesForUris(List<URI> baseUris, String basePackageName, Predicate<Class<?>> classFilter, Predicate<String> classNameFilter) {
        return baseUris.stream().map(baseUri -> this.findClassesForUri((URI)baseUri, basePackageName, classFilter, classNameFilter)).flatMap(Collection::stream).distinct().collect(Collectors.toList());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private List<Class<?>> findClassesForUri(URI baseUri, String basePackageName, Predicate<Class<?>> classFilter, Predicate<String> classNameFilter) {
        try (CloseablePath closeablePath = CloseablePath.create(baseUri);){
            Path baseDir = closeablePath.getPath();
            List<Class<?>> list = this.findClassesForPath(baseDir, basePackageName, classFilter, classNameFilter);
            return list;
        }
        catch (PreconditionViolationException ex) {
            throw ex;
        }
        catch (Exception ex) {
            ClasspathScanner.logWarning(ex, () -> "Error scanning files for URI " + baseUri);
            return Collections.emptyList();
        }
    }

    private List<Class<?>> findClassesForPath(Path baseDir, String basePackageName, Predicate<Class<?>> classFilter, Predicate<String> classNameFilter) {
        Preconditions.condition(Files.exists(baseDir, new LinkOption[0]), () -> "baseDir must exist: " + baseDir);
        ArrayList classes = new ArrayList();
        try {
            Files.walkFileTree(baseDir, new ClassFileVisitor(classFile -> this.processClassFileSafely(baseDir, basePackageName, classFilter, classNameFilter, (Path)classFile, classes::add)));
        }
        catch (IOException ex) {
            ClasspathScanner.logWarning(ex, () -> "I/O error scanning files in " + baseDir);
        }
        return classes;
    }

    private void processClassFileSafely(Path baseDir, String basePackageName, Predicate<Class<?>> classFilter, Predicate<String> classNameFilter, Path classFile, Consumer<Class<?>> classConsumer) {
        Optional<Class<?>> clazz = Optional.empty();
        try {
            String fullyQualifiedClassName = this.determineFullyQualifiedClassName(baseDir, basePackageName, classFile);
            if (classNameFilter.test(fullyQualifiedClassName)) {
                clazz = this.loadClass.apply(fullyQualifiedClassName, this.getClassLoader());
                clazz.filter(classFilter).ifPresent(classConsumer);
            }
        }
        catch (InternalError internalError) {
            this.handleInternalError(classFile, clazz, internalError);
        }
        catch (Throwable throwable) {
            this.handleThrowable(classFile, throwable);
        }
    }

    private String determineFullyQualifiedClassName(Path baseDir, String basePackageName, Path classFile) {
        return Stream.of(basePackageName, this.determineSubpackageName(baseDir, classFile), this.determineSimpleClassName(classFile)).filter(value -> !value.isEmpty()).collect(Collectors.joining(PACKAGE_SEPARATOR_STRING));
    }

    private String determineSimpleClassName(Path classFile) {
        String fileName = classFile.getFileName().toString();
        return fileName.substring(0, fileName.length() - ".class".length());
    }

    private String determineSubpackageName(Path baseDir, Path classFile) {
        Path relativePath = baseDir.relativize(classFile.getParent());
        String pathSeparator = baseDir.getFileSystem().getSeparator();
        String subpackageName = relativePath.toString().replace(pathSeparator, PACKAGE_SEPARATOR_STRING);
        if (subpackageName.endsWith(pathSeparator)) {
            subpackageName = subpackageName.substring(0, subpackageName.length() - pathSeparator.length());
        }
        return subpackageName;
    }

    private void handleInternalError(Path classFile, Optional<Class<?>> clazz, InternalError ex) {
        if (MALFORMED_CLASS_NAME_ERROR_MESSAGE.equals(ex.getMessage())) {
            this.logMalformedClassName(classFile, clazz, ex);
        } else {
            this.logGenericFileProcessingException(classFile, ex);
        }
    }

    private void handleThrowable(Path classFile, Throwable throwable) {
        BlacklistedExceptions.rethrowIfBlacklisted(throwable);
        this.logGenericFileProcessingException(classFile, throwable);
    }

    private void logMalformedClassName(Path classFile, Optional<Class<?>> clazz, InternalError ex) {
        try {
            if (clazz.isPresent()) {
                ClasspathScanner.logWarning(ex, () -> String.format("The java.lang.Class loaded from path [%s] has a malformed class name [%s].", classFile.toAbsolutePath(), ((Class)clazz.get()).getName()));
            } else {
                ClasspathScanner.logWarning(ex, () -> String.format("The java.lang.Class loaded from path [%s] has a malformed class name.", classFile.toAbsolutePath()));
            }
        }
        catch (Throwable t) {
            ex.addSuppressed(t);
            this.logGenericFileProcessingException(classFile, ex);
        }
    }

    private void logGenericFileProcessingException(Path classFile, Throwable throwable) {
        ClasspathScanner.logWarning(throwable, () -> String.format("Failed to load java.lang.Class for path [%s] during classpath scanning.", classFile.toAbsolutePath()));
    }

    private ClassLoader getClassLoader() {
        return this.classLoaderSupplier.get();
    }

    private static void assertPackageNameIsPlausible(String packageName) {
        Preconditions.notNull(packageName, "package name must not be null");
        Preconditions.condition(DEFAULT_PACKAGE_NAME.equals(packageName) || StringUtils.isNotBlank(packageName), "package name must not contain only whitespace");
    }

    private static String packagePath(String packageName) {
        return packageName.replace('.', '/');
    }

    private List<URI> getRootUrisForPackage(String basePackageName) {
        try {
            Enumeration<URL> resources = this.getClassLoader().getResources(ClasspathScanner.packagePath(basePackageName));
            ArrayList<URI> uris = new ArrayList<URI>();
            while (resources.hasMoreElements()) {
                URL resource = resources.nextElement();
                uris.add(resource.toURI());
            }
            return uris;
        }
        catch (Exception ex) {
            ClasspathScanner.logWarning(ex, () -> "Error reading URIs from class loader for base package " + basePackageName);
            return Collections.emptyList();
        }
    }

    private static void logWarning(Throwable throwable, Supplier<String> msgSupplier) {
        LOG.log(Level.WARNING, throwable, msgSupplier);
    }
}

