---- QuerySessionCurrentInfo ----
select current_database(),
       current_schema(),
       current_timestamp as current_timestamp
;


---- IsSuperUser ----
select usesuper
from pg_user
where usename = current_user
;


---- ListDatabases ----
select oid::bigint as id, datname as name
from pg_catalog.pg_database
where not datistemplate
order by case when datname = current_database() then -1::bigint else oid::bigint end
;


---- ListSchemas ----
select oid::bigint as id, nspname as name
from pg_catalog.pg_namespace
order by case when nspname = current_schema() then -1::bigint else oid::bigint end
;


---- RetrieveRoles ----
select oid::bigint as role_id, rolname as role_name,
  rolsuper is_super, rolinherit is_inherit,
  rolcreaterole can_createrole, rolcreatedb can_createdb,
  rolcanlogin can_login, #V91: rolreplication #: false #. is_replication,
  rolconnlimit conn_limit, rolvaliduntil valid_until,
  #V95: rolbypassrls #: false #. bypass_rls, rolconfig config
from pg_catalog.pg_roles
;


---- RetrieveSchemas ----
select S.nspname as schema_name,
       S.oid as schema_id,
       S.xmin as schema_state_number,
       pg_catalog.pg_get_userbyid(S.nspowner) as owner_name,
       D.description
from pg_catalog.pg_namespace S
     left outer join pg_catalog.pg_description D on S.oid = D.objoid
order by (S.nspname = current_schema) desc,
         strpos(S.nspname,'pg_temp'), strpos(S.nspname,'pg_toast'),
         case when D.description is not null then 1 else 2 end
;


---- Now ----
select current_timestamp
;


---- CountSchemaStateNumber ----
select max(last_tx) as state_number
from (
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_type
    where typnamespace = :schema_id::oid
  union all
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_class
    where relnamespace = :schema_id::oid
  union all
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_proc
    where pronamespace = :schema_id::oid
  union all
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_operator
    where oprnamespace = :schema_id::oid
  union all
  select max(A.xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_attribute A join pg_catalog.pg_class K on K.oid = A.attrelid
    where K.relnamespace = :schema_id::oid
  union all
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_constraint
    where connamespace = :schema_id::oid
  union all
  #V91:select max(CO.xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_collation CO
    where CO.collnamespace = :schema_id::oid
  union all
  #.
  select max(IX.xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_index IX,
         pg_catalog.pg_class IC
    where IC.oid = IX.indrelid
      and IC.relnamespace = :schema_id::oid
  union all
  select max(RU.xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_rewrite RU,
         pg_catalog.pg_class RC
    where RC.oid = RU.ev_class
      and RC.relnamespace = :schema_id::oid
  union all
  select max(TG.xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_trigger TG,
         pg_catalog.pg_class TC
    where TC.oid = TG.tgrelid
      and TC.relnamespace = :schema_id::oid
  union all
  select max(DE.xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_description DE,
         pg_catalog.pg_class DC
    where DC.oid = DE.objoid
      and DC.relnamespace = :schema_id::oid
      and DC.relkind in ('r', 'm', 'v', 'f')
) X
;


---- CountDatabaseStateNumber ----
select max(last_tx) as state_number
from (
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_foreign_data_wrapper
  union all
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_foreign_server
  #SUPER:union all
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_user_mapping
  #.
) X
;


---- ListExistentClasses ----
select distinct oid
from pg_catalog.pg_type
where typnamespace = :schema_id::oid
union
select distinct oid
from pg_catalog.pg_class
where relnamespace = :schema_id::oid
union
select distinct oid
from pg_catalog.pg_proc
where pronamespace = :schema_id::oid
union
select distinct oid
from pg_catalog.pg_operator
where oprnamespace = :schema_id::oid
union
select distinct oid
from pg_catalog.pg_constraint
where connamespace = :schema_id::oid
union
#V91:select distinct oid
from pg_catalog.pg_collation
where collnamespace = :schema_id::oid
union
#.
select distinct IX.indexrelid
from pg_catalog.pg_index IX,
     pg_catalog.pg_class IC
where IC.oid = IX.indrelid
  and IC.relnamespace = :schema_id::oid
union
select distinct RU.oid
from pg_catalog.pg_rewrite RU,
     pg_catalog.pg_class RC
where RC.oid = RU.ev_class
  and RC.relnamespace = :schema_id::oid
  and not RU.rulename = '_RETURN'
union
select distinct TG.oid
from pg_catalog.pg_trigger TG,
     pg_catalog.pg_class TC
where TC.oid = TG.tgrelid
  and TC.relnamespace = :schema_id::oid
;


---- ListDatabaseExistentClasses ----
select distinct oid
from pg_catalog.pg_foreign_data_wrapper
union
select distinct oid
from pg_catalog.pg_foreign_server
#SUPER:union
select distinct oid
from pg_catalog.pg_user_mapping
#.
;


---- RetrieveSequences10 ----
select sq.xmin as sequence_state_number,
       sq.seqrelid as sequence_id,
       cls.relname as sequence_name,
       pg_catalog.format_type(sq.seqtypid, null) as data_type,
       sq.seqstart as next_value,
       sq.seqincrement as inc_value,
       sq.seqmin as min_value,
       sq.seqmax as max_value,
       sq.seqcache as cache_size,
       sq.seqcycle as cycle_option
from pg_catalog.pg_sequence sq
    join pg_class cls on sq.seqrelid = cls.oid
    where cls.relnamespace = :schema_id::oid
          #INC> and sq.xmin::varchar::bigint >:tx
;


---- RetrieveSequences ----
with #V91>U as (select usesysid from pg_catalog.pg_user where usename = current_user),
          T as (
                select oid as sequence_id,
                       relname as sequence_name,
                       #V91:
                       case when relowner = (select usesysid from U) then trim(both '()' from pg_catalog.pg_sequence_parameters(oid)::varchar)
                       else null::varchar
                       end as sequence_params,
                       #.
                       xmin as sequence_state_number
                       from pg_catalog.pg_class
                         where relnamespace = :schema_id::oid
                         and relkind = 'S'
                         #INC> and xmin::varchar::bigint > :tx
                )
select sequence_id,
       sequence_name,
       #V91:
       pg_catalog.split_part(sequence_params, ',', 1)::bigint as next_value,
       pg_catalog.split_part(sequence_params, ',', 2)::bigint as min_value,
       pg_catalog.split_part(sequence_params, ',', 3)::bigint as max_value,
       pg_catalog.split_part(sequence_params, ',', 4)::bigint as inc_value,
       lower(pg_catalog.split_part(sequence_params, ',', 5)) LIKE '%t%' as cycle_option,
       #.
       sequence_state_number
from T
;


---- RetrieveDataTypes ----
select T.oid as type_id,
       T.xmin as type_state_number,
       T.typname as type_name,
       T.typtype as type_sub_kind,
       T.typcategory as type_category,
       T.typrelid as class_id,
       T.typbasetype as base_type_id,
       case when T.typtype in ('c','e') then null
            else pg_catalog.format_type(T.typbasetype, T.typtypmod) end as type_def,
       T.typdefault as default_expression,
       T.typnotnull as mandatory
from pg_catalog.pg_type T
         left outer join pg_catalog.pg_class C
             on T.typrelid = C.oid
where T.typnamespace = :schema_id::oid
  #INC> and T.xmin::varchar::bigint > :tx
  and (T.typtype in ('d','e') or C.relkind = 'c'::"char" or T.typtype = 'b' and (T.typelem = 0 OR T.typcategory <> 'A'))
order by 1
;


---- RetrieveDataTypeChecks ----
select T.oid::bigint as type_id,
       C.oid::bigint as constraint_id,
       C.xmin as constraint_state_number,
       C.conname as constraint_name,
       C.consrc as predicate
from pg_catalog.pg_type T,
     pg_catalog.pg_constraint C
where T.typnamespace = :schema_id::oid
  and T.typtype = 'd'::"char"
  and T.oid = C.contypid
  #INC> and (T.xmin::varchar::bigint > :tx or C.xmin::varchar::bigint > :tx)
order by constraint_name
;


---- RetrieveTables ----
select T.relkind as table_kind,
       T.relname as table_name,
       T.oid as table_id,
       T.xmin as table_state_number,
       T.relhasoids as table_with_oids,
       T.relnamespace as tablespace_id,
       T.reloptions as options,
       (select array_agg(inhparent)::varchar from (select inhparent::bigint from pg_catalog.pg_inherits where T.oid = inhrelid order by inhseqno) as bts) as ancestors,
       (select array_agg(inhrelid)::varchar from (select inhrelid::bigint from pg_catalog.pg_inherits where T.oid = inhparent order by inhrelid) as its) as successors
from pg_catalog.pg_class T
where relnamespace = :schema_id::oid
  and relkind in ('r', 'm', 'v', 'f')
  #INC> and T.xmin::varchar::bigint > :tx
order by table_kind, table_id
;


---- RetrieveForeignTables ----
select ft.ftrelid as table_id,
       srv.srvname as table_server,
       ft.ftoptions as table_options
from pg_catalog.pg_foreign_table ft
     left outer join pg_catalog.pg_foreign_server srv on ft.ftserver = srv.oid
     join pg_catalog.pg_class cls on ft.ftrelid = cls.oid
where cls.relnamespace = :schema_id::oid
  #INC> and ft.xmin::varchar::bigint > :tx
order by table_id
;


---- RetrieveStructAttributes ----
with S as ( select T.oid as type_id, C.oid as class_id
            from pg_catalog.pg_type T,
                 pg_catalog.pg_class C
            where T.typnamespace = :schema_id::oid
              and T.typnamespace = C.relnamespace
              #INC> and T.xmin::varchar::bigint > :tx
              and T.typrelid = C.oid
              and T.typtype = 'c'::"char"
              and relkind = 'c'::"char" ),
     A as ( select attrelid as class_id,
                   attnum as att_position,
                   attname as att_name,
                   pg_catalog.format_type(atttypid,atttypmod) as att_type
            from pg_catalog.pg_attribute )
select type_id, att_position, att_name, att_type
from S natural join A
order by type_id::bigint, att_position
;


---- RetrieveEnumLabels ----
select T.oid as enum_id,
       array_agg(E.enumlabel::varchar #V91: order by E.enumsortorder #.) as enum_labels
from pg_catalog.pg_type T,
     pg_catalog.pg_enum E
where T.typnamespace = :schema_id::oid
  #INC> and T.xmin::varchar::bigint > :tx
  and T.oid = E.enumtypid
group by T.oid
;


---- RetrieveColumns ----
with T as ( select distinct T.oid as table_id, T.relname as table_name
            from pg_catalog.pg_class T,
                 pg_catalog.pg_attribute A
            where T.relnamespace = :schema_id::oid
              and T.relkind in ('r', 'm', 'v', 'f')
              #INC> and A.xmin::varchar::bigint > :tx
              and A.attrelid = T.oid )
select T.table_id,
       C.attnum as column_position,
       C.attname as column_name,
       C.xmin as column_state_number,
       C.atttypmod as type_mod,
       C.attndims as dimensions_number,
       pg_catalog.format_type(C.atttypid, C.atttypmod) as type_spec,
       C.atttypid as type_id,
       C.attnotnull as mandatory,
       pg_get_serial_sequence(quote_ident(:schema_name)||'.'||quote_ident(T.table_name), C.attname)::regclass as sequence,
       D.adsrc as column_default_expression,
       not C.attislocal as column_is_inherited,
       #V92> C.attfdwoptions as options,
       C.attisdropped as column_is_dropped
from T
  inner join pg_catalog.pg_attribute C on T.table_id = C.attrelid
  left outer join pg_catalog.pg_attrdef D on (C.attrelid, C.attnum) = (D.adrelid, D.adnum)
where attnum > 0
  and not attisdropped
order by table_id, attnum
;


---- RetrieveIndices ----
with tab as ( select relnamespace as schema_id,
                     oid as table_id,
                     relkind as table_kind
              from pg_catalog.pg_class
              where relnamespace = :schema_id::oid
                and relkind in ('r','m','v')),
    ind_head as ( select indrelid as table_id,
                         indexrelid as index_id,
                         indisunique as is_unique,
                         indisprimary as is_primary,
                         indkey as column_positions,
                         indclass as column_operators,
                         indoption as column_options,
                         pg_get_expr(indpred, indrelid) as condition,
                         case when indexprs is not null
                             then array_to_string(array(
                                      select case when indkey[k] = 0
                                             then chr(27) || pg_get_indexdef(indexrelid, k + 1, true)
                                             else pg_get_indexdef(indexrelid, k + 1, true)
                                             end
                                      from generate_subscripts(indkey, 1) as k
                                      order by k),
                                  chr(28))
                             else null end as expressions
                  from pg_catalog.pg_index ),
    ind_stor as ( select cls.relnamespace as schema_id,
                         cls.oid as index_id,
                         cls.relname as index_name,
                         #V96:pg_catalog.pg_indexam_has_property(cls.relam, 'can_order') #: am.amcanorder #. as can_order
                  from pg_catalog.pg_class cls
                    #V96:#:join pg_catalog.pg_am am on (cls.relam = am.oid)#.
                  where cls.relkind = 'i'
                    #INC: and cls.xmin::varchar::bigint > :tx #.)
select table_id, table_kind,
       index_name, index_id,
       is_unique, is_primary,
       column_positions, column_operators, column_options,
       condition, expressions, can_order
from tab natural join ind_head natural join ind_stor
;


---- RetrieveConstraints ----
with T as ( select oid as table_id,
                   relkind as table_kind,
                   xmin::varchar::bigint as table_state_id
            from pg_catalog.pg_class
            where relkind in ('r','v','f')
              and relnamespace = :schema_id::oid ),
     C as ( select conrelid as table_id,
                   oid::bigint as con_id,
                   xmin::varchar::bigint as con_state_id,
                   conname as con_name,
                   contype as con_kind,
                   conkey as con_columns,
                   conindid as index_id,
                   confrelid as ref_table_id,
                   confkey as ref_columns,
                   condeferrable as is_deferrable,
                   condeferred as is_init_deferred,
                   confupdtype as on_update,
                   confdeltype as on_delete,
                   consrc as con_expression
            from pg_catalog.pg_constraint
            where contype in ('p','u','f','c')
              and connamespace = :schema_id::oid )
select table_id,
       table_kind,
       con_id,
       con_name,
       con_kind,
       con_columns,
       index_id,
       ref_table_id,
       is_deferrable,
       is_init_deferred,
       on_update,
       on_delete,
       con_expression,
       ref_columns
from T natural join C
#INC> where table_state_id > :tx or con_state_id > :tx
order by strpos('pufc', con_kind)
;


---- RetrieveUsedArgumentTypes ----
with schema_procs as (select prorettype, proargtypes, proallargtypes
                      from pg_catalog.pg_proc
                      where pronamespace = :schema_id::oid
                        #INC: and xmin::varchar::bigint > :tx #. ),
     schema_opers as (select oprleft, oprright, oprresult
                      from pg_catalog.pg_operator
                      where oprnamespace = :schema_id::oid
                        #INC: and xmin::varchar::bigint > :tx #. ),
     schema_arg_types as ( select prorettype as type_id
                           from schema_procs
                           union
                           select distinct unnest(proargtypes) as type_id
                           from schema_procs
                           union
                           select distinct unnest(proallargtypes) as type_id
                           from schema_procs
                           union
                           select oprleft as type_id
                           from schema_opers
                           where oprleft is not null
                           union
                           select oprright as type_id
                           from schema_opers
                           where oprright is not null
                           union
                           select oprresult as type_id
                           from schema_opers
                           where oprresult is not null
                           )
select type_id, pg_catalog.format_type(type_id, null) as type_spec
from schema_arg_types
;


---- RetrieveRoutines ----
with languages as (select oid as lang_oid, lanname as lang
                   from pg_catalog.pg_language),
     routines as (select proname as r_name,
                         prolang as lang_oid,
                         oid as r_id,
                         xmin as r_state_number,
                         proargnames as arg_names,
                         proargmodes as arg_modes,
                         proargtypes::int[] as in_arg_types,
                         proallargtypes::int[] as all_arg_types,
                         pg_catalog.pg_get_expr(proargdefaults, 0) as arg_defaults,
                         provariadic as arg_variadic_id,
                         prorettype as ret_type_id,
                         proretset as ret_set,
                         proisagg as is_aggregate,
                         proiswindow as is_window,
                         provolatile as volatile_kind,
                         proisstrict as is_strict,
                         prosecdef as is_security_definer,
                         proconfig as configuration_parameters,
                         procost as cost,
                         prorows as rows #V92:,
                         proleakproof as is_leakproof #. #V96:,
                         proparallel as concurrency_kind #.
                  from pg_catalog.pg_proc
                  where pronamespace = :schema_id::oid
                    #INC: and xmin::varchar::bigint > :tx #.)
select *
from routines natural join languages
;


---- RetrieveOperators ----
select O.oid as op_id,
       oprname as op_name,
       oprkind as op_kind,
       oprleft as arg_left_type_id,
       oprright as arg_right_type_id,
       oprresult as arg_result_type_id,
       oprcode as func_main
from pg_catalog.pg_operator O
where oprnamespace = :schema_id::oid
  #INC> and xmin::varchar::bigint > :tx
;


---- RetrieveRules ----
select R.ev_class                       as table_id,
       R.oid                            as rule_id,
       R.xmin                           as rule_state_number,
       R.rulename                       as rule_name,
       translate(ev_type,'1234','SIUD') as rule_event_code,
       R.ev_enabled                     as rule_fire_mode,
       R.is_instead                     as rule_is_instead
from pg_catalog.pg_rewrite R
where R.ev_class in (select oid
                     from pg_catalog.pg_class
                     where relnamespace = :schema_id::oid)
  #INC> and R.xmin::varchar::bigint > :tx
  and R.rulename != '_RETURN'::name
order by R.ev_class::bigint, ev_type
;


---- RetrieveTriggers ----
select T.tgrelid as table_id,
       T.oid as trigger_id,
       T.xmin as trigger_state_number,
       T.tgname as trigger_name,
       T.tgfoid as function_id,
       T.tgtype & 31 as bits,
       T.tgdeferrable as is_deferrable,
       T.tginitdeferred as is_init_deferred,
       T.tgenabled as trigger_fire_mode
from pg_catalog.pg_trigger T
where T.tgrelid in (select oid
                    from pg_catalog.pg_class
                    where relnamespace = :schema_id::oid)
  #INC> and T.xmin::varchar::bigint > :tx
  and not T.tgisinternal
;


---- RetrieveDescriptions ----
select T.relkind as table_kind,
       T.oid as table_id,
       D.objsubid as column_position,
       D.description
from pg_catalog.pg_class T,
     pg_catalog.pg_description D
where T.relnamespace = :schema_id::oid
  and T.relkind in ('r', 'm', 'v')
  and T.oid = D.objoid
  #INC> and D.xmin::varchar::bigint > :tx
order by table_id, column_position
;


---- RetrieveViewSources ----
select T.relkind as view_kind,
       T.oid as view_id,
       pg_catalog.pg_get_viewdef(T.oid) as source_text
from pg_catalog.pg_class T,
     pg_catalog.pg_namespace N
where N.oid = :schema_id::oid
  and T.relnamespace = N.oid
  and T.relkind in ('m','v')
  #INC> and T.xmin::varchar::bigint > :src_tx
order by T.xmin::varchar::bigint
;


---- RetrieveRuleSources ----
with A as ( select oid as table_id, upper(relkind) as table_kind
            from pg_catalog.pg_class
            where relnamespace = :schema_id::oid
              and relkind in ('r', 'm', 'v', 'f') )
select table_kind,
       table_id,
       R.oid as rule_id,
       pg_catalog.pg_get_ruledef(R.oid) as source_text
from A join pg_catalog.pg_rewrite R
        on A.table_id = R.ev_class
where R.rulename != '_RETURN'::name
  #INC> and R.xmin::varchar::bigint > :src_tx
;


---- RetrieveFunctionSources ----
with system_languages as ( select oid as lang
                           from pg_catalog.pg_language
                           where lanname in ('c','internal') )
select oid as id,
       pg_catalog.pg_get_function_arguments(oid) as arguments_def,
       pg_catalog.pg_get_function_result(oid) as result_def,
       prosrc as source_text
from pg_catalog.pg_proc
where pronamespace = :schema_id::oid
  #INC> and xmin::varchar::bigint > :src_tx
  and not proisagg
  and prolang not in (select lang from system_languages)
  and prosrc is not null
;


---- RetrieveCollations ----
select oid as id,
       collname as name,
       collcollate as lc_collate,
       collctype as lc_ctype
from pg_catalog.pg_collation
where collnamespace = :schema_id::oid
  #INC> and xmin::varchar::bigint > :tx
;


---- RetrieveForeignDataWrappers ----
select fdw.oid as id,
       fdw.fdwname as name,
       #V91>pr.proname as handler,
       #V91>nspc.nspname as handler_schema,
       pr2.proname as validator,
       nspc2.nspname as validator_schema,
       fdw.fdwoptions as options
from pg_catalog.pg_foreign_data_wrapper fdw
     #V91>left outer join pg_catalog.pg_proc pr on fdw.fdwhandler = pr.oid
     #V91>left outer join pg_catalog.pg_namespace nspc on pr.pronamespace = nspc.oid
     left outer join pg_catalog.pg_proc pr2 on fdw.fdwvalidator = pr2.oid
     left outer join pg_catalog.pg_namespace nspc2 on pr2.pronamespace = nspc2.oid
  #INC> where fdw.xmin::varchar::bigint > :tx
;


---- RetrieveForeignServers ----
select srv.oid as id,
       srv.srvname as name,
       srv.srvtype as type,
       srv.srvversion as version,
       srv.srvoptions as options
from pg_catalog.pg_foreign_server srv
where srv.srvfdw = :fdw_id::oid
  #INC> and srv.xmin::varchar::bigint > :tx
;


---- RetrieveUserMappingsSuperUser ----
select um.oid as id,
       case when um.umuser = 0 then null else pg_catalog.pg_get_userbyid(um.umuser) end as user,
       um.umoptions as options
from pg_catalog.pg_user_mapping um
where um.umserver = :server_id::oid
  #INC> and um.xmin::varchar::bigint > :tx
;


---- RetrieveUserMappings ----
select um.umid as id,
       um.usename as user,
       um.umoptions as options
from pg_catalog.pg_user_mappings um
where um.srvid = :server_id::oid
;


---- ListReferencingSchemas ----
select distinct connamespace as schema_id
from pg_catalog.pg_constraint F,
     pg_catalog.pg_class O
where F.contype = 'f'
  and F.confrelid = O.oid
  and O.relnamespace in (SCHEMAIDS)
;
