---- QuerySessionCurrentInfo ----
select current_database(),
       current_schema(),
       current_timestamp::timestamp as current_timestamp
;


---- ListDatabases ----
select oid::bigint as id, datname as name
from pg_catalog.pg_database
where not datistemplate
order by case when datname = current_database() then -1::bigint else oid::bigint end
;


---- ListSchemas ----
select N.oid::bigint as id, N.nspname as name
from pg_catalog.pg_namespace N
  left outer join pg_catalog.pg_external_schema ES on N.oid = ES.esoid
  where ES.esoid is null
order by case when N.nspname = current_schema() then -1::bigint else N.oid::bigint end
;


---- RetrievePermissions ----
select has_table_privilege('pg_catalog.pg_user_info', 'select') user_ext

---- RetrieveUsers ----
select usesysid as user_id, usename as user_name,
  usesuper is_super, usecreatedb can_createdb,
  #[USER_EXT]: case when useconnlimit = 'UNLIMITED' then -1 ELSE useconnlimit::int end #: null #. conn_limit,
  valuntil valid_until, '{' || array_to_string(useconfig, :array_delimiter) || '}' config
from #[USER_EXT]: pg_catalog.pg_user_info #: pg_catalog.pg_user #.
;


---- RetrieveSchemas ----
select S.nspname as schema_name,
       S.oid as schema_id,
       S.xmin as schema_state_number,
       pg_catalog.pg_get_userbyid(S.nspowner) as owner_name,
       D.description
from pg_catalog.pg_namespace S
     left outer join pg_catalog.pg_description D on S.oid = D.objoid
order by (S.nspname = current_schema()) desc,
         strpos(S.nspname,'pg_temp'), strpos(S.nspname,'pg_toast'),
         case when D.description is not null then 1 else 2 end
;


---- RetrieveExternalSchemas ----
select S.nspname as schema_name,
       S.oid as schema_id,
       ES.esdbname as database_name,
       ES.eskind as schema_kind,
       ES.esoptions as schema_options
from pg_catalog.pg_external_schema ES
  left join pg_catalog.pg_namespace S on ES.esoid = S.oid
;


---- Now ----
select current_timestamp::timestamp
;


---- CountSchemaStateNumber ----
select max(last_tx) as state_number
from (
  select max(pg_catalog.int8in(pg_catalog.xidout(xmin))) as last_tx
    from pg_catalog.pg_type
    where typnamespace = oid(:schema_id)
  union all
  select max(pg_catalog.int8in(pg_catalog.xidout(xmin))) as last_tx
    from pg_catalog.pg_class
    where relnamespace = oid(:schema_id)
  union all
  select max(pg_catalog.int8in(pg_catalog.xidout(xmin))) as last_tx
    from pg_catalog.pg_proc
    where pronamespace = oid(:schema_id)
  union all
  select max(pg_catalog.int8in(pg_catalog.xidout(xmin))) as last_tx
    from pg_catalog.pg_operator
    where oprnamespace = oid(:schema_id)
  union all
  select max(pg_catalog.int8in(pg_catalog.xidout(A.xmin))) as last_tx
    from pg_catalog.pg_attribute A join pg_catalog.pg_class K on K.oid = A.attrelid
    where K.relnamespace = oid(:schema_id)
  union all
  select max(pg_catalog.int8in(pg_catalog.xidout(xmin))) as last_tx
    from pg_catalog.pg_constraint
    where connamespace = oid(:schema_id)
  union all
  select max(pg_catalog.int8in(pg_catalog.xidout(DE.xmin))) as last_tx
    from pg_catalog.pg_description DE,
         pg_catalog.pg_class DC
    where DC.oid = DE.objoid
      and DC.relnamespace = oid(:schema_id)
      and DC.relkind in ('r', 'm', 'v')
) X
;


---- ListExistentClasses ----
select distinct oid
from pg_catalog.pg_type
where typnamespace = oid(:schema_id)
union
select distinct oid
from pg_catalog.pg_class
where relnamespace = oid(:schema_id)
union
select distinct oid
from pg_catalog.pg_proc
where pronamespace = oid(:schema_id)
union
select distinct oid
from pg_catalog.pg_operator
where oprnamespace = oid(:schema_id)
union
select distinct oid
from pg_catalog.pg_constraint
where connamespace = oid(:schema_id)
;


---- RetrieveDataTypes ----
select T.oid as type_id,
       T.xmin as type_state_number,
       T.typname as type_name,
       T.typtype as type_sub_kind,
       'X' as type_category,
       T.typrelid as class_id,
       T.typbasetype as base_type_id,
       case when T.typtype in ('c','e') then null
            else pg_catalog.format_type(T.typbasetype, T.typtypmod) end as type_def,
       T.typdefault as default_expression,
       T.typnotnull as mandatory
from pg_catalog.pg_type T
         left outer join pg_catalog.pg_class C
             on T.typrelid = C.oid
where T.typnamespace = oid(:schema_id)
  #INC> and pg_catalog.int8in(pg_catalog.xidout(T.xmin)) > :tx
  and (T.typtype in ('d','e') or C.relkind = 'c'::"char" or T.typtype = 'b' and T.typelem = 0)
order by 1
;


---- RetrieveTables ----
with backup_keys as (
       select split_part(key, '_', 5) as table_id,
              value
       from pg_conf
       where split_part(key, '_', 4) = (select oid from pg_database
                                        where datname = current_database())
       and key like 'pg_class_backup_%'
       and value = 'NO'
)
select T.relkind as table_kind,
       T.relname as table_name,
       T.oid as table_id,
       T.xmin as table_state_number,
       T.relhasoids as table_with_oids,
       T.relnamespace as tablespace_id,
       T.reldiststyle as dist_style,
       (case when BK.value = 'NO' then false else true end) as bckp
from pg_catalog.pg_class T
left outer join backup_keys BK on (T.oid = BK.table_id)
where relnamespace = oid(:schema_id)
  and relkind in ('r', 'm', 'v')
  #INC> and pg_catalog.int8in(pg_catalog.xidout(T.xmin)) > :tx
order by table_kind, table_id
;


---- RetrieveExternalTables ----
select T.tablename as table_name,
       T.location,
       T.input_format,
       T.output_format,
       T.serialization_lib,
       T.serde_parameters as serde_properties,
       T.parameters as props
from pg_catalog.svv_external_tables T
where T.schemaname = :schema_name
;


---- RetrieveStructAttributes ----
with S as ( select T.oid as type_id, C.oid as class_id
            from pg_catalog.pg_type T,
                 pg_catalog.pg_class C
            where T.typnamespace = oid(:schema_id)
              and T.typnamespace = C.relnamespace
              #INC> and pg_catalog.int8in(pg_catalog.xidout(T.xmin)) > :tx
              and T.typrelid = C.oid
              and T.typtype = 'c'::"char"
              and relkind = 'c'::"char" ),
     A as ( select attrelid as class_id,
                   attnum as att_position,
                   attname as att_name,
                   pg_catalog.format_type(atttypid,atttypmod) as att_type
            from pg_catalog.pg_attribute )
select type_id, att_position, att_name, att_type
from S natural join A
order by type_id::bigint, att_position
;


---- RetrieveColumns ----
with T as ( select distinct T.oid as table_id, T.relname as table_name
            from pg_catalog.pg_class T,
                 pg_catalog.pg_attribute A
            where T.relnamespace = oid(:schema_id)
              and T.relkind in ('r', 'm', 'v')
              #INC> and pg_catalog.int8in(pg_catalog.xidout(A.xmin)) > :tx
              and A.attrelid = T.oid )
select T.table_id,
       C.attnum as column_position,
       C.attname as column_name,
       C.xmin as column_state_number,
       C.atttypmod as type_mod,
       C.attndims as dimensions_number,
       pg_catalog.format_type(C.atttypid, C.atttypmod) as type_spec,
       C.attnotnull as mandatory,
       D.adsrc as column_default_expression,
       C.attisdropped as column_is_dropped,
       format_encoding(C.attencodingtype::integer) as encoding,
       C.attisdistkey as distkey,
       C.attsortkeyord as sortkeyord
from T
  inner join pg_catalog.pg_attribute C on T.table_id = C.attrelid
  left outer join pg_catalog.pg_attrdef D on (C.attrelid, C.attnum) = (D.adrelid, D.adnum)
where attnum > 0
  and not attisdropped
order by table_id, attnum
;


---- RetrieveExternalTablesColumns ----
select C.tablename as table_name,
       C.columnnum as position,
       C.columnname as column_name,
       C.external_type as data_type,
       C.part_key as partition_key
from pg_catalog.svv_external_columns C
where C.schemaname = :schema_name
;


---- RetrieveConstraints ----
with T as ( select oid as table_id,
                   relkind as table_kind,
                   pg_catalog.int8in(pg_catalog.xidout(xmin)) as table_state_id
            from pg_catalog.pg_class
            where relkind in ('r','v')
              and relnamespace = oid(:schema_id) ),
     C as ( select conrelid as table_id,
                   oid::bigint as con_id,
                   pg_catalog.int8in(pg_catalog.xidout(xmin)) as con_state_id,
                   conname as con_name,
                   contype as con_kind,
                   conkey as con_columns,
                   confrelid as ref_table_id,
                   confkey as ref_columns,
                   consrc as con_expression
            from pg_catalog.pg_constraint
            where contype in ('p','u','f','c')
              and connamespace = oid(:schema_id) )
select table_id,
       table_kind,
       con_id,
       con_name,
       con_kind,
       '{' || array_to_string(con_columns, ',') || '}' as con_columns,
       ref_table_id,
       con_expression,
       '{' || array_to_string(ref_columns, ',') || '}' as ref_columns
from T natural join C
#INC> where table_state_id > :tx or con_state_id > :tx
order by strpos('pufc', con_kind)
;


---- RetrieveUsedArgumentTypes ----
with schema_procs as (select prorettype, proargtypes
                      from pg_catalog.pg_proc
                      where pronamespace = oid(:schema_id)
                        and pg_catalog.int8in(pg_catalog.xidout(xmin)) > :tx #. ),
     schema_opers as (select oprleft, oprright, oprresult
                      from pg_catalog.pg_operator
                      where oprnamespace = oid(:schema_id)
                        and pg_catalog.int8in(pg_catalog.xidout(xmin)) > :tx #. ),
     schema_arg_types_arrays as (select string_to_array(textin(oidvectorout(proargtypes)), ' ') as type_ids
                                 from schema_procs),
     array_idxs as (select generate_series(1,
                     (select max(array_upper(type_ids, 1))
                      from schema_arg_types_arrays)
                   ) as i),
     schema_arg_types as ( select prorettype as type_id
                           from schema_procs
                           union
                           select A.type_ids[i]::oid AS type_id
                           from array_idxs
                           inner join schema_arg_types_arrays A ON array_idxs.i <= array_upper(type_ids, 1)
                           union
                           select oprleft as type_id
                           from schema_opers
                           where oprleft is not null
                           union
                           select oprright as type_id
                           from schema_opers
                           where oprright is not null
                           union
                           select oprresult as type_id
                           from schema_opers
                           where oprresult is not null
                           )
select type_id, pg_catalog.format_type(type_id, null) as type_spec
from schema_arg_types
;


---- RetrieveRoutines ----
with languages as (select oid as lang_oid, lanname as lang
                   from pg_catalog.pg_language),
     routines as (select proname as r_name,
                         prolang as lang_oid,
                         oid as r_id,
                         xmin as r_state_number,
                         proargnames as arg_names,
                         null::varchar[] as arg_modes,
                         string_to_array(pg_catalog.textin(pg_catalog.oidvectorout(proargtypes)), ' ')::int8[] as in_arg_types,
                         null::int8[] as all_arg_types,
                         null as arg_defaults,
                         0 as arg_variadic_id,
                         prorettype as ret_type_id,
                         proretset as ret_set,
                         proisagg as is_aggregate,
                         FALSE as is_window,
                         provolatile as volatile_kind,
                         false as is_strict,
                         false as is_security_definer,
                         null::text[] as configuration_parameters,
                         0 as cost,
                         0 as rows,
                         false as is_leakproof,
                         null as concurrency_kind
                  from pg_catalog.pg_proc
                  where pronamespace = oid(:schema_id)
                   #INC: and pg_catalog.int8in(pg_catalog.xidout(xmin)) > :tx #.)
select *
from routines natural join languages
;


---- RetrieveOperators ----
select O.oid as op_id,
       oprname as op_name,
       oprkind as op_kind,
       oprleft as arg_left_type_id,
       oprright as arg_right_type_id,
       oprresult as arg_result_type_id,
       oprcode as func_main
from pg_catalog.pg_operator O
where oprnamespace = oid(:schema_id)
  #INC> and pg_catalog.int8in(pg_catalog.xidout(xmin)) > :tx
;


---- RetrieveDescriptions ----
select T.relkind as table_kind,
       T.oid as table_id,
       D.objsubid as column_position,
       D.description
from pg_catalog.pg_class T,
     pg_catalog.pg_description D
where T.relnamespace = oid(:schema_id)
  and T.relkind in ('r', 'm', 'v')
  and T.oid = D.objoid
  #INC> and pg_catalog.int8in(pg_catalog.xidout(D.xmin)) > :tx
order by table_id, column_position
;


---- RetrieveViewSources ----
select T.relkind as view_kind,
       T.oid as view_id,
       pg_catalog.pg_get_viewdef(T.oid) as source_text
from pg_catalog.pg_class T,
     pg_catalog.pg_namespace N
where N.oid = oid(:schema_id)
  and T.relnamespace = N.oid
  and T.relkind in ('m','v')
  #INC> and pg_catalog.int8in(pg_catalog.xidout(T.xmin)) > :src_tx
order by pg_catalog.int8in(pg_catalog.xidout(T.xmin))
;


---- RetrieveFunctionSources ----
with system_languages as ( select oid as lang
                           from pg_catalog.pg_language
                           where lanname in ('c','internal') )
select oid as id,
       (select distinct array_to_string(ARRAY(select (case when arg_names is null then '' else arg_names[k] + ' ' end) + pg_catalog.format_type(T.oid, null)
                                                from pg_catalog.generate_series(1, array_upper(arg_types, 1)) as k
                                                join pg_catalog.pg_type T ON T.oid = arg_types[k]
                                                order by k), ', ')
          from (select string_to_array(pg_catalog.textin(pg_catalog.oidvectorout(PGPC.proargtypes)), ' ') arg_types,
                       PGPC.proargnames arg_names
                  from pg_proc PGPC
                  where PGPC.oid = PC.oid) oids_array) as arguments_def,
       (select distinct pg_catalog.format_type(PGT.oid, null)
          from pg_catalog.pg_type PGT
          where PGT.oid = PC.prorettype) as result_def,
       prosrc as source_text
from pg_catalog.pg_proc PC
where pronamespace = oid(:schema_id)
  #INC> and pg_catalog.int8in(pg_catalog.xidout(xmin)) > :src_tx
  and not proisagg
  and prolang not in (select lang from system_languages)
  and prosrc is not null
;


---- ListReferencingSchemas ----
select distinct connamespace as schema_id
from pg_catalog.pg_constraint F,
     pg_catalog.pg_class O
where F.contype = 'f'
  and F.confrelid = O.oid
  and O.relnamespace in (SCHEMAIDS)
;