-- noinspection SqlResolveForFile


---- QueryCurrentSessionInfo ----
select db_id()                                                 as db_id,
       db_name()                                               as db_name,
       schema_id()                                             as schema_id,
       schema_name()                                           as schema_name,
       cast(databasepropertyex(db_name(),'collation') as char) as collation_name
;


---- ListDatabaseNames ----
select name
from master.sys.databases


---- ListSchemaNames ----
select name
from #CAT.schemas


---- RetrieveSchemas ----
select schema_id as id, name
from #CAT.schemas
;


---- CalculateCustomTypesCheckSums ----
with CT as ( select cast(1 + #IS_TABLE_TYPE + is_assembly_type * 2 as tinyint) as type_kind,
                    abs(cast(checksum(user_type_id, name) as bigint)) as c_s
                    from #CAT.types
                    where schema_id = ?
                      and is_user_defined >= 1 )
select type_kind, sum(c_s) as check_sum
from CT
group by type_kind
order by type_kind
;


---- RetrieveAliasTypes ----
with A as ( select user_type_id as type_id, system_type_id, name,
                   max_length, precision, scale, is_nullable
            from #CAT.types
            where schema_id = ?
              and is_user_defined >= 1
              and not (is_assembly_type >= 1)
              and not (#IS_TABLE_TYPE >= 1) )
select A.type_id, A.name,
       B.name as base_name,
       A.max_length, A.precision, A.scale, A.is_nullable
from A left outer join #CAT.types B on A.system_type_id = B.user_type_id
order by A.type_id
;


---- RetrieveTableTypes ----
with T as ( select user_type_id as type_id,
                   type_table_object_id as object_id,
                   name
            from #CAT.table_types
            where schema_id = ?
              and is_user_defined = 1
              and #IS_TABLE_TYPE = 1 )
select T.type_id, O.object_id, T.name,
       O.create_date, O.modify_date
from T join #CAT.all_objects O on T.object_id = O.object_id
order by create_date, type_id
;


---- ListExistentObjects ----
select object_id
from #CAT.all_objects
where schema_id = ?
;


---- RetrieveObjects ----
select object_id,
       name,
       rtrim(type) as type,
       create_date,
       modify_date,
       is_ms_shipped
from #CAT.all_objects
where type in ('U','S','IT','V','FN','IF','P','X','PC','TF','TT')
  and schema_id = ?
  and modify_date >= ?
order by create_date
;


---- RetrieveColumns ----
with T as ( select object_id, create_date
            from #CAT.all_objects
            where schema_id = ?
              and type in ('U','S','IT','V','TT')
              and modify_date >= ? )
select T.create_date,
       C.object_id, C.column_id as column_position, C.name,
       Y.name as type_name,
       C.max_length, C.precision, C.scale, C.collation_name,
       C.is_nullable, C.is_computed,
       cast(I.seed_value as bigint) as identity_seed_value,
       cast(I.increment_value as bigint) as identity_increment_value,
       cast(I.last_value as bigint) as identity_last_value,
       coalesce(D.definition, X.definition) as default_expression
from T join #CAT.all_columns C on T.object_id = C.object_id
       left outer join #CAT.types Y on C.system_type_id = Y.system_type_id and C.user_type_id = Y.user_type_id
       left outer join #CAT.identity_columns I on C.object_id = I.object_id and C.column_id = I.column_id
       left outer join #CAT.default_constraints D on C.object_id = D.parent_object_id and C.column_id = D.parent_column_id
       left outer join #CAT.computed_columns X on C.object_id = X.object_id and C.column_id = X.column_id
order by T.create_date, C.object_id, C.column_id
;


---- RetrieveTypeColumns ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = ?
              and is_user_defined = 1 )
select C.object_id, C.column_id as column_position, C.name,
       Y.name as type_name,
       C.max_length, C.precision, C.scale, C.collation_name,
       C.is_nullable, C.is_computed,
       cast(I.seed_value as bigint) as identity_seed_value,
       cast(I.increment_value as bigint) as identity_increment_value,
       cast(I.last_value as bigint) as identity_last_value,
       coalesce(D.definition, X.definition) as default_expression
from T join #CAT.all_columns C on T.object_id = C.object_id
       left outer join #CAT.types Y on C.system_type_id = Y.system_type_id and C.user_type_id = Y.user_type_id
       left outer join #CAT.identity_columns I on C.object_id = I.object_id and C.column_id = I.column_id
       left outer join #CAT.default_constraints D on C.object_id = D.parent_object_id and C.column_id = D.parent_column_id
       left outer join #CAT.computed_columns X on C.object_id = X.object_id and C.column_id = X.column_id
order by C.object_id, C.column_id
;


---- RetrieveIndices ----
select T.object_id,
       I.index_id % 30000 as position,
       I.name,
       I.type,
       I.is_unique,
       I.is_disabled
from #CAT.all_objects T,
     #CAT.indexes I
where T.schema_id = ?
  and T.type in ('U','S','IT','V','TT')
  and T.modify_date >= ?
  and T.object_id = I.object_id
  and I.type != 0
order by T.create_date, I.object_id, I.index_id
;


---- RetrieveTypeIndices ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = ?
              and is_user_defined = 1 )
select T.object_id,
       I.index_id % 30000 as position,
       I.name,
       I.type,
       I.is_unique,
       I.is_disabled
from T join #CAT.indexes I on T.object_id = I.object_id
where I.type != 0
order by I.object_id, I.index_id
;


---- RetrieveIndexColumns ----
select T.object_id,
       X.index_id % 30000 as index_position,
       C.name,
       X.is_descending_key as is_desc
from #CAT.all_objects T,
     #CAT.indexes I,
     #CAT.index_columns X,
     #CAT.all_columns C
where T.schema_id = ?
  and T.type in ('U','S','IT','V','TT')
  and T.modify_date >= ?
  and T.object_id = I.object_id
  and I.type != 0
  and I.object_id = X.object_id
  and I.index_id = X.index_id
  and X.object_id = C.object_id
  and X.column_id = C.column_id
order by T.create_date, X.object_id, X.index_id, X.index_column_id
;


---- RetrieveTypeIndexColumns ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = ?
              and is_user_defined = 1 )
select T.object_id,
       X.index_id % 30000 as index_position,
       C.name,
       X.is_descending_key as is_desc
from T,
     #CAT.indexes I,
     #CAT.index_columns X,
     #CAT.all_columns C
where T.object_id = I.object_id
  and I.type != 0
  and I.object_id = X.object_id
  and I.index_id = X.index_id
  and X.object_id = C.object_id
  and X.column_id = C.column_id
order by X.object_id, X.index_id, X.index_column_id
;



---- RetrieveKeys ----
select T.object_id as table_id,
       K.object_id as key_id,
       K.unique_index_id as index_position,
       K.name,
       case when K.type = 'PK' then 1 else 0 end as is_primary,
       K.is_system_named,
       K.create_date,
       K.modify_date
from #CAT.all_objects T
     join #CAT.key_constraints K on T.object_id = K.parent_object_id and T.schema_id = K.schema_id
where T.schema_id = ?
  and T.modify_date >= ?
order by T.create_date, T.object_id, is_primary desc, K.create_date, K.object_id
;


---- RetrieveTypeKeys ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = ?
              and is_user_defined = 1 )
select T.object_id as table_id,
       K.object_id as key_id,
       K.unique_index_id as index_position,
       K.name,
       case when K.type = 'PK' then 1 else 0 end as is_primary,
       K.is_system_named,
       K.create_date,
       K.modify_date
from T join #CAT.key_constraints K on T.object_id = K.parent_object_id
order by T.object_id, is_primary desc, K.create_date, K.object_id
;


---- RetrieveForeignKeys ----
select T.object_id as table_id,
       F.object_id as constraint_id,
       F.name,
       F.is_system_named,
       F.referenced_object_id,
       F.key_index_id,
       F.is_disabled,
       F.update_referential_action,
       F.delete_referential_action,
       F.create_date,
       F.modify_date
from #CAT.all_objects T
     join #CAT.foreign_keys F on T.object_id = F.parent_object_id and T.schema_id = F.schema_id
where T.schema_id = ?
  and T.modify_date >= ?
order by T.create_date, T.object_id
;


---- RetrieveForeignKeyColumns ----
select K.constraint_object_id as constraint_id,
       CD.name as domestic_column_name,
       CR.name as reference_column_name
from #CAT.all_objects T
     join #CAT.foreign_keys F on T.object_id = F.parent_object_id and T.schema_id = F.schema_id
     join #CAT.foreign_key_columns K on F.object_id = K.constraint_object_id
     left outer join #CAT.all_columns CD on K.parent_object_id = CD.object_id and K.parent_column_id = CD.column_id
     left outer join #CAT.all_columns CR on K.referenced_object_id = CR.object_id and K.referenced_column_id = CR.column_id
where T.schema_id = ?
  and T.modify_date >= ?
order by T.create_date, T.object_id, K.constraint_object_id, K.constraint_column_id
;


---- RetrieveCheckConstraints ----
select T.object_id as table_id,
       X.object_id as constraint_id,
       X.name,
       X.is_system_named,
       C.name as column_name,
       X.definition,
       X.create_date,
       X.modify_date
from #CAT.all_objects T
     join #CAT.check_constraints X on T.object_id = X.parent_object_id and T.schema_id = X.schema_id
     left outer join #CAT.all_columns C on X.parent_object_id = C.object_id and X.parent_column_id = C.column_id
where T.schema_id = ?
  and T.modify_date >= ?
order by T.create_date, T.object_id, X.parent_column_id, X.create_date, X.object_id
;


---- RetrieveTypeCheckConstraints ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = ?
              and is_user_defined = 1 )
select T.object_id as table_id,
       X.object_id as constraint_id,
       X.name,
       X.is_system_named,
       C.name as column_name,
       X.definition,
       X.create_date,
       X.modify_date
from T
     join #CAT.check_constraints X on T.object_id = X.parent_object_id
     left outer join #CAT.all_columns C on X.parent_object_id = C.object_id and X.parent_column_id = C.column_id
order by T.object_id, X.parent_column_id, X.create_date, X.object_id
;


---- RetrieveArguments ----
select P.object_id,
       P.name,
       T.name as type_name,
       P.max_length,
       P.precision,
       P.scale,
       P.is_output,
       P.is_cursor_ref,
       P.has_default_value
from #CAT.all_objects R
     join #CAT.all_parameters P on R.object_id = P.object_id
     left outer join #CAT.types T on P.system_type_id = T.system_type_id
where R.schema_id = ?
  and R.type in ('FN','IF','TF','P','X','PC')
  and R.modify_date >= ?
order by R.create_date, P.object_id, P.parameter_id
;



---- RetrieveSynonyms ----
select object_id,
       name,
       create_date,
       modify_date,
       parsename(base_object_name, 3) as origin_db_name,
       parsename(base_object_name, 2) as origin_schema_name,
       parsename(base_object_name, 1) as origin_object_name,
       object_id(base_object_name)    as origin_id
from #CAT.synonyms
where schema_id = ?
  and modify_date >= ?
;


---- RetrieveDescriptions ----
select O.object_id, X.minor_id as item_position, cast(X.value as varchar) as description
from #CAT.all_objects O,
     #CAT.extended_properties X
where O.schema_id = ?
  and O.type in ('U','V')
  and O.modify_date >= ?
  and O.object_id = X.major_id
  and X.name = N'MS_Description'
order by O.create_date, O.object_id, X.minor_id
;


---- RetrieveSources ----
select M.object_id, M.definition
from #CAT.sql_modules M
     inner join #CAT.objects O on M.object_id = O.object_id
where O.schema_id = ?
  and O.modify_date >= ?
;

