/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.dataSource;

import com.intellij.database.DatabaseFamilyId;
import com.intellij.database.model.DasModel;
import com.intellij.database.model.DasObject;
import com.intellij.database.model.DatabaseSystem;
import com.intellij.database.model.ModelSerializer;
import com.intellij.database.remote.jdbc.impl.ReflectionHelper;
import com.intellij.database.util.Case;
import com.intellij.database.util.Casing;
import com.intellij.database.util.DasUtil;
import com.intellij.database.util.LoaderContext;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.SimpleModificationTracker;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.CharsetToolkit;
import com.intellij.util.Base64;
import com.intellij.util.KeyedLazyInstance;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.ui.UIUtil;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.io.xml.XppReader;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.HashSet;

/**
 * Prefer com.intellij.database.model.DatabaseSystem.
 * @see DatabaseSystem
 * @see com.intellij.database.psi.DbDataSource
 *
 * @deprecated
 *
 * @author cdr
 */
public abstract class DataSource extends SimpleModificationTracker implements DatabaseSystem {

  public enum SaveMode {
    ALL, CONFIG, LOCAL_CONFIG, SCHEMA;
    public boolean includeSchema() { return this == ALL || this == SCHEMA;}
    public boolean includeConfig() { return this == ALL || this == CONFIG;}
    public boolean includeLocalConfig() { return this == ALL || this == LOCAL_CONFIG;}
    public boolean includeDatabaseInfo() { return includeSchema(); }
  }

  @NonNls public static final String ELEMENT_NAME = "data-source";

  private String myUUID;
  @NotNull
  private String myName = "";
  private String myComment;

  private boolean myReadOnly;

  private boolean myGlobal;

  private final Info myInfo = new Info();
  protected volatile DasModel myState = DasUtil.emptyModel();

  protected DataSource() {
  }

  public boolean isGlobal() {
    return myGlobal;
  }

  public void setGlobal(final boolean global) {
    myGlobal = global;
  }

  public String getUniqueId() {
    return myUUID;
  }

  public boolean isReadOnly() {
    return myReadOnly;
  }

  public void setReadOnly(boolean readOnly) {
    myReadOnly = readOnly;
  }

  public void deserialize(final Project project, final Reader reader) throws Exception {
    deserialize(project, new XppReader(reader), SaveMode.ALL);
  }

  public void deserialize(@Nullable final Project project, @NotNull final HierarchicalStreamReader reader, @NotNull SaveMode mode) {
    HashSet<String> legacyTags = ContainerUtil.newHashSet("table", "procedure", "schema");
    if (ELEMENT_NAME.equals(reader.getNodeName())) {
      setName(StringUtil.notNullize(reader.getAttribute("name")));
      if (mode.includeConfig()) {
        myReadOnly = "true".equals(reader.getAttribute("read-only"));
      }
      myUUID = reader.getAttribute("uuid");

      DasModel state = null;
      if (mode.includeDatabaseInfo()) {
        // todo TBR: migration
        deserializeDatabaseInfo(reader, true);
      }
      while (reader.hasMoreChildren()) {
        reader.moveDown();
        boolean skipMoveUp = false;
        String nodeName = reader.getNodeName();
        if (mode.includeDatabaseInfo() && "database-info".equals(nodeName)) {
          deserializeDatabaseInfo(reader, false);
        }
        else if (mode.includeDatabaseInfo() && "case-sensitivity".equals(nodeName)) {
          Case plain = Case.fromString(reader.getAttribute("plain-identifiers"), DasUtil.CASING_MIXED.plain);
          Case quoted = Case.fromString(reader.getAttribute("quoted-identifiers"), DasUtil.CASING_MIXED.quoted);
          myInfo.myCasing = Casing.create(plain, quoted);
        }
        // todo TBR: migration
        else if (mode.includeDatabaseInfo() && "extra-name-characters".equals(nodeName)) myInfo.myExtraNameCharacters = reader.getValue();
        else if (mode.includeDatabaseInfo() && "identifier-quote-string".equals(nodeName)) myInfo.myIdentifierQuoteString = reader.getValue();

        else if (!deserializeHeaderInner(project, reader, mode) && mode.includeSchema()) {
          String serializerName = reader.getAttribute("serializer");
          if (serializerName == null && legacyTags.contains(nodeName)) {
            serializerName = "legacy";
            skipMoveUp = true;
          }
          ModelSerializer importer = serializerName == null ? null : ModelSerializer.INSTANCES.findSingle(serializerName);
          if (importer != null) {
            state = importer.deserialize(reader);
          }
        }
        if (!skipMoveUp) {
          reader.moveUp();
        }
      }
      if (mode.includeSchema() && state != null) {
        updateState(state);
      }
    }
  }

  private void deserializeDatabaseInfo(@NotNull HierarchicalStreamReader xmlReader, boolean skipChildren) {
    myInfo.myProductName = xmlReader.getAttribute("product");
    myInfo.myProductVersion = xmlReader.getAttribute("version");
    myInfo.myJDBCVersion = xmlReader.getAttribute("jdbc-version");
    myInfo.myDriverName = xmlReader.getAttribute("driver-name");
    myInfo.myDriverVersion = xmlReader.getAttribute("driver-version");
    if (skipChildren) return; // todo TBR: migration
    while (xmlReader.hasMoreChildren()) {
      xmlReader.moveDown();
      String nodeName = xmlReader.getNodeName();
      if ("extra-name-characters".equals(nodeName)) myInfo.myExtraNameCharacters = xmlReader.getValue();
      else if ("identifier-quote-string".equals(nodeName)) myInfo.myIdentifierQuoteString = xmlReader.getValue();
      xmlReader.moveUp();
    }
  }

  protected boolean deserializeHeaderInner(@Nullable final Project project, HierarchicalStreamReader xmlReader, SaveMode mode) {
    return false;
  }

  public void serialize(@Nullable Project project, @NotNull HierarchicalStreamWriter serializer, @NotNull SaveMode mode) {
    serializer.startNode(ELEMENT_NAME);
    serializeHeader(project, serializer, mode);
    if (mode.includeSchema()) {
      ModelSerializer ms = findSerializer(getModel());
      if (ms != null) ms.serialize(getModel(), serializer);
    }
    serializer.endNode();
    serializer.flush();
  }

  @Nullable
  private static ModelSerializer findSerializer(@NotNull DasModel model) {
    for (KeyedLazyInstance<ModelSerializer> o : ModelSerializer.EP_NAME.getExtensions()) {
      if (o.getInstance().canSerialize(model)) return o.getInstance();
    }
    return null;
  }

  public void serializeHeader(Project project, @NotNull HierarchicalStreamWriter serializer, @NotNull SaveMode mode) {
    if (mode.includeConfig()) {
      serializer.addAttribute("source", getSourceName());
    }
    serializer.addAttribute("name", StringUtil.notNullize(myName));
    if (mode.includeConfig() && myReadOnly) {
      serializer.addAttribute("read-only", String.valueOf(myReadOnly));
    }
    serializer.addAttribute("uuid", StringUtil.notNullize(myUUID));

    if (mode.includeDatabaseInfo()) {
      serializer.startNode("database-info");
      serializer.addAttribute("product", StringUtil.notNullize(myInfo.myProductName));
      serializer.addAttribute("version", StringUtil.notNullize(myInfo.myProductVersion));
      serializer.addAttribute("jdbc-version", StringUtil.notNullize(myInfo.myJDBCVersion));
      serializer.addAttribute("driver-name", StringUtil.notNullize(myInfo.myDriverName));
      serializer.addAttribute("driver-version", StringUtil.notNullize(myInfo.myDriverVersion));
      if (myInfo.myExtraNameCharacters != null) {
        writeTag(serializer, "extra-name-characters", myInfo.myExtraNameCharacters);
      }
      if (myInfo.myIdentifierQuoteString != null) {
        writeTag(serializer, "identifier-quote-string", myInfo.myIdentifierQuoteString);
      }
      serializer.endNode();
      if (myInfo.myCasing.plain != DasUtil.CASING_MIXED.plain || myInfo.myCasing.quoted != DasUtil.CASING_MIXED.quoted) {
        serializer.startNode("case-sensitivity");
        serializer.addAttribute("plain-identifiers", StringUtil.toLowerCase(myInfo.myCasing.plain.name()));
        serializer.addAttribute("quoted-identifiers", StringUtil.toLowerCase(myInfo.myCasing.quoted.name()));
        serializer.endNode();
      }
    }
    serializeHeaderInner(project, serializer, mode);
  }

  protected void serializeHeaderInner(@Nullable Project project, HierarchicalStreamWriter serializer, SaveMode mode) {
  }

  @NotNull
  public String getName() {
    return myName;
  }

  @Nullable
  public String getComment() {
    return myComment;
  }

  public void setComment(@Nullable String comment) {
    myComment = comment;
  }

  public String getDatabaseProductName() {
    return myInfo.myProductName;
  }

  public String getDatabaseProductVersion() {
    return myInfo.myProductVersion;
  }

  public String getJDBCVersion() {
    return myInfo.myJDBCVersion;
  }

  public String getDriverName() {
    return myInfo.myDriverName;
  }

  public String getDriverVersion() {
    return myInfo.myDriverVersion;
  }

  public Casing getCaseModes() {
    return myInfo.myCasing;
  }

  public void init() {
    if (myUUID == null) {
      myUUID = java.util.UUID.randomUUID().toString();
    }
  }

  @NotNull
  @Override
  public DasModel getModel() {
    return myState;
  }

  protected void releaseConnection(Project project, @Nullable final Connection connection) {
    try {
      if (connection != null) {
        connection.close();
      }
    }
    catch (Exception e) {
      // ignore
    }
  }

  public abstract boolean refreshMetaData(Project project, @NotNull LoaderContext context);

  public void updateState(@NotNull final DasModel state) {
    UIUtil.invokeLaterIfNeeded(new Runnable() {
      @Override
      public void run() {
        myState = state;
        incModificationCount();
        // compatibility with legacy model
        ReflectionHelper.tryInvokeMethod(state, "setCasing", new Class[]{Casing.class}, new Object[]{myInfo.myCasing});
      }
    });
  }

  public void refreshDatabaseInfo(@NotNull DatabaseMetaData metaData) throws SQLException {
    myInfo.myProductName = metaData.getDatabaseProductName();
    myInfo.myProductVersion = metaData.getDatabaseProductVersion();
    myInfo.myDriverName = metaData.getDriverName();
    myInfo.myDriverVersion = metaData.getDriverVersion();
    try {
      myInfo.myJDBCVersion = metaData.getJDBCMajorVersion()+"."+metaData.getJDBCMinorVersion();
    }
    catch (Throwable e) {
      try {
        // since 1.2 call
        metaData.getConnection();
        myInfo.myJDBCVersion = "2.1";
      }
      catch (Throwable e1) {
        myInfo.myJDBCVersion = "1.2";
      }
    }
    try {
      myInfo.myIdentifierQuoteString = metaData.getIdentifierQuoteString();
    }
    catch (Exception ignored) {}
    try {
      myInfo.myExtraNameCharacters = metaData.getExtraNameCharacters();
    }
    catch (SQLException ignored) { }
    if (StringUtil.isEmptyOrSpaces(myInfo.myIdentifierQuoteString)) myInfo.myIdentifierQuoteString = null;
    if (StringUtil.isEmptyOrSpaces(myInfo.myExtraNameCharacters)) myInfo.myExtraNameCharacters = null;
    try {
      boolean supportsMixedCase = metaData.supportsMixedCaseIdentifiers();
      boolean storesUpperCase = metaData.storesUpperCaseIdentifiers();
      boolean storesLowerCase = metaData.storesLowerCaseIdentifiers();
      boolean storesMixedCase = metaData.storesMixedCaseIdentifiers();
      boolean supportsMixedCaseQuoted = metaData.supportsMixedCaseQuotedIdentifiers();
      boolean storesUpperCaseQuoted = metaData.storesUpperCaseQuotedIdentifiers();
      boolean storesLowerCaseQuoted = metaData.storesLowerCaseQuotedIdentifiers();
      boolean storesMixedCaseQuoted = metaData.storesMixedCaseQuotedIdentifiers();

      Case plain = grokCaseMode(supportsMixedCase, storesLowerCase, storesUpperCase, storesMixedCase);
      Case quoted = grokCaseMode(supportsMixedCaseQuoted, storesLowerCaseQuoted, storesUpperCaseQuoted, storesMixedCaseQuoted);
      myInfo.myCasing = Casing.create(plain, quoted);
    }
    catch (Exception ignored) { }
  }

  public void copyDatabaseInfo(@NotNull DataSource other) {
    myInfo.copyFrom(other.myInfo);
  }

  @NotNull
  private Case grokCaseMode(boolean sensitive, boolean forceLower, boolean forceUpper, boolean asIs) {
    DatabaseFamilyId familyId = DatabaseFamilyId.forDataSource(this);
    if (sensitive && asIs) {
      // derby, H2, oracle, sqlite, native ms drivers return both flags true
      if (familyId.isSqlite() || familyId.isMicrosoft()) return Case.MIXED;
      return Case.EXACT;
    }
    else if (asIs && familyId.isSybase()) {
      return Case.EXACT;
    }
    return sensitive? Case.EXACT : forceLower ? Case.LOWER : forceUpper ? Case.UPPER : Case.MIXED;
  }

  protected boolean shouldIncludeElement(final DasObject element) {
    return true;
  }

  public void setName(@NotNull String name) {
    myName = name;
  }

  public abstract String getSourceName();

  /**
   * Should be used in queries and not in DatabaseMetaData inquiries
   * @param identifier database entity name
   * @return identifier or its quoted version
   */
  @Nullable
  public String quoteIdentifierIfNeeded(@Nullable final String identifier) {
    if (myInfo.myIdentifierQuoteString == null || identifier == null) return identifier;
    if (identifier.startsWith(myInfo.myIdentifierQuoteString)) return identifier;
    for (int i = 0, len = identifier.length(); i<len; i++) {
      final char c = identifier.charAt(i);
      if (i == 0 && !StringUtil.isJavaIdentifierStart(c) || i > 0 && !StringUtil.isJavaIdentifierPart(c)) {
        if (myInfo.myExtraNameCharacters == null || myInfo.myExtraNameCharacters.indexOf(c) == -1) {
          return quoteIdentifier(identifier);
        }
      }
    }
    return identifier;
  }

  @Nullable
  public String quoteIdentifier(@Nullable final String identifier) {
    if (myInfo.myIdentifierQuoteString == null || identifier == null) return identifier;
    return myInfo.myIdentifierQuoteString + identifier + myInfo.myIdentifierQuoteString;
  }

  static void serializeAttribute(@NotNull HierarchicalStreamWriter serializer, String name, @Nullable String value, boolean allowEmpty) {
    if (value != null && (allowEmpty || !value.isEmpty())) {
      try {
        serializer.addAttribute(name, Base64.encode(value.getBytes(CharsetToolkit.UTF8)));
      }
      catch (UnsupportedEncodingException ignored) {
      }
    }
  }

  public static void writeTag(@NotNull HierarchicalStreamWriter serializer, String name, String value) {
    if (StringUtil.isEmpty(value)) return;
    serializer.startNode(name);
    serializer.setValue(value);
    serializer.endNode();
  }

  @NotNull
  @Override
  public String toString() {
    return getClass().getSimpleName() + "@" + Integer.toHexString(hashCode()) + ": " + getName();
  }

  @Override
  public boolean equals(Object o) {
    return this == o;
  }

  @Override
  public int hashCode() {
    return super.hashCode();
  }

  static class Info {
    String myProductName;
    String myProductVersion;
    String myJDBCVersion;
    @Nullable String myIdentifierQuoteString;
    @Nullable String myExtraNameCharacters;
    String myDriverName;
    String myDriverVersion;
    Casing myCasing = DasUtil.CASING_MIXED;

    void copyFrom(@NotNull Info info) {
      myProductName = info.myProductName;
      myProductVersion = info.myProductVersion;
      myJDBCVersion = info.myJDBCVersion;
      myIdentifierQuoteString = info.myIdentifierQuoteString;
      myExtraNameCharacters = info.myExtraNameCharacters;
      myDriverName = info.myDriverName;
      myDriverVersion = info.myDriverVersion;
      myCasing = info.myCasing;
    }
  }
}
