/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.util;

import com.intellij.database.model.*;
import com.intellij.openapi.util.Comparing;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Conditions;
import com.intellij.util.Function;
import com.intellij.util.Functions;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.JBIterable;
import com.intellij.util.containers.JBTreeTraverser;
import com.intellij.util.text.CaseInsensitiveStringHashingStrategy;
import gnu.trove.TObjectHashingStrategy;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * @author gregsh
 */
public class DasUtil {

  public static final Function<DasObject, String> TO_NAME = new Function<DasObject, String>() {
    @Override
    public String fun(DasObject t) {
      return t.getName();
    }
  };
  public static final Function<DasObject, ObjectKind> TO_KIND = new Function<DasObject, ObjectKind>() {
    @Override
    public ObjectKind fun(DasObject t) {
      return t.getKind();
    }
  };
  public static final Function<DasObject, Iterable<DasObject>> ALL_CHILDREN = new Function<DasObject, Iterable<DasObject>>() {
    @Override
    public Iterable<DasObject> fun(DasObject t) {
      return t.getDbChildren(DasObject.class, ObjectKind.NONE);
    }
  };
  public static final Function<DasObject, DasObject> TO_PARENT = new Function<DasObject, DasObject>() {
    @Override
    public DasObject fun(DasObject t) {
      DasObject o = t.getDbParent();
      return o != null && o.getKind() == ObjectKind.NONE ? null : o;
    }
  };
  public static final Condition<DasArgument> OUTPUT_ARGUMENT = new Condition<DasArgument>() {
    @Override
    public boolean value(@NotNull DasArgument definition) {
      return definition.getArgumentDirection().isOut();
    }
  };
  public static final Condition<DasArgument> INPUT_ARGUMENT = new Condition<DasArgument>() {
    @Override
    public boolean value(@NotNull DasArgument definition) {
      DasArgument.Direction type = definition.getArgumentDirection();
      return type.isIn() || type == DasArgument.Direction.SELF;
    }
  };
  public static final Condition<DasArgument> PARAMETER = new Condition<DasArgument>() {
    @Override
    public boolean value(@NotNull DasArgument definition) {
      return !definition.getArgumentDirection().isReturnOrResult();
    }
  };
  public static final String NO_NAME = new String("");
  public static final Set<DasColumn.Attribute> NO_ATTRS = Collections.emptySet();
  public static final Casing CASING_MIXED = Casing.create(Case.MIXED, Case.EXACT);
  public static final Casing CASING_EXACT = Casing.create(Case.EXACT, Case.EXACT);
  public static final CasingProvider NO_CASING_PROVIDER = new CasingProvider() {
    @NotNull
    @Override
    public Casing getCasing(@NotNull ObjectKind kind, @Nullable DasObject context) {
      return CASING_MIXED;
    }
  };

  private DasUtil() {
  }

  public static <F, T> Function<F, T> uncheckedCast() {
    return new Function<F, T>() {
      @SuppressWarnings("unchecked")
      @Override
      public T fun(F f) {
        return (T)f;
      }
    };
  }

  @NotNull
  public static <V> Map<String, V> newCasingAwareMap(boolean sensitive) {
    return ContainerUtil.newTroveMap(
      sensitive ? (TObjectHashingStrategy<String>)TObjectHashingStrategy.CANONICAL
                : CaseInsensitiveStringHashingStrategy.INSTANCE);
  }

  @NotNull
  public static Set<String> newCasingAwareSet(boolean sensitive) {
    return ContainerUtil.newTroveSet(
      sensitive ? (TObjectHashingStrategy<String>)TObjectHashingStrategy.CANONICAL
                : CaseInsensitiveStringHashingStrategy.INSTANCE);
  }

  private static final Function<DasObject, Iterable<? extends DasObject>> DAS_STRUCTURE = new Function<DasObject, Iterable<? extends DasObject>>() {
    @Override
    public Iterable<? extends DasObject> fun(DasObject o) {
      return o.getDbChildren(DasObject.class, ObjectKind.NONE);
    }
  };
  private static final JBTreeTraverser<DasObject> DAS_TRAVERSER = new JBTreeTraverser<DasObject>(DAS_STRUCTURE);
  @NotNull
  public static JBTreeTraverser<DasObject> dasTraverser() {
    return DAS_TRAVERSER;
  }

  @NotNull
  public static <C> Condition<C> byKind(@NotNull final ObjectKind kind) {
    if (kind == ObjectKind.NONE) return Conditions.alwaysTrue();
    Condition<DasObject> result = new Condition<DasObject>() {
      @Override
      public boolean value(DasObject object) {
        return object.getKind() == kind;
      }
    };
    return (Condition<C>)result;
  }

  @NotNull
  public static <C> Condition<C> byKindNot(@NotNull final ObjectKind kind) {
    if (kind == ObjectKind.NONE) return Conditions.alwaysTrue();
    Condition<DasObject> result = new Condition<DasObject>() {
      @Override
      public boolean value(DasObject object) {
        return object.getKind() != kind;
      }
    };
    return (Condition<C>)result;
  }

  @NotNull
  public static <C> Condition<C> byName(@Nullable String name) {
    return byName(name, NO_CASING_PROVIDER);
  }

  @NotNull
  public static <C> Condition<C> byName(@Nullable final String name, @NotNull final CasingProvider casing) {
    if (name == null) return Conditions.alwaysFalse();
    Condition<DasObject> result = new Condition<DasObject>() {
      @Override
      public boolean value(DasObject object) {
        return nameEqual(object, name, casing);
      }
    };
    return (Condition<C>)result;
  }

  @NotNull
  public static <C> Condition<C> byClass(@NotNull final Class clazz) {
    return Conditions.instanceOf(clazz);
  }

  @NotNull
  public static DasModel emptyModel() {
    return EmptyModel.INSTANCE;
  }

  @NotNull
  public static DasModel loadingModel() {
    return EmptyModel.LOADING;
  }

  public static boolean nameEqual(@NotNull DasObject obj, @Nullable String name, @NotNull CasingProvider casing) {
    return nameEqual(obj, name, casing.getCasing(obj.getKind(), obj));
  }

  public static boolean nameEqual(@NotNull DasObject obj, @Nullable String name, @NotNull Casing casing) {
    return equal(obj.getName(), name, casing);
  }

  public static boolean isCaseSensitive(@NotNull Casing casing) {
    return casing.plain == Case.EXACT;
  }

  public static boolean equal(@Nullable String name1, @Nullable String name2, @NotNull Casing casing) {
    return Comparing.equal(name1, name2, isCaseSensitive(casing));
  }

  @NotNull
  public static <T> MultiRef<T> emptyMultiRef() {
    //noinspection unchecked
    return EmptyMultiRef.INSTANCE;
  }

  @NotNull
  public static <S, T> MultiRef<? extends T> transform(@NotNull MultiRef<S> ref, @NotNull Function<? super S, T> fun) {
    return new MappedMultiRef<S, T>(ref, fun);
  }

  @NotNull
  public static <T> MultiRef.It<T> emptyMultiRefIterator() {
    //noinspection unchecked
    return EmptyMultiRef.INSTANCE.iterate();
  }

  @Nullable
  public static DasObject getParentOfKind(@Nullable DasObject object, @Nullable ObjectKind kind, boolean strict) {
    if (kind == null) return null;
    for (DasObject o = strict && object != null? object.getDbParent() : object; o != null; o = o.getDbParent()) {
      if (o.getKind() == kind) return o;
    }
    return null;
  }

  @Nullable
  public static <T> T getParentOfClass(@Nullable DasObject object, @Nullable Class<? extends T> clazz, boolean strict) {
    if (clazz == null) return null;
    for (DasObject o = strict && object != null? object.getDbParent() : object; o != null; o = o.getDbParent()) {
      if (clazz.isInstance(o)) return (T)o;
    }
    return null;
  }

  @Nullable
  public static DasNamespace getNamespace(@Nullable DasObject object) {
    return getParentOfClass(object, DasNamespace.class, true);
  }

  @NotNull
  public static JBIterable<DasObject> dasParents(@Nullable final DasObject object) {
    if (object == null) return JBIterable.empty();
    return JBIterable.generate(object, TO_PARENT);
  }

  @Nullable
  public static <C> C findChild(@Nullable DasObject parent, Class<C> clazz, ObjectKind kind, String name) {
    return parent == null ? null : parent.getDbChildren(clazz, kind).filter(byName(name)).first();
  }

  @Nullable
  public static DasObject getCatalogObject(@Nullable DasObject object) {
    return getParentOfKind(object, ObjectKind.DATABASE, false);
  }

  @Nullable
  public static DasObject getSchemaObject(@Nullable DasObject object) {
    return getParentOfKind(object, ObjectKind.SCHEMA, false);
  }

  @NotNull
  public static String getName(@Nullable DasObject object) {
    return object == null ? NO_NAME : object.getName();
  }

  @NotNull
  public static String getCatalog(@Nullable DasObject object) {
    return getName(getCatalogObject(object));
  }

  @NotNull
  public static String getSchema(@Nullable DasObject object) {
    return getName(getSchemaObject(object));
  }

  public static <T extends DasObject> MultiRef<T> asRef(final Iterable<T> objects) {
    //noinspection unchecked
    return asRef(objects, TO_NAME, Function.ID);
  }

  public static <S, T extends DasObject> MultiRef<T> asRef(final Iterable<S> objects, final Function<? super S, String> namer, final Function<S, T> resolver) {
    final JBIterable<S> fi = JBIterable.from(objects);
    return new MultiRef<T>() {
      @Override
      public It<T> iterate() {
        final Iterator<S> it = objects.iterator();
        return new It<T>() {
          S cur;

          @Nullable
          @Override
          public T resolve() {
            return resolver.fun(cur);
          }

          @Override
          public boolean hasNext() {
            return it.hasNext();
          }

          @Override
          public String next() {
            return namer.fun(cur = it.next());
          }

          @Override
          public void remove() {
            throw new UnsupportedOperationException();
          }
        };
      }

      @Override
      public Iterable<String> names() {
        return fi.transform(namer);
      }

      @Override
      public Iterable<? extends T> resolveObjects() {
        return fi.transform(resolver).filter(Condition.NOT_NULL);
      }

      @Override
      public int size() {
        return fi.size();
      }
    };
  }

  @Nullable
  public static DasObject resolveFinalTarget(@Nullable DasSynonym synonym) {
    int k = 9;
    DasObject o = synonym;
    while (o instanceof DasSynonym && k-- > 0) {
      o = ((DasSynonym)o).resolveTarget();
    }
    return k == 0 ? null : o;
  }

  // todo remove the following?
  public static boolean isPrimary(@NotNull DasColumn column) {
    return column.getTable().getColumnAttrs(column).contains(DasColumn.Attribute.PRIMARY_KEY);
  }

  public static boolean isForeign(@NotNull DasColumn column) {
    return column.getTable().getColumnAttrs(column).contains(DasColumn.Attribute.FOREIGN_KEY);
  }

  public static boolean isAutoVal(@NotNull DasColumn column) {
    return column.getTable().getColumnAttrs(column).contains(DasColumn.Attribute.AUTO_GENERATED);
  }

  public static boolean isAncestor(@Nullable DasObject ancestor, @Nullable DasObject element, boolean strict) {
    if (ancestor == null || element == null) return false;
    if (ancestor == element) return !strict;
    for (DasObject object : dasParents(element)) {
      if (object == ancestor) return true;
    }
    return false;
  }

  @NotNull
  public static JBIterable<? extends DasNamespace> getSchemas(@NotNull DatabaseSystem info) {
    return getSchemas(info.getModel());
  }

  @NotNull
  public static JBIterable<? extends DasNamespace> getSchemas(@NotNull DasModel model) {
    return model.traverser().expand(byKindNot(ObjectKind.SCHEMA)).filter(byKind(ObjectKind.SCHEMA)).filter(DasNamespace.class);
  }

  @NotNull
  public static JBIterable<? extends DasNamespace> getNamespaces(@NotNull DatabaseSystem info) {
    return info.getModel().traverser().expand(byKindNot(ObjectKind.SCHEMA)).filter(DasNamespace.class);
  }

  public static JBIterable<? extends DasSchemaChild> getPackages(@NotNull DatabaseSystem info) {
    return getSchemaElements(info, DasSchemaChild.class).filter(byKind(ObjectKind.PACKAGE));
  }

  public static JBIterable<? extends DasTable> getTables(@NotNull DatabaseSystem info) {
    return getSchemaElements(info, DasTable.class);
  }

  public static <T> JBIterable<? extends T> getSchemaElements(@NotNull DatabaseSystem info, Class<T> clazz) {
    return info.getModel().traverser().expandAndSkip(byClass(DasNamespace.class)).filter(clazz);
  }

  public static JBIterable<? extends DasColumn> getColumns(@NotNull DasObject table) {
    return table.getDbChildren(DasColumn.class, ObjectKind.COLUMN);
  }

  public static JBIterable<? extends DasForeignKey> getForeignKeys(@NotNull DasTable table) {
    return table.getDbChildren(DasForeignKey.class, ObjectKind.FOREIGN_KEY);
  }

  public static JBIterable<? extends DasIndex> getIndices(@NotNull DasTable table) {
    return table.getDbChildren(DasIndex.class, ObjectKind.INDEX);
  }

  public static JBIterable<? extends DasTableKey> getTableKeys(@NotNull DasTable table) {
    return table.getDbChildren(DasTableKey.class, ObjectKind.KEY);
  }

  @Nullable
  public static DasTableKey getPrimaryKey(@NotNull DasTable table) {
    for (DasTableKey key : getTableKeys(table)) {
      if (key.isPrimary()) return key;
    }
    return null;
  }

  public static JBIterable<? extends DasTrigger> getTriggers(DasTable table) {
    return table.getDbChildren(DasTrigger.class, ObjectKind.TRIGGER);
  }

  public static boolean containsName(@NotNull String name, @NotNull MultiRef<?> ref) {
    return ContainerUtil.find(ref.names(), name) != null; // todo case insensitive?
  }

  @NotNull
  public static JBIterable<? extends DasArgument> getParameters(@NotNull DasRoutine routine) {
    return JBIterable.from(routine.getArguments()).filter(PARAMETER);
  }

  @NotNull
  public static <C> JBIterable<C> getDbChildren(@NotNull DasObject object, @NotNull DasModel model, @NotNull Class<C> clazz, @NotNull ObjectKind kind) {
    JBTreeTraverser<DasObject> traverser = model == emptyModel() ? dasTraverser() : model.traverser();
    return traverser.withRoot(object).expandAndSkip(Conditions.is(object)).filter(byKind(kind)).filter(clazz);
  }

  private static class EmptyMultiRef implements MultiRef, MultiRef.It {

    static final EmptyMultiRef INSTANCE = new EmptyMultiRef();

    private EmptyMultiRef() {
    }

    @Override
    public It iterate() {
      return this;
    }

    @Override
    public Iterable<String> names() {
      return Collections.emptyList();
    }

    @Override
    public Iterable resolveObjects() {
      return JBIterable.empty();
    }

    @Override
    public int size() {
      return 0;
    }

    @Nullable
    @Override
    public Object resolve() {
      throw new NoSuchElementException();
    }

    @Override
    public boolean hasNext() {
      return false;
    }

    @Override
    public Object next() {
      throw new NoSuchElementException();
    }

    @Override
    public void remove() {
      throw new UnsupportedOperationException();
    }
  }

  private static class MappedMultiRef<S, T> implements MultiRef<T> {

    final MultiRef<? extends S> original;
    final Function<? super S, T> fun;

    private MappedMultiRef(MultiRef<? extends S> original, Function<? super S, T> fun) {
      this.original = original;
      this.fun = fun;
    }

    @Override
    public It<T> iterate() {
      final It<? extends S> it = original.iterate();
      return new It<T>() {
        @Nullable
        @Override
        public T resolve() {
          return fun.fun(it.resolve());
        }

        @Override
        public boolean hasNext() {
          return it.hasNext();
        }

        @Override
        public String next() {
          return it.next();
        }

        @Override
        public void remove() {
          it.remove();
        }
      };
    }

    @Override
    public Iterable<String> names() {
      return original.names();
    }

    @Override
    public Iterable<T> resolveObjects() {
      return JBIterable.from(original.resolveObjects()).transform(fun).filter(Condition.NOT_NULL);
    }

    @Override
    public int size() {
      return original.size();
    }

  }

  private static class EmptyModel extends JBTreeTraverser<DasObject> implements DasModel {

    private static final EmptyModel INSTANCE = new EmptyModel("empty");
    private static final EmptyModel LOADING = new EmptyModel("loading");
    final String debugName;

    private EmptyModel(String debugName) {
      super(Functions.<DasObject, Iterable<DasObject>>constant(JBIterable.<DasObject>empty()));
      this.debugName = debugName;
    }

    @NotNull
    @Override
    public JBIterable<? extends DasObject> getModelRoots() {
      return JBIterable.empty();
    }

    @Nullable
    @Override
    public DasNamespace getCurrentRoot() {
      return null;
    }

    @NotNull
    @Override
    public Casing getCasing(@NotNull ObjectKind kind, @Nullable DasObject context) {
      return CASING_MIXED;
    }

    @NotNull
    @Override
    public JBTreeTraverser<DasObject> traverser() {
      return this;
    }

    @NotNull
    @Override
    protected JBTreeTraverser<DasObject> newInstance(Meta<DasObject> meta) {
      return this;
    }

    @NotNull
    @Override
    public JBIterable<? extends DasConstraint> getExportedKeys(DasTable table) {
      return JBIterable.empty();
    }

    @Override
    public String toString() {
      return debugName;
    }
  }
}
