package com.intellij.sql.psi;

import com.intellij.lang.InjectableLanguage;
import com.intellij.lang.Language;
import com.intellij.lang.LanguageUtil;
import com.intellij.sql.dialects.SqlLanguageDialect;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * @author gregsh
 */
public class SqlDialectsUtil {

  private static final Map<String, SqlLanguageDialect> ourDialectMap;

  static {
    Map<String, SqlLanguageDialect> map = ContainerUtil.newLinkedHashMap();
    for (Language language : SqlLanguage.INSTANCE.getDialects()) {
      if (!(language instanceof SqlLanguageDialect)) continue;
      map.put(language.getID(), (SqlLanguageDialect)language);
    }
    ourDialectMap = Collections.unmodifiableMap(map);
  }

  public static final SqlLanguageDialect GENERIC;
  static {
    GENERIC = findById("GenericSQL");
    if (GENERIC == null) {
      throw new AssertionError("<Generic> SQL dialect not found");
    }
  }

  private SqlDialectsUtil() {
  }

  @NotNull
  public static SqlLanguage getSqlLanguage() {
    return SqlLanguage.INSTANCE;
  }

  @NotNull
  public static Collection<SqlLanguageDialect> getSqlDialects() {
    return ourDialectMap.values();
  }

  @NotNull
  public static List<SqlLanguageDialect> getTopLevelSqlDialects() {
    List<SqlLanguageDialect> result = ContainerUtil.newArrayList(getSqlDialects());
    for (Iterator<SqlLanguageDialect> it = result.iterator(); it.hasNext(); ) {
      if (it.next() instanceof InjectableLanguage) it.remove();
    }
    Collections.sort(result, LanguageUtil.LANGUAGE_COMPARATOR);
    return result;
  }

  @Nullable
  public static SqlLanguageDialect findById(@NotNull String id) {
    return ourDialectMap.get(id);
  }

}
