/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.sql.psi;

import com.intellij.database.model.DataType;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * @author Gregory.Shrago
 */
public abstract class SqlTableType extends SqlType {

  public boolean isPrimitive() {
    return false;
  }

  @Nullable
  public abstract String getTypeName();

  @Nullable
  public abstract PsiElement getTypeElement();

  public abstract int getColumnCount();

  public abstract String getColumnName(int i);

  public abstract SqlType getColumnType(int i);

  public abstract DataType getDataType(int i);

  @NotNull
  public abstract PsiElement getColumnElement(int i);

  @Nullable
  public abstract PsiElement getSourceColumnElement(int i);

  @Nullable
  public abstract PsiElement getColumnQualifier(int i);

  @Nullable
  public abstract String getColumnTypeAlias(int i);

  public abstract SqlTableType join(SqlTableType type);

  public abstract SqlTableType add(SqlTableType type);

  public abstract SqlTableType subtract(PsiElement element);

  public abstract SqlTableType alias(String aliasName, PsiElement aliasElement, List<? extends SqlNameElement> newColumns);

  public String getDisplayName() {
    final StringBuilder sb = new StringBuilder();
    sb.append(StringUtil.notNullize(getTypeName())).append("(");
    boolean first = true;
    for (int i = 0, len = getColumnCount(); i < len; i++) {
      if (first) first = false;
      else sb.append(", ");
      DataType dataType = getDataType(i);
      SqlType type = getColumnType(i);
      sb.append(getColumnName(i));
      if (dataType != DataType.UNKNOWN) {
        sb.append(":").append(dataType.getSpecification());
      }
      else if (type != UNKNOWN) {
        sb.append(":").append(type.getDisplayName());
      }
    }
    sb.append(")");
    return sb.toString();
  }
}