/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.facet;

import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.module.Module;
import com.intellij.psi.PsiFile;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Set;

/**
 * @author Yann C&eacute;bron
 */
public abstract class SpringFileSetService {

  public static SpringFileSetService getInstance() {
    return ServiceManager.getService(SpringFileSetService.class);
  }

  public abstract String getUniqueId(final Set<SpringFileSet> existing);

  public abstract String getUniqueName(String prefix, final Set<SpringFileSet> existing);

  /**
   * Returns all configured and provided file sets.
   *
   * @param facet Facet instance.
   * @return all working file sets for the facet.
   * @see com.intellij.spring.facet.SpringFacet#getFileSets()
   * @see #getModelProviderSets(com.intellij.spring.facet.SpringFacet)
   */
  @NotNull
  public abstract Set<SpringFileSet> getAllSets(final @NotNull SpringFacet facet);

  /**
   * Returns filesets provided by all {@link com.intellij.spring.SpringModelProvider}s.
   *
   * @param facet Facet instance.
   * @return filesets provided by {@link com.intellij.spring.SpringModelProvider}.
   * @see com.intellij.spring.SpringModelProvider
   */
  @NotNull
  public abstract List<SpringFileSet> getModelProviderSets(final @NotNull SpringFacet facet);

  /**
   * @param module  Module to search usage in.
   * @param psiFile Configuration file.
   * @return {@code null} if given file not configured.
   * @since 14
   */
  @Nullable
  public abstract SpringFileSet findFileSet(Module module, PsiFile psiFile);

  // cross-fileset/module dependency management (internal)

  abstract String getQualifiedName(@NotNull SpringFileSet fileset);

  abstract String getDependencyIdFor(SpringFileSet current, SpringFileSet otherFileSet);

  @Nullable
  abstract SpringFileSet findDependencyFileSet(SpringFileSet current, @NotNull final String dependencyId);
}
