/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model;

import com.intellij.openapi.util.Comparing;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiClass;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.SearchScope;
import com.intellij.psi.search.searches.ClassInheritorsSearch;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.util.Query;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

public abstract class SpringModelSearchParameters {

  protected SpringModelSearchParameters() {
  }

  /**
   * Returns whether given search parameters are valid to start a search.
   *
   * @return {@code false} if search cannot be performed.
   */
  public abstract boolean canSearch();

  public static BeanClass byClass(@NotNull PsiClass psiClass) {
    return new BeanClass(psiClass);
  }

  public static class BeanClass extends SpringModelSearchParameters {
    private final PsiClass myPsiClass;

    private final boolean myCanSearch;

    private boolean myWithInheritors;

    @Nullable
    private GlobalSearchScope myInheritorsSearchScope;
    private Collection<PsiClass> myInheritors;
    private boolean myEffectiveBeanTypes;

    private BeanClass(@NotNull PsiClass psiClass) {
      myPsiClass = psiClass;
      myCanSearch = SpringCommonUtils.isSpringBeanCandidateClass(psiClass);
    }

    @Override
    public boolean canSearch() {
      return myCanSearch;
    }

    public BeanClass withInheritors() {
      myWithInheritors = true;
      return this;
    }

    /**
     * Use with library classes search to avoid using their default use scope.
     *
     * @param inheritorsSearchScope Search scope for searching inheritors, e.g. module.
     * @return This.
     * @since 15
     */
    public BeanClass withInheritors(GlobalSearchScope inheritorsSearchScope) {
      myWithInheritors = true;
      myInheritorsSearchScope = inheritorsSearchScope;
      return this;
    }

    public BeanClass effectiveBeanTypes() {
      myEffectiveBeanTypes = true;
      return this;
    }

    @NotNull
    public PsiClass getSearchClass() {
      return myPsiClass;
    }

    public boolean isWithInheritors() {
      return myWithInheritors;
    }

    @NotNull
    private SearchScope getInheritorsSearchScope() {
      return myInheritorsSearchScope == null ? myPsiClass.getUseScope() : myInheritorsSearchScope;
    }

    @NotNull
    public Collection<PsiClass> getInheritors() {
      if (myInheritors == null) {
        final Query<PsiClass> inheritorsQuery = ClassInheritorsSearch.search(myPsiClass, getInheritorsSearchScope(), true, true, false);
        myInheritors = inheritorsQuery.findAll();
      }
      return myInheritors;
    }

    public boolean isEffectiveBeanTypes() {
      return myEffectiveBeanTypes;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (!(o instanceof BeanClass)) return false;

      BeanClass beanClass = (BeanClass)o;

      if (myEffectiveBeanTypes != beanClass.myEffectiveBeanTypes) return false;
      if (myWithInheritors != beanClass.myWithInheritors) return false;
      if (myCanSearch != beanClass.myCanSearch) return false;
      if (!myPsiClass.equals(beanClass.getSearchClass())) return false;
      if (!Comparing.equal(myInheritorsSearchScope, beanClass.myInheritorsSearchScope)) return false;
      return true;
    }

    @Override
    public int hashCode() {
      int result = myPsiClass.hashCode();
      result = 31 * result + (myWithInheritors ? 1 : 0);
      result = 31 * result + (myEffectiveBeanTypes ? 1 : 0);
      if (myInheritorsSearchScope != null) {
        result = 31 * result + myInheritorsSearchScope.hashCode();
      }
      return result;
    }
  }

  public static BeanName byName(@NotNull String beanName) {
    return new BeanName(beanName);
  }

  public static class BeanName extends SpringModelSearchParameters {

    @NotNull
    private final String myBeanName;
    private final boolean myCanSearch;

    private BeanName(@NotNull String beanName) {
      myBeanName = beanName;
      myCanSearch = StringUtil.isNotEmpty(beanName);
    }

    @Override
    public boolean canSearch() {
      return myCanSearch;
    }

    @NotNull
    public String getBeanName() {
      return myBeanName;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (!(o instanceof BeanName)) return false;

      BeanName name = (BeanName)o;

      if (myCanSearch != name.myCanSearch) return false;
      if (!myBeanName.equals(name.myBeanName)) return false;

      return true;
    }

    @Override
    public int hashCode() {
      return myBeanName.hashCode();
    }
  }
}
