/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.actions.patterns;

import com.intellij.codeInsight.generation.PatternDescriptor;
import com.intellij.codeInsight.generation.PatternProvider;
import com.intellij.codeInsight.template.Template;
import com.intellij.codeInsight.template.impl.TemplateSettings;
import com.intellij.internal.statistic.UsageTrigger;
import com.intellij.openapi.actionSystem.*;
import com.intellij.psi.PsiFile;
import com.intellij.psi.xml.XmlFile;
import com.intellij.spring.model.actions.GenerateSpringDomElementAction;
import com.intellij.spring.dom.SpringDomUtils;
import com.intellij.spring.statistics.SpringStatisticsConstants;
import com.intellij.util.xml.actions.generate.AbstractDomGenerateProvider;
import org.jetbrains.annotations.NotNull;

import javax.swing.*;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Dmitry Avdeev
 */
public abstract class PatternActionAdapter extends PatternDescriptor implements PatternProvider {

  private String myId;
  private List<PatternDescriptor> myChildren = new ArrayList<PatternDescriptor>();

  public PatternActionAdapter(String id) {
    myId = id;
    addMe();
  }

  protected void addMe() {
    myChildren.add(this);
  }

  protected void add(final GenerateSpringDomElementAction action) {
    myChildren.add(new PatternDescriptor() {

      @NotNull
      @Override
      public String getParentId() {
        return myId;
      }

      @NotNull
      @Override
      public String getName() {
        return action.getTemplatePresentation().getText();
      }

      @Override
      public Icon getIcon() {
        return action.getTemplatePresentation().getIcon();
      }

      @Override
      public Template getTemplate() {
        String mappingId = getMappingId();
        return TemplateSettings.getInstance().getTemplateById(mappingId);
      }

      private String getMappingId() {
        return ((AbstractDomGenerateProvider)action.getProvider()).getMappingId();
      }

      @Override
      public void actionPerformed(DataContext context) {
        UsageTrigger.trigger(SpringStatisticsConstants.USAGE_TRIGGER_PREFIX + "GenerateByPattern." + getParentId() + "." + getMappingId());
        action.actionPerformed(AnActionEvent.createFromAnAction(action, null, ActionPlaces.UNKNOWN, context));
      }
    });
  }

  @NotNull
  @Override
  public String getId() {
    return myId;
  }

  @NotNull
  @Override
  public String getParentId() {
    return ROOT;
  }

  @NotNull
  @Override
  public String getName() {
    return getDescription();
  }

  @Override
  public Template getTemplate() {
    return null;
  }

  @Override
  public void actionPerformed(DataContext context) {

  }


  protected abstract String getDescription();

  protected void addSeparator() {

  }

  public List<PatternDescriptor> getChildren() {
    return myChildren;
  }

  @Override
  public final PatternDescriptor[] getDescriptors() {
    List<PatternDescriptor> children = getChildren();
    return children.toArray(new PatternDescriptor[children.size()]);
  }

  @Override
  public final boolean isAvailable(DataContext context) {
    PsiFile file = CommonDataKeys.PSI_FILE.getData(context);
    return file instanceof XmlFile && SpringDomUtils.isSpringXml((XmlFile)file);
  }
}
