/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.highlighting.dom;

import com.intellij.psi.PsiClass;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringApiBundle;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.utils.SpringModelUtils;
import com.intellij.spring.model.xml.RequiredBeanType;
import com.intellij.util.Function;
import com.intellij.util.Processor;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomJavaUtil;
import com.intellij.util.xml.DomUtil;
import com.intellij.util.xml.GenericAttributeValue;
import com.intellij.util.xml.highlighting.DefineAttributeQuickFix;
import com.intellij.util.xml.highlighting.DomElementAnnotationHolder;
import com.intellij.util.xml.highlighting.RemoveDomElementQuickFix;

/**
 * @author Yann C&eacute;bron
 */
public class SpringDomInspectionUtils {

  private static final Function<DomElement, String> DOM_NAME_FUNCTION = new Function<DomElement, String>() {
    @Override
    public String fun(DomElement value) {
      return value.getXmlElementName();
    }
  };

  private final DomElementAnnotationHolder holder;

  public SpringDomInspectionUtils(DomElementAnnotationHolder holder) {
    this.holder = holder;
  }

  public static boolean hasAny(DomElement... values) {
    for (DomElement value : values) {
      if (DomUtil.hasXml(value)) {
        return true;
      }
    }
    return false;
  }

  public static boolean hasMoreThanOne(DomElement... values) {
    short count = 0;
    for (DomElement value : values) {
      if (DomUtil.hasXml(value)) {
        count++;
        if (count > 1) {
          return true;
        }
      }
    }
    return false;
  }

  public void oneOfRequired(DomElement element,
                            DomElement... elements) {
    if (hasMoreThanOne(elements) || !hasAny(elements)) {
      onlyOneOfProblem(element, elements);
    }
  }

  public void oneOfOrAllRequired(DomElement element,
                                 GenericAttributeValue... values) {
    if (!hasAny(values)) {
      holder.createProblem(element, getDomElementNamesMessage("spring.dom.one.or.all.of.attributes", values));
    }
  }

  public boolean onlyOneOf(DomElement element,
                           GenericAttributeValue... values) {
    if (hasMoreThanOne(values)) {
      onlyOneOfProblem(element, values);
      return true;
    }
    return false;
  }

  public void ifExistsOtherRequired(DomElement element,
                                    GenericAttributeValue exists,
                                    GenericAttributeValue required) {
    if (hasAny(exists) &&
        !hasAny(required)) {

      final String requiredName = DOM_NAME_FUNCTION.fun(required);
      holder.createProblem(element,
                           SpringApiBundle.message("spring.dom.if.exists.other.required",
                                                   DOM_NAME_FUNCTION.fun(exists),
                                                   requiredName),
                           new DefineAttributeQuickFix(requiredName));
    }
  }

  public void attributeSuperfluous(GenericAttributeValue value) {
    if (hasAny(value)) {
      holder.createProblem(value,
                           SpringApiBundle.message("spring.dom.superfluous.attribute",
                                                   DOM_NAME_FUNCTION.fun(value)),
                           new RemoveDomElementQuickFix(value)).highlightWholeElement();
    }
  }

  public <T> void attributeWithDefaultSuperfluous(GenericAttributeValue<T> value,
                                                  T defaultValue) {
    if (hasAny(value) &&
        defaultValue.equals(value.getValue())) {
      holder.createProblem(value,
                           SpringApiBundle.message("spring.dom.superfluous.attribute.with.default",
                                                   DOM_NAME_FUNCTION.fun(value),
                                                   value.getStringValue()),
                           new RemoveDomElementQuickFix(value)).highlightWholeElement();
    }
  }

  private void onlyOneOfProblem(DomElement element, DomElement... elements) {
    holder.createProblem(element, getDomElementNamesMessage("spring.dom.only.one.of", elements));
  }

  private static String getDomElementNamesMessage(String key,
                                                  DomElement... elements) {
    StringBuilder elementNames = new StringBuilder();
    for (int i = 0; i < elements.length; i++) {
      String elementName = DOM_NAME_FUNCTION.fun(elements[i]);
      if (i == elements.length - 1) {
        elementNames.append(" or ");
      }
      else if (i != 0) {
        elementNames.append(", ");
      }
      if (elements[i] instanceof GenericAttributeValue) {
        elementNames.append("'").append(elementName).append("'");
      }
      else {
        elementNames.append("<").append(elementName).append(">");
      }
    }

    return SpringApiBundle.message(key, elementNames.toString());
  }

  public void beanOfType(GenericAttributeValue<SpringBeanPointer> springBeanAttribute, String beanClass) {
    SpringBeanPointer pointer = springBeanAttribute.getValue();
    if (pointer == null) {
      return;
    }

    if (!InheritanceUtil.isInheritor(pointer.getBeanClass(), false, beanClass)) {
      holder.createProblem(springBeanAttribute,
                           SpringApiBundle.message("bean.must.be.of.type", beanClass));
    }
  }

  public void explicitBeanRequired(DomElement domElement,
                                   GenericAttributeValue<SpringBeanPointer> springBeanAttribute,
                                   String defaultBeanName) {
    if (hasAny(springBeanAttribute)) {
      return;
    }

    final RequiredBeanType annotation = springBeanAttribute.getAnnotation(RequiredBeanType.class);
    assert annotation != null : springBeanAttribute;
    final String[] classNames = annotation.value();
    for (String beanClass : classNames) {
      if (existsBean(domElement, beanClass, defaultBeanName)) {
        return;
      }
    }

    holder.createProblem(domElement,
                         SpringApiBundle.message("spring.dom.explicit.bean.reference.required",
                                                 springBeanAttribute.getPresentation().getTypeName(),
                                                 defaultBeanName),
                         new DefineAttributeQuickFix(DOM_NAME_FUNCTION.fun(springBeanAttribute)));
  }

  public static boolean existsBean(DomElement domElement, String beanClass, final String defaultBeanName) {
    final PsiClass psiClass = DomJavaUtil.findClass(beanClass, domElement);
    if (psiClass == null) {
      return false;
    }

    final SpringModelSearchParameters.BeanClass searchParameters =
      SpringModelSearchParameters.byClass(psiClass).withInheritors().effectiveBeanTypes();
    if (!searchParameters.canSearch()) return false;

    final CommonSpringModel model = SpringModelUtils.getInstance().getSpringModel(domElement.getXmlTag());
    return !model.processByClass(searchParameters, new Processor<SpringBeanPointer>() {
      @Override
      public boolean process(SpringBeanPointer pointer) {
        return !defaultBeanName.equals(pointer.getName());
      }
    });
  }
}
