/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.jam.JamElement;
import com.intellij.jam.JamService;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiConnector;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiMember;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.DefaultSpringBeanQualifier;
import com.intellij.spring.model.SpringProfile;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.jam.profile.SpringJamProfile;
import com.intellij.spring.model.jam.qualifiers.SpringJamQualifier;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import com.intellij.util.ArrayUtil;
import com.intellij.util.SmartList;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.annotation.ElementType;
import java.util.Collection;
import java.util.List;

public abstract class JamPsiMemberSpringBean<T extends PsiMember> extends CommonModelElement.PsiBase
  implements JamElement, CommonSpringBean {

  public static final SemKey<JamPsiMemberSpringBean> PSI_MEMBER_SPRING_BEAN_JAM_KEY =
    JamService.JAM_ALIASING_ELEMENT_KEY.subKey("PsiMemberSpringBean");

  @NotNull
  @JamPsiConnector
  public abstract T getPsiElement();

  @Override
  @JamPsiValidity
  public abstract boolean isValid();

  @Nullable
  public PsiClass getBeanClass(final boolean considerFactories) {
    return getBeanClass();
  }

  @NotNull
  public String[] getAliases() {
    return ArrayUtil.EMPTY_STRING_ARRAY;
  }

  protected List<String> getStringNames(List<JamStringAttributeElement<String>> elements) {
    List<String> aliases = new SmartList<String>();
    for (JamStringAttributeElement<String> element : elements) {
      String aliasName = element.getStringValue();
      if (!StringUtil.isEmptyOrSpaces(aliasName)) {
        aliases.add(aliasName);
      }
    }
    return aliases;
  }

  public SpringQualifier getSpringQualifier() {
    SpringJamQualifier jamQualifier = getQualifier();
    return jamQualifier == null ? DefaultSpringBeanQualifier.create(this) : jamQualifier;
  }

  @Nullable
  public SpringJamQualifier getQualifier() {
    final Module module = getModule();
    if (module == null) {
      return null;
    }

    return SpringJamQualifier.findSpringJamQualifier(module, getPsiElement());
  }

  public boolean isPrimary() {
    return AnnotationUtil.isAnnotated(getPsiElement(), SpringAnnotationsConstants.PRIMARY, false);
  }

  @NotNull
  @Override
  public SpringProfile getProfile() {
    final Module module = getModule();
    if (module == null) {
      return SpringProfile.DEFAULT;
    }

    final JamAnnotationTypeUtil jamAnnotationTypeUtil = JamAnnotationTypeUtil.getInstance(module);
    final Collection<PsiClass> annotationTypeClasses =
      jamAnnotationTypeUtil.getAnnotationTypesWithChildren(SpringAnnotationsConstants.PROFILE);

    PsiClass profiledPsiElement = getProfiledPsiElement();
    if (profiledPsiElement != null) {
      for (PsiClass annotationTypeClass : annotationTypeClasses) {
        if (JamCommonUtil.isAcceptedFor(annotationTypeClass, ElementType.TYPE)) {
          final String qname = annotationTypeClass.getQualifiedName();
          if (qname != null) {
            final PsiAnnotation annotation = AnnotationUtil.findAnnotation(profiledPsiElement, true, qname);
            if (annotation != null) {
              return new SpringJamProfile(annotation, profiledPsiElement);
            }
          }
        }
      }
    }

    return SpringProfile.DEFAULT;
  }

  @Nullable
  protected PsiClass getProfiledPsiElement() {
    return null;
  }
}
