/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.jam.JamElement;
import com.intellij.jam.reflect.JamMemberMeta;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.util.Pair;
import com.intellij.patterns.PsiClassPattern;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.semantic.SemKey;
import com.intellij.semantic.SemRegistrar;
import com.intellij.semantic.SemService;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import com.intellij.util.Consumer;
import com.intellij.util.Function;
import com.intellij.util.NullableFunction;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;

public final class SpringSemContributorUtil {
  private SpringSemContributorUtil() {
  }

  public static <T extends JamElement> void registerMetaComponents(@NotNull final SemService semService,
                                                                   @NotNull SemRegistrar registrar,
                                                                   @NotNull PsiClassPattern psiClassPattern,
                                                                   @NotNull final SemKey<JamMemberMeta<PsiClass, T>> metaKey,
                                                                   @NotNull final SemKey<T> semKey,
                                                                   @NotNull final NullableFunction<PsiClass, JamMemberMeta<PsiClass, T>> metaFunction) {
    registrar.registerSemElementProvider(metaKey, psiClassPattern, metaFunction);

    registrar.registerSemElementProvider(semKey, psiClassPattern,
                                         new NullableFunction<PsiClass, T>() {
                                           public T fun(PsiClass member) {
                                             final JamMemberMeta<PsiClass, T> memberMeta = semService.getSemElement(metaKey, member);
                                             return memberMeta != null ? memberMeta.createJamElement(PsiElementRef.real(member)) : null;
                                           }
                                         }
    );
  }

  /**
   * @deprecated use {@link #createFunction(SemKey, Class, Function, Function, Consumer)}. To remove in IDEA 16.
   */
  public static <T extends SpringStereotypeElement> NullableFunction<PsiClass, JamMemberMeta<PsiClass, T>> getMetaFunction(@NotNull final SemKey<T> semKey,
                                                                                                                           @NotNull final Class<T> jamClass,
                                                                                                                           @NotNull final Function<Module, Collection<String>> annotationsGetter,
                                                                                                                           @NotNull final Function<Pair<String, PsiClass>, T> producer) {
    return createFunction(semKey, jamClass, annotationsGetter, producer, SpringSemContributorUtil.<T>createStereotypeConsumer());
  }

  /**
   * @since 15
   */
  public static <T extends JamElement> NullableFunction<PsiClass, JamMemberMeta<PsiClass, T>> createFunction(@NotNull final SemKey<T> semKey,
                                                                                                             @NotNull final Class<T> jamClass,
                                                                                                             @NotNull final Function<Module, Collection<String>> annotationsGetter,
                                                                                                             @NotNull final Function<Pair<String, PsiClass>, T> producer,
                                                                                                             @Nullable final Consumer<JamMemberMeta<PsiClass, T>> metaConsumer) {
    return new NullableFunction<PsiClass, JamMemberMeta<PsiClass, T>>() {
      public JamMemberMeta<PsiClass, T> fun(final PsiClass psiClass) {
        if (DumbService.isDumb(psiClass.getProject())) return null;
        if (psiClass.isAnnotationType()) return null;
        final Module module = ModuleUtilCore.findModuleForPsiElement(psiClass);
        for (final String anno : annotationsGetter.fun(module)) {
          if (AnnotationUtil.isAnnotated(psiClass, anno, true)) {
            final JamMemberMeta<PsiClass, T> meta = new JamMemberMeta<PsiClass, T>(null, jamClass, semKey) {
              @Override
              public T createJamElement(PsiElementRef<PsiClass> psiMemberPsiRef) {
                return producer.fun(Pair.create(anno, psiMemberPsiRef.getPsiElement()));
              }
            };
            if (metaConsumer != null) {
              metaConsumer.consume(meta);
            }
            return meta;
          }
        }
        return null;
      }
    };
  }

  /**
   * @param <T>
   * @return Consumer.
   * @see SpringStereotypeElement#addPomTargetProducer(JamMemberMeta)
   * @since 15
   */
  public static <T extends SpringStereotypeElement> Consumer<JamMemberMeta<PsiClass, T>> createStereotypeConsumer() {
    return new Consumer<JamMemberMeta<PsiClass, T>>() {
      @Override
      public void consume(JamMemberMeta<PsiClass, T> meta) {
        SpringStereotypeElement.addPomTargetProducer(meta);
      }
    };
  }

  /**
   * @param anno Annotation FQN.
   * @return Custom annotation types.
   * @since 15
   */
  public static Function<Module, Collection<String>> getCustomMetaAnnotations(@NotNull final String anno) {
    return new Function<Module, Collection<String>>() {
      @Override
      public Collection<String> fun(Module module) {
        if (module == null) return Collections.emptySet();
        Collection<PsiClass> psiClasses =
          JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildren(anno);

        return ContainerUtil.mapNotNull(psiClasses, new NullableFunction<PsiClass, String>() {
          public String fun(final PsiClass psiClass) {
            String qualifiedName = psiClass.getQualifiedName();
            if (anno.equals(qualifiedName)) return null;

            return qualifiedName;
          }
        });
      }
    };
  }
}
