/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.converters;

import com.intellij.jam.JamSimpleReferenceConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.lang.properties.PropertiesImplUtil;
import com.intellij.lang.properties.psi.PropertiesFile;
import com.intellij.lang.properties.psi.impl.PropertiesFileImpl;
import com.intellij.openapi.util.Condition;
import com.intellij.psi.*;
import com.intellij.spring.model.utils.SpringReferenceUtils;
import com.intellij.spring.model.utils.resources.SpringResourcesBuilder;
import com.intellij.spring.model.utils.resources.SpringResourcesUtil;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.NullableFunction;
import com.intellij.util.PairProcessor;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.HashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public class PropertiesFileConverter extends JamSimpleReferenceConverter<Set<PropertiesFile>> {

  @Override
  public Set<PropertiesFile> fromString(@Nullable String s, JamStringAttributeElement<Set<PropertiesFile>> context) {

    final PsiAnnotationMemberValue psiElement = context.getPsiElement();
    if (s != null && psiElement != null) {
      return new HashSet<PropertiesFile>(getResourceFiles(psiElement, s, ",",
                                                          new Condition<PsiFileSystemItem>() {
                                                            public boolean value(final PsiFileSystemItem item) {
                                                              return item instanceof PropertiesFileImpl;
                                                            }
                                                          }));
    }
    return Collections.emptySet();
  }

  @NotNull
  @Override
  public PsiReference[] createReferences(JamStringAttributeElement<Set<PropertiesFile>> context) {
    return getReferences(context);
  }

  private static PsiReference[] getReferences(JamStringAttributeElement<Set<PropertiesFile>> context) {
    PsiLiteral psiLiteral = context.getPsiLiteral();
    if (psiLiteral == null) return PsiReference.EMPTY_ARRAY;

    String s = ElementManipulators.getValueText(psiLiteral);

    return SpringResourcesUtil.getInstance()
      .getReferences(SpringResourcesBuilder.create(psiLiteral, s).fromRoot(s.startsWith("/")).soft(false));
  }

  private static Collection<PropertiesFile> getResourceFiles(final @NotNull PsiElement element,
                                                             final @NotNull String s,
                                                             final String delimiter,
                                                             final Condition<PsiFileSystemItem> filter) {
    final List<PsiReference> references = new ArrayList<PsiReference>();
    if (element instanceof PsiLiteral) {
      addPsiLiteralExpressionReferences((PsiLiteral)element, s, delimiter, references);
    } else {
      try {
        final PsiExpression fakeExpression =
          JavaPsiFacade.getElementFactory(element.getProject()).createExpressionFromText("\"" + s + "\"", element);
        if (fakeExpression instanceof PsiLiteral) {
          addPsiLiteralExpressionReferences((PsiLiteral)fakeExpression, s, delimiter, references);
        }
      }
      catch (IncorrectOperationException e) {
        // ignore
      }
    }

    Collection<PsiFile> files =
      SpringResourcesUtil.getInstance().getResourceItems(references.toArray(new PsiReference[references.size()]), filter);

    return ContainerUtil.mapNotNull(files, new NullableFunction<PsiFile, PropertiesFile>() {
      @Override
      public PropertiesFile fun(PsiFile psiFile) {
        return PropertiesImplUtil.getPropertiesFile(psiFile);
      }
    });
  }

  private static boolean addPsiLiteralExpressionReferences(@NotNull final PsiLiteral element,
                                                           @NotNull String s,
                                                           @NotNull String delimiter, final
                                                           @NotNull List<PsiReference> references) {
    final int startInElement = ElementManipulators.getOffsetInElement(element);
    return SpringReferenceUtils.processSeparatedString(s, delimiter, new PairProcessor<String, Integer>() {
      @Override
      public boolean process(final String s, final Integer offset) {
        final SpringResourcesBuilder builder =
          SpringResourcesBuilder.create(element, s).offset(offset.intValue() + startInElement);
        ContainerUtil.addAll(references, SpringResourcesUtil.getInstance().getReferences(builder));
        return true;
      }
    });
  }
}
