/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.psi.css;

import com.intellij.psi.NavigatablePsiElement;
import com.intellij.psi.PsiElement;
import com.intellij.util.ArrayFactory;
import com.intellij.util.IncorrectOperationException;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

public interface CssDeclaration extends CssOneLineStatement, CssDescriptorOwner, CssValueOwner, NavigatablePsiElement {
  CssDeclaration[] EMPTY_ARRAY = new CssDeclaration[0];

  ArrayFactory<CssDeclaration> ARRAY_FACTORY = new ArrayFactory<CssDeclaration>() {
    @NotNull
    @Override
    public CssDeclaration[] create(final int count) {
      return count == 0 ? EMPTY_ARRAY : new CssDeclaration[count];
    }
  };

  /**
   * Returns property name of declaration element.
   * Use {@link com.intellij.psi.css.impl.util.table.CssDescriptorsUtil#getCanonicalPropertyName} in order
   * to get real property name.
   *
   * @return property name
   */
  @NotNull
  String getPropertyName();

  @Nullable
  CssTermList getValue();

  boolean isImportant();

  boolean isShorthandProperty();

  /**
   * Returns property names which values are defined inside this shorthand property
   * Note: this method will not return property-names which values are not defined!
   *
   * @return empty array if no values defined or non-shorthand property
   */
  String[] expandShorthandProperty();

  @Nullable
  String[] getShorthandValue(@NotNull final String propertyName);

  @Nullable
  PsiElement[] getShorthandPsiValue(@NotNull final String propertyName);

  boolean isEqualTo(Object object);

  int equalityHashCode();

  void setValue(final String propertyValue) throws IncorrectOperationException;

  /**
   * @deprecated use {@link this#getDescriptors()}
   */
  @Nullable
  CssPropertyDescriptor getDescriptor();

  boolean isHack();

  @Nullable
  PsiElement getPropertyNameElement();

  @NotNull
  @Override
  Collection<? extends CssPropertyDescriptor> getDescriptors();
  
  @NotNull
  @Override
  Collection<? extends CssPropertyDescriptor> getDescriptors(@NotNull PsiElement context);
}
