/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.psi.css;

import com.intellij.css.util.CssPsiUtil;
import com.intellij.lang.css.CSSLanguage;
import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.fileTypes.FileType;
import com.intellij.openapi.fileTypes.LanguageFileType;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.util.ArrayUtil;
import com.intellij.util.IncorrectOperationException;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class CssElementFactory {
  private final PsiManager myManager;

  public CssElementFactory(PsiManager _manager) {
    myManager = _manager;
  }

  public static CssElementFactory getInstance(@NotNull Project project) {
    return ServiceManager.getService(project, CssElementFactory.class);
  }

  @NotNull
  public CssString createString(char quoteChar, String value, @Nullable FileType fileType) {
    final String stringText = quoteChar + value + quoteChar;
    LanguageFileType languageFileType = getFileType(fileType);
    final CssFile fileByFileType = getFileByFileType(fileType);
    String declarationsTerminator = CssPsiUtil.getDeclarationsTerminatorFromContext(fileByFileType);
    final String importText = "@import " + stringText + declarationsTerminator;
    final CssFile cssFile = (CssFile)PsiFileFactory.getInstance(myManager.getProject())
      .createFileFromText("dummy.css", languageFileType, importText);
    final CssImport imports = PsiTreeUtil.findChildOfType(cssFile, CssImport.class);
    final CssString cssString = PsiTreeUtil.getChildOfType(imports, CssString.class);
    if (cssString == null) {
      throw new IllegalArgumentException("Can't create css string with text: " + stringText);
    }
    return cssString;
  }

  @NotNull
  public PsiElement createStringToken(char quoteChar, String value, @Nullable FileType fileType) {
    final CssString string = createString(quoteChar, value, fileType);
    assert string.getChildren().length > 0;
    return string.getFirstChild();
  }

  @NotNull
  public CssStylesheet createStylesheet(@NotNull String text, @Nullable FileType fileType) {
    fileType = getFileType(fileType);
    final PsiFile fileFromText = PsiFileFactory.getInstance(myManager.getProject()).createFileFromText("_dummy_", fileType, text);
    if (!(fileFromText instanceof CssFile)) {
      throw new IllegalArgumentException("Can't create ruleset for given file type: " + fileFromText.getFileType().getName());
    }
    CssStylesheet stylesheet = ((CssFile)fileFromText).getStylesheet();
    if (stylesheet == null) {
      throw new IllegalArgumentException("Can't create css stylesheet from text: " + text);
    }
    return stylesheet;
  }

  @NotNull
  public CssRuleset createRuleset(@NotNull @NonNls String text, @Nullable FileType fileType) {
    final CssRuleset[] rulesets = createStylesheet(text, fileType).getRulesets();
    if (rulesets.length == 0) {
      throw new IllegalArgumentException("Can't create ruleset from text: " + text);
    }
    final CssRuleset ruleset = rulesets[0];
    if (ruleset == null) {
      throw new IllegalArgumentException("Can't create ruleset from text: " + text);
    }
    return ruleset;
  }

  @NotNull
  public CssRuleset createRuleset(@NotNull String selector, @NotNull CssDeclaration[] declarations, @Nullable FileType fileType) {
    final CssFile cssFile = getFileByFileType(fileType);
    String declarationsTerminator = CssPsiUtil.getDeclarationsTerminatorFromContext(cssFile);
    StringBuilder blockText = new StringBuilder();
    for (CssDeclaration declaration : declarations) {
      blockText.append("  ").append(declaration.getPropertyName()).append(": ");
      final CssTermList declarationValue = declaration.getValue();
      if (declarationValue != null) {
        blockText.append(declarationValue.getText());
      }
      blockText.append(declarationsTerminator).append("\n");
    }
    return createRuleset(createRulesetText(selector, blockText, cssFile).toString(), fileType);
  }

  @NotNull
  public CssDeclaration[] createProperties(@NotNull String text, @Nullable FileType fileType) {
    final StringBuilder rulesetText = createRulesetText("div", new StringBuilder(text), fileType);
    CssBlock block = createRuleset(rulesetText.toString(), fileType).getBlock();
    if (block == null) {
      throw new IllegalArgumentException("Can't create css declarations from text: " + text);
    }
    return block.getDeclarations();
  }

  @NotNull
  public CssDeclaration createProperty(@NotNull String propertyText, @Nullable FileType fileType) {
    CssDeclaration[] properties = createProperties(propertyText, fileType);
    if (properties.length == 0) {
      throw new IllegalArgumentException("Can't create css property from text: " + propertyText);
    }
    return properties[0];
  }

  @NotNull
  public CssDeclaration createProperty(@NotNull String name, @NotNull String value, @Nullable FileType fileType) {
    final CssFile cssFile = (CssFile)PsiFileFactory.getInstance(myManager.getProject()).createFileFromText("", getFileType(fileType), "");
    String declarationsTerminator = CssPsiUtil.getDeclarationsTerminatorFromContext(cssFile);
    return createProperty(name + ": " + value + declarationsTerminator, fileType);
  }
  
  @Nullable
  public CssMedia createMedia(@NotNull String mediaQuery, @Nullable FileType fileType) {
    String mediaText = createRulesetText("@media " + mediaQuery + "{}", new StringBuilder(), fileType).toString();
    CssStylesheet stylesheet = createStylesheet(mediaText, fileType);
    for (CssAtRule rule : stylesheet.getRulesetList().getAtRules()) {
      if (rule instanceof CssMedia) {
        return (CssMedia)rule;
      }
    }
    return null;
  }

  @NotNull
  public CssMediaFeature createMediaFeature(@NotNull String name, @NotNull String value, @Nullable FileType fileType) {
    String mediaText = createRulesetText("@media screen and (" + name + ":" + value + ")", new StringBuilder(), fileType).toString();
    CssStylesheet stylesheet = createStylesheet(mediaText, fileType);
    for (CssAtRule rule : stylesheet.getRulesetList().getAtRules()) {
      if (rule instanceof CssMedia) {
        CssMediumList mediumList = ((CssMedia)rule).getMediumList();
        if (mediumList != null) {
          CssMediaQuery mediaQuery = ArrayUtil.getFirstElement(mediumList.getMediaQueries());
          if (mediaQuery != null) {
            CssMediaExpression mediaExpression = ArrayUtil.getFirstElement(mediaQuery.getExpressions());
            CssMediaFeature mediaFeature = PsiTreeUtil.getChildOfType(mediaExpression, CssMediaFeature.class);
            if (mediaFeature != null) {
              return mediaFeature;
            }
          }
        }
      }
    }

    throw new IllegalArgumentException("Can't create css media feature: " + mediaText);
  }

  @NotNull
  public CssTermList createPropertyValue(@NotNull String value, @Nullable FileType fileType) {
    final CssTermList propertyValue = createProperty("width", value, fileType).getValue();
    if (propertyValue == null) {
      throw new IllegalArgumentException("Can't create css property value: " + value);
    }
    return propertyValue;
  }

  @NotNull
  public CssDescriptorOwner createFunction(@NotNull String name, @Nullable FileType type) {
    CssTerm term = createTerm(name + "(param)", type);
    CssFunction function = PsiTreeUtil.findChildOfType(term, CssFunction.class);
    if (function == null) {
      throw new IllegalArgumentException("Can't create css function with name: " + name);
    }
    return function;
  }

  @NotNull
  public CssTerm createTerm(@NotNull String value, @Nullable FileType fileType) {
    CssTermList propertyValue = createPropertyValue(value, fileType);
    final CssTerm term = PsiTreeUtil.findChildOfType(propertyValue, CssTerm.class);
    if (term == null) {
      throw new IllegalArgumentException("Can't create css term with value: " + value);
    }
    return term;
  }

  @NotNull
  public CssElement createExpressionParameter(@NotNull String value, @Nullable FileType fileType) {
    CssTermList propertyValue = createPropertyValue("expression(" + value + ")", fileType);
    final CssExpression expression = PsiTreeUtil.findChildOfType(propertyValue, CssExpression.class);
    final CssElement expressionParameter = expression != null ? expression.getValueNode() : null;
    if (expressionParameter == null) {
      throw new IllegalArgumentException("Can't create css expression with value: " + value);
    }
    return expressionParameter;
  }

  @NotNull
  public CssPseudoSelector createPseudoSelector(String name, @Nullable FileType fileType) {
    return createPseudoSelector(name, fileType, false);
  }

  @NotNull
  public CssPseudoSelector createPseudoSelector(String name, @Nullable FileType fileType, boolean isEscaped) {
    assert name.startsWith(":");
    String selector = isEscaped ? "\\" + name : name;
    CssRuleset ruleset = createRuleset(createRulesetText(selector, new StringBuilder(), fileType).toString(), fileType);
    final CssPseudoSelector pseudoSelector = PsiTreeUtil.findChildOfType(ruleset, CssPseudoSelector.class);
    if (pseudoSelector == null) {
      throw new IllegalArgumentException("Can't create css pseudo selector from text: " + name);
    }
    return pseudoSelector;
  }

  @NotNull
  public CssSelectorList createSelectorList(@NotNull String text, @Nullable FileType fileType) {
    CssRuleset ruleset = createRuleset(createRulesetText(text, new StringBuilder(), fileType).toString(), fileType);
    final CssSelectorList selectorList = PsiTreeUtil.findChildOfType(ruleset, CssSelectorList.class);
    if (selectorList == null) {
      throw new IllegalArgumentException("Can't create css selector list from text: " + text);
    }
    return selectorList;
  }

  @NotNull
  public CssSimpleSelector createSimpleSelector(@NotNull String simpleSelectorText, @Nullable FileType fileType) {
    CssRuleset ruleset = createRuleset(simpleSelectorText + " {}", fileType);
    final CssSimpleSelector simpleSelector = PsiTreeUtil.findChildOfType(ruleset, CssSimpleSelector.class);
    if (simpleSelector == null) {
      throw new IllegalArgumentException("Can't create css simple selector from text: " + simpleSelectorText);
    }
    return simpleSelector;
  }

  @NotNull
  public CssImportList createImportList(@Nullable FileType fileType, String... imports) {
    String text = StringUtil.join(imports, ";");
    final CssStylesheet stylesheet = createStylesheet(text, fileType);
    final CssImportList importList = stylesheet.getImportList();
    if (importList == null) {
      throw new IllegalArgumentException("Can't create css import list from text: " + text);
    }
    return importList;
  }

  @NotNull
  public PsiElement createToken(@NotNull String text, @Nullable FileType fileType) {
    fileType = getFileType(fileType);

    final PsiFile fileFromText = PsiFileFactory.getInstance(myManager.getProject()).createFileFromText("_dummy_", fileType, text);
    final PsiElement leaf = PsiTreeUtil.getDeepestFirst(fileFromText);
    final PsiElement result = leaf instanceof PsiErrorElement ? PsiTreeUtil.nextVisibleLeaf(leaf) : leaf;
    if (result == null) {
      throw new IllegalArgumentException("Can't create css token from text: " + text);
    }
    return result;
  }

  @NotNull
  public PsiComment createComment(@NotNull String commentText) {
    final PsiFile fileFromText = PsiFileFactory.getInstance(myManager.getProject()).createFileFromText("_dummy_", CssFileType.INSTANCE,
                                                                                                       commentText);
    final PsiComment comment = PsiTreeUtil.findChildOfType(fileFromText, PsiComment.class);
    if (comment != null) {
      return comment;
    }
    throw new IncorrectOperationException("Invalid comment text: " + commentText);
  }

  @NotNull
  public CssUri createUri(String string, @Nullable FileType type) {
    final CssTermList termList = createPropertyValue("url(" + string + ")", type);
    CssUri cssUri = PsiTreeUtil.findChildOfType(termList, CssUri.class);
    if (cssUri == null) {
      throw new IllegalArgumentException("Can't create css uri with value: " + string);
    }
    return cssUri;
  }

  @NotNull
  private static LanguageFileType getFileType(@Nullable FileType fileType) {
    return fileType instanceof LanguageFileType && ((LanguageFileType)fileType).getLanguage().isKindOf(CSSLanguage.INSTANCE)
           ? (LanguageFileType)fileType
           : CssFileType.INSTANCE;
  }

  @NotNull
  private StringBuilder createRulesetText(@NotNull String selector, @NotNull StringBuilder blockText, @Nullable FileType fileType) {
    final CssFile cssFile = (CssFile)PsiFileFactory.getInstance(myManager.getProject()).createFileFromText("", getFileType(fileType), "");
    return createRulesetText(selector, blockText, cssFile);
  }

  @NotNull
  private static StringBuilder createRulesetText(@NotNull String selector, @NotNull StringBuilder blockText, @NotNull CssFile cssFile) {
    boolean identBasedCssLanguage = CssPsiUtil.isIndentBasedCssLanguage(cssFile);
    if (identBasedCssLanguage) {
      if (!blockText.toString().startsWith(" ")) {
        blockText.insert(0, "\n  ");
      } else {
        blockText.insert(0, '\n');
      }
    } else {
      blockText.insert(0, " {\n");
    }
    blockText.insert(0, selector);
    if (!identBasedCssLanguage) {
      blockText.append('}');
    }
    return blockText;
  }

  @NotNull
  private CssFile getFileByFileType(FileType fileType) {
    return (CssFile)PsiFileFactory.getInstance(myManager.getProject()).createFileFromText("", getFileType(fileType), "");
  }
}
