/// <reference path="v8debugger.d.ts" />
/// <reference path="DebuggerSupport.ts" />

"use strict";

module $_JbV8DebuggerSupport {
  import Request = v8debug.Request
  import DebugCommandProcessor = v8debug.DebugCommandProcessor
  import BreakEvent = v8debug.BreakEvent
  import CompileEvent = v8debug.CompileEvent
  import ScriptMirror = v8debug.ScriptMirror
  import BreakPoint = v8debug.BreakPoint
  import ExecutionState = v8debug.ExecutionState

  const debuggerSupport = new DebuggerSupport()

  DebugCommandProcessor.prototype.super$processDebugJSONRequest = DebugCommandProcessor.prototype.processDebugJSONRequest
  DebugCommandProcessor.prototype.processDebugJSONRequest = function (jsonRequest: string): string {
    var request: Request
    try {
      request = JSON.parse(jsonRequest)
    }
    catch (e) {
      //noinspection TypeScriptUnresolvedVariable
      return '{"success": false, "type": "internal error", "message": ' + JSON.stringify('Cannot parse incoming message:' + jsonRequest + "\n\n" + e) + '}'
    }

    var debugCommandProcessor: DebugCommandProcessor = <DebugCommandProcessor>this
    try {
      if (request.command == null) {
        throw new Error('Command not specified')
      }

      var result: string;
      //noinspection FallThroughInSwitchStatementJS
      switch (request.command) {
        case 'getScopeObject':
          result = debuggerSupport.getScopeObject(request.arguments, debugCommandProcessor.exec_state_);
          break;

        case 'getFrames':
          result = debuggerSupport.getFrames(request.arguments, debugCommandProcessor.exec_state_);
          break;

        case 'getFrameReceiver':
          result = debuggerSupport.getFrameReceiver(request.arguments, debugCommandProcessor.exec_state_);
          break;

        case 'getObjects':
          result = debuggerSupport.getObjects(request.arguments);
          break;

        case 'getProperties':
          result = debuggerSupport.getProperties(request.arguments);
          break;

        case 'getIndexedProperties':
          result = debuggerSupport.getIndexedProperties(request.arguments);
          break;

        case 'getFunction':
          result = debuggerSupport.getFunction(request.arguments)
          break;

        case 'getScripts':
          var scripts = v8debug.Debug.scripts()
          var descriptors = <any>[]

          var ids = request.arguments == null ? null : request.arguments.ids
          var idsToInclude: { [id: number]: boolean; }
          if (ids != null) {
            idsToInclude = {}
            for (var i = 0; i < ids.length; i++) {
              idsToInclude[ids[i]] = true
            }
          }

          for (var i = 0, n = scripts.length; i < n; i++) {
            var script = scripts[i]
            if (ids == null) {
              if (script.type === 0) {
                continue
              }
            }
            else if (!idsToInclude[script.id]) {
              continue
            }

            var descriptor = debuggerSupport.describeScript(script)
            if (descriptor != null) {
              descriptors.push(descriptor)
            }
          }
          result = '{"scripts": ' + JSON.stringify(descriptors) + '}'
          break

        case 'evaluate':
          result = debuggerSupport.evaluate(request.arguments, debugCommandProcessor.exec_state_);
          break;

        case 'continue':
        case 'disconnect':
          cacheManager.clearCache()
        // fall through

        default:
          return debugCommandProcessor.super$processDebugJSONRequest(jsonRequest);
      }

      return '{"request_seq": ' + request.seq +
        ', "success": true' +
        ', "running": ' + debugCommandProcessor.running_ +
        ', "body": ' + result +
        '}';
    }
    catch (e) {
      //noinspection TypeScriptUnresolvedVariable
      return '{"request_seq": ' + request.seq +
        ', "success": false' +
        ', "running": ' + debugCommandProcessor.running_ +
        ', "message":' + JSON.stringify(e.stack || e.toString()) +
        '}';
    }
  }

  if (BreakEvent != null) {
    BreakEvent.prototype.super$toJSONProtocol = BreakEvent.prototype.toJSONProtocol
    BreakEvent.prototype.toJSONProtocol = function (): string {
      try {
        var breakPointsHit = this.breakPointsHit()
        if (breakPointsHit != null) {
          var breakpointIds = new Array<number>(breakPointsHit.length)
          for (var i = 0; i < breakPointsHit.length; i++) {
            var breakpoint = breakPointsHit[i]
            var scriptBreakpoint = breakpoint.script_break_point()
            breakpointIds[i] = scriptBreakpoint == null ? breakpoint.number() : scriptBreakpoint.number()
          }
        }

        var objects: Array<ObjectDescriptor> = []
        var frame = this.frame_ || this.exec_state_.frame(0)
        return '{"event": "paused", "body": {"frame": ' + JSON.stringify(debuggerSupport.describeFrame(frame, objects)) + $_JbV8DebuggerSupport.stringifyList(breakpointIds, "breakpoints") + $_JbV8DebuggerSupport.stringifyObjects(objects) + '}}'
      }
      catch (e) {
        return errorResponse(e, "break")
      }
    }
  }

  function errorResponse(e: any, eventName: string) {
    return '{"success": false, "message": ' + JSON.stringify('Error while sending ' + eventName + ' event:' + "\n\n" + (e.stack || e.message)) + '}';
  }

  if (CompileEvent != null) {
    CompileEvent.prototype.super$toJSONProtocol = CompileEvent.prototype.toJSONProtocol
    CompileEvent.prototype.toJSONProtocol = function (): string {
      var event: CompileEvent = <CompileEvent>this
      try {
        if (event.eventType() != 5) {
          return event.super$toJSONProtocol()
        }

        var script: v8debug.Script = event.script().value()
        // skip native script
        var descriptor = script.type === 0 ? null : debuggerSupport.describeScript(script)
        if (descriptor == null) {
          return '{"event": "scriptParsed", "ignoredScriptId": ' + (script.id == null ? '""' : script.id)  + '}'
        }
        else {
          return '{"event": "scriptParsed", "body": {"script": ' + JSON.stringify(descriptor) + '}}'
        }
      }
      catch (e) {
        return errorResponse(e, "scriptParsed")
      }
    }
  }

  if (BreakPoint != null) {
    BreakPoint.prototype.super$isTriggered = BreakPoint.prototype.isTriggered
    BreakPoint.prototype.isTriggered = function (state: ExecutionState): boolean {
      var breakPoint: BreakPoint = <BreakPoint>this
      if (!breakPoint.active()) {
        return false;
      }

      var condition = breakPoint.condition()
      if (condition != null && condition.lastIndexOf("// jb sourcemap condition", 0) === 0) {
        var script = state.frame(0).func().script()
        if (script != null) {
          return script.source().lastIndexOf('sourceMappingURL=') !== -1
        }
      }
      return breakPoint.super$isTriggered(state)
    }
  }
}