/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiClass;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.jam.JamSpringBeanPointer;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.hash.HashSet;
import com.intellij.xml.util.PsiElementPointer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

public abstract class AbstractSimpleSpringModel extends CachedLocalModel {

  @NotNull
  @Override
  public Collection<SpringBeanPointer> calculateDomBeans() {
    Set<SpringBeanPointer> domPointers = new HashSet<SpringBeanPointer>();
    for (SpringBeanPointer pointer : getAllCommonBeans()) {
      if (!(pointer instanceof JamSpringBeanPointer)) {
        domPointers.add(pointer);
      }
    }
    return Collections.unmodifiableSet(domPointers);
  }

  @NotNull
  @Override
  public Set<String> getAllBeanNames(@NotNull String beanName) {
    SpringBeanPointer beanPointer = SpringModelSearchers.findBean(this, beanName);
    if (beanPointer == null) {
      return Collections.singleton(beanName);
    }

    final String[] aliases = beanPointer.getAliases();
    if (aliases.length == 0) {
      return Collections.singleton(beanName);
    }

    final Set<String> allBeanNames = new java.util.HashSet<String>(aliases.length + 1);
    allBeanNames.add(beanName);
    for (String aliasName : aliases) {
      if (!StringUtil.isEmptyOrSpaces(aliasName)) {
        allBeanNames.add(aliasName);
      }
    }
    return allBeanNames;
  }

  @Override
  public Collection<PsiElementPointer> getDuplicatedNames(@NotNull String beanName) {
    return Collections.emptySet();
  }

  @NotNull
  @Override
  public List<SpringBeanPointer> getPlaceholderConfigurers() {
    return Collections.emptyList();
  }

  @NotNull
  @Override
  public List<SpringBeansPackagesScan> getComponentScans() {
    return Collections.emptyList();
  }

  @NotNull
  @Override
  public List<SpringBeanPointer> getAnnotationConfigApplicationContexts() {
    return Collections.emptyList();
  }

  @Override
  public Collection<XmlTag> getCustomBeanCandidates(String id) {
    return Collections.emptySet();
  }

  @NotNull
  @Override
  public List<SpringBeanPointer> getDescendants(@NotNull SpringBeanPointer context) {
    return Collections.emptyList();
  }

  @NotNull
  @Override
  public Set<String> getActiveProfiles() {
    return Collections.emptySet();
  }

  @Override
  public boolean processByClass(@NotNull SpringModelSearchParameters.BeanClass params,
                                @NotNull Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;

    final PsiClass searchClass = params.getSearchClass();
    final boolean withInheritors = params.isWithInheritors();

    if (params.isEffectiveBeanTypes()) {
      for (SpringBeanPointer beanPointer : getLocalBeans()) {
        for (PsiClass effectiveBeanType : beanPointer.getEffectiveBeanType()) {
          if (!processLocalBeanClass(processor, searchClass, beanPointer, effectiveBeanType, withInheritors)) return false;
        }
      }
    }
    else {
      for (SpringBeanPointer beanPointer : getLocalBeans()) {
        if (!processLocalBeanClass(processor, searchClass, beanPointer, beanPointer.getBeanClass(), withInheritors)) return false;
      }
    }

    return super.processByClass(params, processor);
  }

  private static boolean processLocalBeanClass(@NotNull Processor<SpringBeanPointer> processor,
                                               @NotNull PsiClass searchClass,
                                               SpringBeanPointer beanPointer,
                                               @Nullable PsiClass beanClass,
                                               boolean withInheritors) {
    if (beanClass != null) {
      if (beanClass.getManager().areElementsEquivalent(searchClass, beanClass) ||
          withInheritors && beanClass.isInheritor(searchClass, true)) {
        return processor.process(beanPointer);
      }
    }
    return true;
  }

  @Override
  public boolean processByName(@NotNull SpringModelSearchParameters.BeanName params,
                               @NotNull Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;

    for (SpringBeanPointer beanPointer : getLocalBeans()) {
      if (matchesName(params, beanPointer)) {
        if (!processor.process(beanPointer)) return false;
      }
    }

    return super.processByName(params, processor);
  }

  private static boolean matchesName(SpringModelSearchParameters.BeanName params, SpringBeanPointer pointer) {
    final String paramsBeanName = params.getBeanName();
    if (paramsBeanName.equals(pointer.getName())) return true;

    for (String aliasName : pointer.getAliases()) {
      if (paramsBeanName.equals(aliasName)) return true;
    }

    return false;
  }

  @NotNull
  protected Set<CommonSpringModel> getPackageScanModels(@NotNull Set<? extends LocalModel> localModels,
                                                        @NotNull Module module,
                                                        @NotNull Set<String> activeProfiles) {
    Set<CommonSpringModel> models = ContainerUtil.newLinkedHashSet();

    for (LocalModel model : localModels) {
      final List<SpringBeansPackagesScan> scans = model.getComponentScans();
      for (SpringBeansPackagesScan scan : scans) {
        models.add(new SpringComponentScanModel<SpringBeansPackagesScan>(module, scan, activeProfiles));
      }
    }

    return models;
  }
}
