/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.AtomicNotNullLazyValue;
import com.intellij.openapi.util.Comparing;
import com.intellij.openapi.util.Computable;
import com.intellij.openapi.util.RecursionManager;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.model.*;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.spring.model.jam.qualifiers.SpringJamQualifier;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import com.intellij.util.Processor;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ConcurrentFactoryMap;
import com.intellij.util.containers.hash.HashSet;
import gnu.trove.THashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public abstract class CachedLocalModel implements CommonSpringModel {

  private volatile Collection<SpringBeanPointer> myLocalBeans; // beans defined in current file

  private final AtomicNotNullLazyValue<Collection<SpringBeanPointer>> myLocalDomBeans =
    new AtomicNotNullLazyValue<Collection<SpringBeanPointer>>() {
      @NotNull
      protected Collection<SpringBeanPointer> compute() {
        return calculateDomBeans();
      }
    };

  @SuppressWarnings({"MismatchedQueryAndUpdateOfCollection"})
  private final Map<SpringQualifier, List<SpringBeanPointer>> myLocalBeansByQualifier =
    new ConcurrentFactoryMap<SpringQualifier, List<SpringBeanPointer>>() {
      protected List<SpringBeanPointer> create(final SpringQualifier key) {
        return computeLocalBeansByQualifier(key);
      }
    };

  @NotNull
  public final Collection<SpringBeanPointer> getAllCommonBeans() {
    final Set<SpringBeanPointer> pointers = new HashSet<SpringBeanPointer>();
    pointers.addAll(getLocalBeans());

    processRelatedModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        if (model instanceof CachedLocalModel) {
          pointers.addAll(((CachedLocalModel)model).getLocalBeans());
        }
        else {
          pointers.addAll(model.getAllCommonBeans());
        }
        return true;
      }
    });
    return Collections.unmodifiableSet(pointers);
  }

  public Collection<SpringBeanPointer> getLocalBeans() {
    return myLocalBeans == null ? myLocalBeans = calculateLocalBeans() : myLocalBeans;
  }

  @NotNull
  public final Collection<SpringBeanPointer> getAllDomBeans() {
    return myLocalDomBeans.getValue();
  }

  /**
   * All related models: imported, component-scan, implicit, etc. (these models could be cached with other dependencies)
   */
  protected Iterable<CommonSpringModel> getRelatedModels() {
    return Collections.emptySet();
  }

  protected abstract Collection<SpringBeanPointer> calculateLocalBeans();

  protected abstract Collection<SpringBeanPointer> calculateDomBeans();

  @NotNull
  public List<SpringBeanPointer> findQualifiedBeans(@NotNull final SpringQualifier qualifier) {
    return myLocalBeansByQualifier.get(qualifier);
  }

  private List<SpringBeanPointer> computeLocalBeansByQualifier(final SpringQualifier springQualifier) {
    final List<SpringBeanPointer> beans = new SmartList<SpringBeanPointer>();
    final Collection<SpringBeanPointer> pointers = getAllCommonBeans();
    for (SpringBeanPointer beanPointer : pointers) {
      if (!beanPointer.isValid()) continue;
      final CommonSpringBean bean = beanPointer.getSpringBean();
      final SpringQualifier qualifier = bean.getSpringQualifier();
      if (qualifier != null) {
        if (compareQualifiers(qualifier, springQualifier) || compareInheritorQualifier(qualifier, springQualifier)) {
          beans.add(beanPointer);
        }
      }
    }
    return beans.isEmpty() ? Collections.<SpringBeanPointer>emptyList() : beans;
  }

  private boolean compareInheritorQualifier(@Nullable SpringQualifier childrenQualifier, @Nullable final SpringQualifier baseQualifier) {
    if (childrenQualifier instanceof SpringJamQualifier && baseQualifier instanceof SpringJamQualifier) {
      if (Comparing.equal(childrenQualifier.getQualifierType(), baseQualifier.getQualifierType())) return false;
      final PsiClass baseType = baseQualifier.getQualifierType();
      final PsiClass childrenType = childrenQualifier.getQualifierType();
      final Module module = getModule();
      if (module != null && baseType != null && childrenType != null) {
        final String baseAnnoQualifiedName = baseType.getQualifiedName();
        if (baseAnnoQualifiedName != null) {
          final Collection<PsiClass> children =
            JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildren(baseAnnoQualifiedName);
          final PsiAnnotation definingMetaAnnotation =
            SpringAliasForUtils.findDefiningMetaAnnotation(childrenType, baseAnnoQualifiedName, children);
          if (definingMetaAnnotation != null) {
            return compareQualifiers(new SpringJamQualifier(definingMetaAnnotation, null), baseQualifier);
          }
        }
      }
    }

    return false;
  }

  private static boolean compareQualifiers(@Nullable SpringQualifier one, @Nullable final SpringQualifier two) {
    if (one == null || two == null) return false;
    if (!Comparing.equal(one.getQualifierType(), two.getQualifierType())) return false;
    if (!Comparing.equal(one.getQualifierValue(), two.getQualifierValue())) return false;
    final List<? extends QualifierAttribute> list1 = one.getQualifierAttributes();
    final int size1 = list1.size();
    final List<? extends QualifierAttribute> list2 = two.getQualifierAttributes();
    final int size2 = list2.size();
    if (size1 != size2) return false;
    if (size1 == 0) return true;
    final THashSet<QualifierAttribute> set = new THashSet<QualifierAttribute>(QualifierAttribute.HASHING_STRATEGY);
    set.addAll(list1);
    return set.containsAll(list2);
  }

  public boolean processRelatedModels(@NotNull final Processor<CommonSpringModel> processor) {
    final Boolean aBoolean = RecursionManager.doPreventingRecursion(this, false, new Computable<Boolean>() {
      @Override
      public Boolean compute() {
        for (CommonSpringModel model : getRelatedModels()) {
          if (!processor.process(model)) return Boolean.FALSE;
        }
        return Boolean.TRUE;
      }
    });

    return aBoolean == null ? true : aBoolean;
  }

  @Override
  public boolean processByClass(@NotNull final SpringModelSearchParameters.BeanClass params,
                                @NotNull final Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;

    return processRelatedModels(params, processor);
  }

  @Override
  public boolean processByName(@NotNull final SpringModelSearchParameters.BeanName params,
                               @NotNull final Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;

    return processRelatedModels(params, processor);
  }

  protected boolean processRelatedModels(final SpringModelSearchParameters.BeanClass params, final Processor<SpringBeanPointer> processor) {
    return processRelatedModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        return shouldProcess(model, processor) ? model.processByClass(params, processor) : true;
      }
    });
  }

  protected boolean processRelatedModels(final SpringModelSearchParameters.BeanName params, final Processor<SpringBeanPointer> processor) {
    return processRelatedModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        return shouldProcess(model, processor) ? model.processByName(params, processor) : true;
      }
    });
  }

  private static boolean shouldProcess(@NotNull CommonSpringModel model, @NotNull Processor<SpringBeanPointer> processor) {
    if (processor instanceof VisitedModelsDelegateProcessor) {
      if (((VisitedModelsDelegateProcessor)processor).hasBeenVisited(model)) return false;
      ((VisitedModelsDelegateProcessor)processor).addVisited(model);
    }
    return true;
  }
}
