/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model;

import com.intellij.spring.model.utils.SpringBeanUtils;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.spring.model.utils.SpringPropertyUtils;
import com.intellij.spring.model.xml.DomSpringBean;
import com.intellij.spring.model.xml.beans.*;
import com.intellij.util.Processor;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomUtil;
import org.jetbrains.annotations.Nullable;

import java.util.Set;

/**
 * @author Dmitry Avdeev
 */
public abstract class SpringModelVisitor {

  private final Processor<CommonSpringBean> myBeanProcessor = new Processor<CommonSpringBean>() {
    @Override
    public boolean process(CommonSpringBean bean) {
      return visitBean(SpringModelVisitor.this, bean);
    }
  };

  @Nullable
  protected Set<String> getActiveProfiles() {
    return null;
  }

  /**
   * Visits a bean.
   *
   * @param bean bean to be visited.
   * @return false to stop traversing.
   */
  protected boolean visitBean(CommonSpringBean bean) {
    return true;
  }

  protected boolean visitProperty(SpringPropertyDefinition property) {
    return true;
  }

  protected boolean visitConstructorArg(ConstructorArg arg) {
    return true;
  }

  protected boolean visitValueHolder(SpringValueHolder valueHolder) {
    return true;
  }

  protected boolean visitMapEntry(SpringEntry entry) {
    return true;
  }

  protected boolean visitRef(SpringRef ref) {
    return true;
  }

  protected boolean visitIdref(Idref idref) {
    return true;
  }

  private boolean visitChildrenBeans(DomElement parent) {
    if (!DomUtil.hasXml(parent)) return true;

    return SpringBeanUtils.getInstance().processChildBeans(parent, true, myBeanProcessor);
  }

  public static boolean visitBeans(final SpringModelVisitor visitor, Beans beans) {
    final Set<String> activeProfiles = visitor.getActiveProfiles();
    if (activeProfiles == null || SpringProfileUtils.isActiveProfile(beans, activeProfiles)) {
      if (!visitor.visitChildrenBeans(beans)) return false;

      for (Beans beansProfiles : beans.getBeansProfiles()) {
        if (!visitBeans(visitor, beansProfiles)) return false;
      }
    }
    return true;
  }

  public static boolean visitBean(SpringModelVisitor visitor, CommonSpringBean bean) {
    return visitBean(visitor, bean, true);
  }

  public static boolean visitBean(SpringModelVisitor visitor, CommonSpringBean bean, boolean deep) {
    if (bean instanceof DomSpringBean && !DomUtil.hasXml(((DomSpringBean)bean))) return true;
    if (!visitor.visitBean(bean)) return false;

    if (deep) {
      for (SpringPropertyDefinition property : SpringPropertyUtils.getProperties(bean)) {
        if (!visitor.visitProperty(property)) return false;
        if (property instanceof SpringValueHolder && !visitValueHolder(visitor, (SpringValueHolder)property)) return false;
      }
      if (bean instanceof SpringBean) {
        for (ConstructorArg arg : ((SpringBean)bean).getConstructorArgs()) {
          if (!visitor.visitConstructorArg(arg)) return false;
          if (!visitValueHolder(visitor, arg)) return false;
        }
      }
      if (bean instanceof ListOrSet) {
        if (!visitCollection(visitor, (ListOrSet)bean)) return false;
      }
      if (bean instanceof SpringMap) {
        if (!visitMap(visitor, (SpringMap)bean)) return false;
      }
    }
    return true;
  }

  private static boolean visitValueHolder(SpringModelVisitor visitor, SpringValueHolder elementsHolder) {
    if (!DomUtil.hasXml(elementsHolder)) return true;

    return visitor.visitValueHolder(elementsHolder) &&
           visitElementsHolder(visitor, elementsHolder);
  }

  private static boolean visitElementsHolder(final SpringModelVisitor visitor, final SpringElementsHolder elementsHolder) {
    if (!visitor.visitChildrenBeans(elementsHolder)) return false;

    final SpringRef ref = elementsHolder.getRef();
    if (DomUtil.hasXml(ref) && !visitor.visitRef(ref)) return false;
    final Idref idref = elementsHolder.getIdref();
    if (DomUtil.hasXml(idref) && !visitor.visitIdref(idref)) return false;
    if (!visitCollection(visitor, elementsHolder.getList())) return false;
    if (!visitCollection(visitor, elementsHolder.getSet())) return false;
    if (!visitCollection(visitor, elementsHolder.getArray())) return false;
    if (!visitMap(visitor, elementsHolder.getMap())) return false;
    return true;
  }

  private static boolean visitCollection(final SpringModelVisitor visitor,
                                         ListOrSet collection) {
    if (!visitor.visitChildrenBeans(collection)) return false;

    for (ListOrSet listOrSet : collection.getSets()) {
      if (!visitCollection(visitor, listOrSet)) return false;
    }
    for (ListOrSet listOrSet : collection.getArrays()) {
      if (!visitCollection(visitor, listOrSet)) return false;
    }
    for (ListOrSet listOrSet : collection.getLists()) {
      if (!visitCollection(visitor, listOrSet)) return false;
    }
    for (SpringMap map : collection.getMaps()) {
      if (!visitMap(visitor, map)) return false;
    }
    for (SpringRef ref : collection.getRefs()) {
      if (!visitor.visitRef(ref)) return false;
    }
    for (Idref idref : collection.getIdrefs()) {
      if (!visitor.visitIdref(idref)) return false;
    }
    return true;
  }

  private static boolean visitMap(SpringModelVisitor visitor, SpringMap map) {
    if (!DomUtil.hasXml(map)) return true;
    for (SpringEntry entry : map.getEntries()) {
      if (!visitor.visitMapEntry(entry)) return false;
      if (!visitValueHolder(visitor, entry)) return false;
      final SpringKey key = entry.getKey();
      if (DomUtil.hasXml(key) && !visitElementsHolder(visitor, key)) return false;
    }
    return true;
  }
}
