/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.converters;

import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiClassType;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.util.SmartList;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.DomJavaUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;

/**
 * Do not use for "plain" references in {@link com.intellij.util.xml.Convert}, use {@link com.intellij.spring.model.xml.RequiredBeanType} instead.
 */
public abstract class SpringBeanResolveConverterForDefiniteClasses extends SpringBeanResolveConverter {

  @Nullable
  protected abstract String[] getClassNames(final ConvertContext context);

  @NotNull
  public List<PsiClassType> getRequiredClasses(final ConvertContext context) {
    final String[] classNames = getClassNames(context);
    if (classNames == null || classNames.length == 0) return Collections.emptyList();

    final List<PsiClassType> required = new SmartList<PsiClassType>();
    for (String className : classNames) {
      final PsiClass psiClass = DomJavaUtil.findClass(className, context.getInvocationElement());
      if (psiClass != null) {
        required.add(PsiTypesUtil.getClassType(psiClass));
      }
    }
    return required;
  }
}
