/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.values.converters;

import com.intellij.codeInsight.completion.JavaLookupElementBuilder;
import com.intellij.codeInsight.daemon.EmptyResolveMessageProvider;
import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.*;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReferenceProvider;
import com.intellij.util.ArrayUtil;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.Converter;
import com.intellij.util.xml.CustomReferenceConverter;
import com.intellij.util.xml.GenericDomValue;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Yann C&eacute;bron
 */
public abstract class FieldRetrievingFactoryBeanConverter extends Converter<String> implements CustomReferenceConverter<String> {

  protected final boolean mySoft;

  protected FieldRetrievingFactoryBeanConverter() {
    this(true);
  }

  protected FieldRetrievingFactoryBeanConverter(boolean soft) {
    mySoft = soft;
  }

  protected boolean requireFieldReference() {
    return false;
  }

  @NotNull
  public PsiReference[] createReferences(final GenericDomValue<String> genericDomValue,
                                         final PsiElement element,
                                         final ConvertContext context) {
    return createReferences(genericDomValue, element);
  }

  public String fromString(@Nullable @NonNls String s, final ConvertContext context) {
    return s;
  }

  public String toString(@Nullable String s, final ConvertContext context) {
    return s;
  }


  public PsiReference[] createReferences(final GenericDomValue<String> genericDomValue, final PsiElement element) {
    final String stringValue = genericDomValue.getStringValue();
    if (stringValue == null) {
      return PsiReference.EMPTY_ARRAY;
    }

    List<PsiReference> collectedReferences = new ArrayList<PsiReference>();

    final JavaClassReferenceProvider provider = new JavaClassReferenceProvider();
    provider.setSoft(mySoft);
    provider.setOption(JavaClassReferenceProvider.ALLOW_DOLLAR_NAMES, Boolean.TRUE);
    final PsiReference[] javaClassReferences = provider.getReferencesByElement(element);

    PsiClass psiClass = null;
    for (PsiReference reference : javaClassReferences) {
      final PsiElement psiElement = reference.resolve();
      if (psiElement == null) break;


      collectedReferences.add(reference);
      if (psiElement instanceof PsiClass) {
        psiClass = (PsiClass)psiElement;
      }
    }

    if (psiClass == null ||
        !requireFieldReference() && stringValue.endsWith(psiClass.getQualifiedName())) {
      return javaClassReferences;
    }

    collectedReferences.add(createFieldReference(psiClass, element, stringValue, genericDomValue));

    return collectedReferences.toArray(new PsiReference[collectedReferences.size()]);
  }

  private PsiReference createFieldReference(final PsiClass psiClass,
                                            final PsiElement element,
                                            final String stringValue,
                                            final GenericDomValue<String> genericDomValue) {
    final String className = psiClass.getName();
    assert className != null;
    final int fieldNameIdx = stringValue.lastIndexOf(className) + className.length();
    final String fieldName = stringValue.substring(Math.min(stringValue.length(), fieldNameIdx + 1)).trim();

    final TextRange textRange;
    if (fieldName.isEmpty()) {
      textRange = TextRange.from(element.getText().indexOf(className) + className.length() + 1, 0);
    }
    else {
      textRange = TextRange.from(Math.max(0, element.getText().lastIndexOf(fieldName)), fieldName.length());
    }

    return new FieldReference(element, textRange, fieldName, psiClass, genericDomValue);
  }


  protected class FieldReference extends PsiReferenceBase<PsiElement> implements EmptyResolveMessageProvider {

    private final String myFieldName;
    private final PsiClass myPsiClass;
    private final GenericDomValue<String> myGenericDomValue;

    protected FieldReference(PsiElement element,
                             TextRange textRange,
                             String fieldName,
                             PsiClass psiClass,
                             GenericDomValue<String> genericDomValue) {
      super(element, textRange, FieldRetrievingFactoryBeanConverter.this.mySoft);
      myFieldName = fieldName;
      myPsiClass = psiClass;
      myGenericDomValue = genericDomValue;
    }

    public PsiElement resolve() {
      if (myFieldName.length() != 0) {
        final PsiField[] psiFields = myPsiClass.getAllFields();
        for (PsiField psiField : psiFields) {
          if (psiField.hasModifierProperty(PsiModifier.PUBLIC) &&
              psiField.hasModifierProperty(PsiModifier.STATIC) &&
              myFieldName.equals(psiField.getName())) {
            return psiField;
          }
        }
      }
      return null;
    }

    @Override
    public PsiElement bindToElement(@NotNull final PsiElement element) throws IncorrectOperationException {
      if (element instanceof PsiField) {
        final PsiField field = (PsiField)element;
        myGenericDomValue.setStringValue(field.getContainingClass().getQualifiedName() + "." + field.getName());
      }
      return getElement();
    }

    @NotNull
    public Object[] getVariants() {
      List<LookupElement> staticFields = new ArrayList<LookupElement>();
      final PsiField[] psiFields = myPsiClass.getFields();
      for (PsiField psiField : psiFields) {
        if (psiField.hasModifierProperty(PsiModifier.PUBLIC) &&
            psiField.hasModifierProperty(PsiModifier.STATIC) &&
            psiField.getName() != null) {
          staticFields.add(JavaLookupElementBuilder.forField(psiField, psiField.getName(), myPsiClass)
                             .withTypeText(psiField.getType().getPresentableText()));
        }
      }
      return ArrayUtil.toObjectArray(staticFields);
    }

    @NotNull
    @Override
    public String getUnresolvedMessagePattern() {
      final String fieldName = getValue();
      if (fieldName.isEmpty() || fieldName.equals(".")) {
        return "Field name expected";
      }
      return "Cannot resolve field '" + fieldName + "'";
    }
  }

  /**
   * Highlights missing field reference.
   */
  public static class FieldReferenceRequired extends FieldRetrievingFactoryBeanConverter {

    @Override
    protected boolean requireFieldReference() {
      return true;
    }
  }
}
