---- RetrieveSessionInfo ----
with oraver as ( select min(version) as oracle_version
                 from product_component_version
                 where lower(product) like 'oracle%'
                    or lower(product) like '%oracle%database%' )
select sys_context('userenv', 'current_schema') as current_schema,
       user as current_user,
       -- sys_context('userenv','instance_name') as database_name,
       sys_context('userenv','db_name') as database_name,
       sysdate as now,
       oracle_version
from oraver
/


---- CheckBaseDictionariesAccess ----
with U as (select 1 from sys.user$       where user#  < 0),
     O as (select 1 from sys.obj$        where obj#   < 0),
     D as (select 1 from sys.dependency$ where d_obj# < 0)
select count(0) as "0"
from U, O, D
/


---- RetrieveAccessibleSchemas ----
select username
from sys.all_users U
where exists (select 1 from sys.all_objects where owner = U.username)
   or username = user
   or username = sys_context('userenv','current_schema')
/


---- RetrieveSchemaInfo ----
with U as ( select nvl(max(user_id), -1) as user_id
            from all_users
            where username = :schema_name ),
     O as ( select min(created) as created_timestamp,
                   max(last_ddl_time) as modified_timestamp
            from all_objects
            where owner = :schema_name )
select *
from U, O
/

---- RetrievePublicSchemaInfo ----
with U as ( select 1 as user_id from dual ),
     O as ( select min(created) as created_timestamp,
                   max(last_ddl_time) as modified_timestamp
            from all_objects
            where owner = 'PUBLIC' )
select *
from U, O
/


---- RetrieveObjectIdentifiersFast ----
select /*+all_rows*/ obj#
from sys.obj$
where owner# = :schema_id
  and type# between 1 and 42
  and type# not between 19 and 41
  and type# != 10 -- NON-EXISTENT
  and bitand(flags,128) = 0 -- skip dropped tables
  and linkname is null
/


---- RetrieveObjectIdentifiersSlow ----
select object_id
from sys.all_objects
where owner = :schema_name
  and object_type in ( 'SEQUENCE','TABLE','INDEX','VIEW','TRIGGER',
                       'TYPE','TYPE BODY','PACKAGE','PACKAGE BODY',
                       'PROCEDURE','FUNCTION','OPERATOR','SYNONYM' )
/


---- RetrieveSequences ----
select sequence_name, min_value, max_value, increment_by,
       cycle_flag, order_flag, cache_size, last_number,
       object_id, created_timestamp, modified_timestamp
from ( select sequence_owner as schema_name,
              sequence_name,
              min_value,
              decode(max_value, 999999999999999999999999999,null, max_value) as max_value,
              least(increment_by, 9223372036854775807) as increment_by,
              decode(cycle_flag, 'Y',1,0) as cycle_flag,
              decode(order_flag, 'Y',1,0) as order_flag,
              decode(least(cache_size, 9223372036854775807), 0,-2147483648, 20,null, cache_size) as cache_size,
              last_number
       from all_sequences )
     natural join
     ( select owner as schema_name,
              object_name as sequence_name,
              object_id,
              created as created_timestamp,
              last_ddl_time as modified_timestamp
       from all_objects
       where owner = :schema_name
         #INC>> and last_ddl_time >= :since
         and object_type = 'SEQUENCE' )
order by created_timestamp, object_id
/


---- RetrieveObjectTypes ----
select type_name, supertype_schema_name, supertype_name,
       object_id, created_timestamp, modified_timestamp
from ( select owner as schema_name, type_name,
              supertype_owner as supertype_schema_name, supertype_name,
              decode(incomplete, 'YES', 1, 0) as incomplete,
              decode(final, 'YES', 1, 0) as final,
              decode(instantiable, 'YES', 1, 0) as instantiable
       from all_types
       where typecode = 'OBJECT' )
     natural join
     ( select owner as schema_name, object_name as type_name,
              object_id,
              created as created_timestamp, last_ddl_time as modified_timestamp
       from all_objects
       where owner = :schema_name
         #INC>> and last_ddl_time >= :since
         and object_type = 'TYPE' )
order by created_timestamp, object_id
/


---- RetrieveCollectionTypes ----
select type_name, collection_kind, upper_bound,
       data_type_schema_name, data_type, data_size, data_scale, data_size_units, mandatory,
       object_id, created_timestamp, modified_timestamp
from ( select owner as schema_name, type_name,
              case when coll_type like '%ARRAY%' then 'A' else 'T' end as collection_kind,
              upper_bound,
              elem_type_owner as data_type_schema_name,
              elem_type_name as data_type,
              case when elem_type_name in ('NUMBER','FLOAT') then precision
                   when elem_type_name like '%CHAR%' then length
                   when elem_type_name like '%RAW' then length
                   else null
                   end as data_size,
              scale as data_scale,
              case when elem_type_name in ('NUMBER','FLOAT') then 'D'
                   when elem_type_name like '%N%CHAR%' then 'C'
                   when elem_type_name like '%CHAR%' then 'B' --char_used
                   when elem_type_name like '%RAW' then 'B'
                   when elem_type_name like 'TIMESTAMP' then 'D'
                   else null
                   end as data_size_units,
              decode(nulls_stored, 'YES', 0, 1) as mandatory
       from all_coll_types
       where not (owner = 'SYS' and type_name like 'SYS\_PLSQL\_%' escape '\') )
     natural join
     ( select owner as schema_name, object_name as type_name,
              object_id,
              created as created_timestamp, last_ddl_time as modified_timestamp
       from all_objects
       where owner = :schema_name
         #INC>> and last_ddl_time >= :since
         and object_type = 'TYPE'
         and not (owner = 'SYS' and object_name like 'SYS\_PLSQL\_%' escape '\') )
order by created_timestamp, object_id
/


---- RetrieveObjectTypeAttributes ----
with my_objects as ( select owner, object_name as type_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type = 'TYPE' )
select type_name,
       attr_no,
       attr_name,
       attr_type_owner as data_type_schema,
       attr_type_name as data_type,
       case when attr_type_name in ('NUMBER','FLOAT') then nvl(precision,-1)
          when attr_type_name like '%RAW' then nvl(length,-1)
          else -1
          end as data_size,
       scale as data_scale,
       case when attr_type_name in ('NUMBER','FLOAT') then 'D'
          when attr_type_name like '%N%CHAR%' then 'C'
          when attr_type_name like '%CHAR%' then 'B' --char_used
          when attr_type_name like '%RAW' then 'B'
          when attr_type_name like 'TIMESTAMP' then 'D'
          else null
          end as data_size_units
from all_type_attrs
     natural join
     my_objects
where inherited = 'NO'
order by owner, type_name, attr_no
/


---- RetrieveTables09 ----
with schemas as ( select :schema_name as schema_name from dual),
     ldt as (select :since as last_ddl_time from dual)
--
select table_name, comments, organization_kind,
       row_type_schema_name, row_type_name,
       object_id, surrogate, created_timestamp, modified_timestamp
from
  (
  select table_name,
         nvl2(iot_type, 'I', 'E') as organization_kind,
         null as row_type_schema_name,
         null as row_type_name
  from all_tables
  where owner = (select schema_name from schemas)
    and secondary = 'N'
    and iot_name is null
  )
  natural join
  (
  select object_name as table_name,
         object_id, decode(generated, 'Y', 1, 0) as surrogate,
         created as created_timestamp, last_ddl_time as modified_timestamp
  from all_objects
  where owner = (select schema_name from schemas)
    and last_ddl_time >= (select last_ddl_time from ldt)
    and object_type = 'TABLE'
  )
  natural join
  (
  select table_name, comments
  from all_tab_comments
  where owner = (select schema_name from schemas)
    and table_type = 'TABLE'
  )
--
union all
--
select table_name, comments, organization_kind,
       row_type_schema_name, row_type_name,
       object_id, surrogate, created_timestamp, modified_timestamp
from
  (
  select table_name,
         nvl2(iot_type, 'I', 'E') as organization_kind,
         table_type_owner as row_type_schema_name,
         table_type as row_type_name
  from all_object_tables
  where owner = (select schema_name from schemas)
    and secondary = 'N'
    and iot_name is null
  )
  natural join
  (
  select object_name as table_name,
         object_id, decode(generated, 'Y', 1, 0) as surrogate,
         created as created_timestamp, last_ddl_time as modified_timestamp
  from all_objects
  where owner = (select schema_name from schemas)
    and last_ddl_time >= (select last_ddl_time from ldt)
    and object_type = 'TABLE'
  )
  natural join
  (
  select table_name, comments
  from all_tab_comments
  where owner = (select schema_name from schemas)
    and table_type = 'TABLE'
  )
--
order by created_timestamp, object_id
/


---- RetrieveTables10 ----
with schemas as ( select :schema_name as schema_name from dual),
     ldt as (select :since as last_ddl_time from dual)
--
select table_name, comments, organization_kind,
       row_type_schema_name, row_type_name,
       object_id, surrogate, created_timestamp, modified_timestamp
from
  (
  select table_name,
         nvl2(iot_type, 'I', 'E') as organization_kind,
         null as row_type_schema_name,
         null as row_type_name
  from all_tables
  where owner = (select schema_name from schemas)
    and dropped = 'NO'
    and secondary = 'N'
    and iot_name is null
  )
  natural join
  (
  select object_name as table_name,
         object_id, decode(generated, 'Y', 1, 0) as surrogate,
         created as created_timestamp, last_ddl_time as modified_timestamp
  from all_objects
  where owner = (select schema_name from schemas)
    and last_ddl_time >= (select last_ddl_time from ldt)
    and object_type = 'TABLE'
  )
  natural join
  (
  select table_name, comments
  from all_tab_comments
  where owner = (select schema_name from schemas)
    and table_type = 'TABLE'
  )
--
union all
--
select table_name, comments, organization_kind,
       row_type_schema_name, row_type_name,
       object_id, surrogate, created_timestamp, modified_timestamp
from
  (
  select table_name,
         nvl2(iot_type, 'I', 'E') as organization_kind,
         table_type_owner as row_type_schema_name,
         table_type as row_type_name
  from all_object_tables
  where owner = (select schema_name from schemas)
    and dropped = 'NO'
    and secondary = 'N'
    and iot_name is null
  )
  natural join
  (
  select object_name as table_name,
         object_id, decode(generated, 'Y', 1, 0) as surrogate,
         created as created_timestamp, last_ddl_time as modified_timestamp
  from all_objects
  where owner = (select schema_name from schemas)
    and last_ddl_time >= (select last_ddl_time from ldt)
    and object_type = 'TABLE'
  )
  natural join
  (
  select table_name, comments
  from all_tab_comments
  where owner = (select schema_name from schemas)
    and table_type = 'TABLE'
  )
--
order by created_timestamp, object_id
/


---- RetrieveViews10 ----
select view_name, comments, modifiable,
       object_id, is_invalid,
       created_timestamp, modified_timestamp
from
  (
  select owner as schema_name, view_name,
         to_number(null) as modifiable
  from all_views
  )
  natural join
  (
  select owner as schema_name, object_name as view_name,
         object_id, decode(status, 'INVALID', 1, 0) as is_invalid,
         created as created_timestamp, last_ddl_time as modified_timestamp
  from all_objects
  where owner = :schema_name
    #INC>> and last_ddl_time >= :since
    and object_type = 'VIEW'
  )
  natural join
  (
  select owner as schema_name, table_name as view_name, comments
  from all_tab_comments
  where table_type = 'VIEW'
  )
order by created_timestamp, object_id
/

---- RetrieveViews11 ----
select view_name, comments, modifiable,
       object_id, is_invalid,
       created_timestamp, modified_timestamp
from
  (
  select owner as schema_name, view_name,
         decode(read_only, 'N', 1, 0) as modifiable
  from all_views
  )
  natural join
  (
  select owner as schema_name, object_name as view_name,
         object_id, decode(status, 'INVALID', 1, 0) as is_invalid,
         created as created_timestamp, last_ddl_time as modified_timestamp
  from all_objects
  where owner = :schema_name
    #INC>> and last_ddl_time >= :since
    and object_type = 'VIEW'
  )
  natural join
  (
  select owner as schema_name, table_name as view_name, comments
  from all_tab_comments
  where table_type = 'VIEW'
  )
order by created_timestamp, object_id
/


---- RetrieveColumns ----
with my_objects as ( select owner, object_id as table_id, object_name as table_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type in ('TABLE','VIEW') )
select table_id,
       column_id as column_position,
       column_name,
       data_type_owner as data_type_schema_name,
       data_type,
       case when data_type in ('NUMBER','FLOAT') then nvl(data_precision,2147483646)
          when data_type like '%CHAR%' then nvl(char_length,-1)
          when data_type like 'LONG RAW' then -1
          when data_type like 'RAW' then nvl(data_length,-1)
          else -1
          end as data_size,
       data_scale,
       case when data_type in ('NUMBER','FLOAT') then 'D'
          when data_type like '%N%CHAR%' then 'C'
          when data_type like '%CHAR%' then 'B' --char_used
          when data_type like '%RAW' then 'B'
          when data_type like 'TIMESTAMP' then 'D'
          else null
          end as data_size_units,
       decode(nullable, 'N', 1, 0) as mandatory
from my_objects
     natural join
     all_tab_columns
order by table_id, column_id
/


---- RetrieveColumnDefaults ----
with my_objects as ( select owner, object_name as table_name, last_ddl_time
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type in ('TABLE','VIEW') )
select table_name,
       column_name,
       data_default as default_expression
from my_objects
     natural join
     all_tab_columns
where data_default is not null
order by last_ddl_time
/


---- RetrieveColumnComments ----
with my_objects as ( select owner, object_name as table_name, last_ddl_time
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type in ('TABLE','VIEW') )
select table_name,
       column_name,
       comments
from my_objects
     natural join
     all_col_comments
where comments is not null
order by last_ddl_time
/


---- RetrieveIndices ----
select table_name, index_schema_name, index_name,
       uniqueness, clustering, tablespace_name,
       object_id, surrogate,
       created_timestamp, modified_timestamp
from ( select table_owner as schema_name, table_name,
              owner as index_schema_name, index_name,
              decode(uniqueness, 'UNIQUE', 1, 0) as uniqueness,
              decode(substr(index_type,1,3), 'IOT', 1,0) as clustering,
              tablespace_name
       from all_indexes
       where table_name not like 'BIN$%' )
     natural join
     ( select owner as index_schema_name, object_name as index_name,
              object_id, decode(generated, 'Y', 1, 0) as surrogate,
              created as created_timestamp, last_ddl_time as modified_timestamp
       from all_objects
       where owner = :schema_name
         #INC>> and last_ddl_time >= :since
         and object_type = 'INDEX' )
order by created_timestamp, object_id
/


---- RetrieveIndicesColumns ----
with my_objects as ( select owner as index_owner, object_name as index_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type = 'INDEX' )
select C.table_name,
       C.index_owner as index_schema_name,
       C.index_name,
       C.column_position,
       C.column_name,
       E.column_expression,
       decode(substr(C.descend,1,1), 'D', 1, 0) as descendant
from my_objects O,
     all_ind_columns C,
     all_ind_expressions E
where O.index_owner = C.index_owner
  and O.index_name = C.index_name
  and C.table_name not like 'BIN$%'
  and C.index_owner = E.index_owner (+)
  and C.index_name = E.index_name (+)
  and C.table_owner = E.table_owner (+)
  and C.table_name = E.table_name (+)
  and C.column_position = E.column_position (+)
order by C.table_owner, C.table_name, C.index_name, C.column_position
/


---- RetrieveKeys ----
with my_objects as ( select owner, object_name as table_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type = 'TABLE'
                       and object_name not like 'BIN$%' )
select table_name,
       constraint_name,
       decode(substr(generated,1,1), 'G',1,0) as is_surrogate_name,
       decode(constraint_type,'P',1,0) as is_pk,
       decode(substr(deferrable,1,1),'D',1,0) as is_deferrable,
       decode(substr(deferred,1,1),'I',1,0) as is_initially_deferred,
       nvl(index_owner,owner) as index_schema_name, index_name,
       last_change as modified_timestamp
from my_objects
     natural join
     all_constraints
where table_name not like 'BIN$%'
  and constraint_type in ('P','U')
  and constraint_name not like 'BIN$%'
order by constraint_type, last_change
/


---- RetrieveForeignKeys ----
with my_objects as ( select owner, object_name as table_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type = 'TABLE'
                       and object_name not like 'BIN$%' )
select table_name, constraint_name,
       decode(substr(generated,1,1), 'G',1,0) as is_surrogate_name,
       r_owner as ref_schema_name, r_constraint_name as ref_key_name,
       decode(substr(delete_rule,1,1), 'C',1,0) as cascade_delete,
       decode(substr(deferrable,1,1),'D',1,0) as is_deferrable,
       decode(substr(deferred,1,1),'I',1,0) as is_initially_deferred,
       last_change as modified_timestamp
from my_objects
     natural join
     all_constraints
where table_name not like 'BIN$%'
  and constraint_type = 'R'
  and constraint_name not like 'BIN$%'
/


---- RetrieveChecks ----
with my_objects as ( select owner, object_name as table_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type = 'TABLE'
                       and object_name not like 'BIN$%' )
select table_name, constraint_name,
       decode(substr(generated,1,1), 'G',1,0) as is_surrogate_name,
       search_condition as predicate,
       decode(substr(deferrable,1,1),'D',1,0) as is_deferrable,
       decode(substr(deferred,1,1),'I',1,0) as is_initially_deferred,
       last_change as modified_timestamp
from my_objects
     natural join
     all_constraints
where table_name not like 'BIN$%'
  and constraint_type = 'C'
  and search_condition is not null
/


---- RetrieveConstraintColumns ----
with sch as (select :schema_name as owner from dual),
     my_objects as ( select object_name as table_name
                     from all_objects
                     where owner = (select owner from sch)
                       #INC>> and last_ddl_time >= :since
                       and object_type = 'TABLE'
                       and object_name not like 'BIN$%' )
select table_name, constraint_name,
       nvl(position,1) as column_position, column_name
from my_objects
     natural join
     all_cons_columns
where all_cons_columns.owner = (select owner from sch)
  and table_name not like 'BIN$%'
  and constraint_name not like 'BIN$%'
order by 1,2,3,4
/


---- RetrieveTriggers ----
select table_name,
       trigger_name,
       decode(substr(turn,1,1), 'B',-1, 'A',+1, 0) *
           (case when turn like '%STAT%' then 2 else 1 end) as turn,
       cast ( case when events like '%INSERT%' then 'I' else '' end
           || case when events like '%UPDATE%' then 'U' else '' end
           || case when events like '%DELETE%' then 'D' else '' end as varchar(3)) as events,
       case when lower(trigger_description) like '%update of%' then 1 else 0 end as has_on_update_columns,
       referencing_names,
       condition,
       action_type,
       object_id,
       created_timestamp,
       modified_timestamp
from ( select table_owner as schema_name,
              table_name,
              owner as trigger_schema_name,
              trigger_name,
              trigger_type as turn,
              triggering_event as events,
              when_clause as condition,
              action_type,
              description as trigger_description,
              decode(referencing_names, 'REFERENCING NEW AS NEW OLD AS OLD', null, referencing_names) as referencing_names,
              trigger_body
       from all_triggers
       where table_name not like 'BIN$%' )
     natural join
     ( select owner as trigger_schema_name,
              object_name as trigger_name,
              object_id,
              status,
              created as created_timestamp,
              last_ddl_time as modified_timestamp
       from all_objects
       where owner = :schema_name
         #INC>> and last_ddl_time >= :since
         and object_type = 'TRIGGER' )
order by created_timestamp, object_id
/


---- RetrieveTriggerUpdateColumns ----
select table_name,
       trigger_name,
       column_name
from ( select table_owner as schema_name,
              table_name,
              trigger_owner as trigger_schema_name,
              trigger_name,
              column_name
       from all_trigger_cols
       where table_name not like 'BIN$%' )
     natural join
     ( select owner as trigger_schema_name,
              object_name as trigger_name
       from all_objects
       where owner = :schema_name
         #INC>> and last_ddl_time >= :since
         and object_type = 'TRIGGER' )
/


---- RetrievePackages ----
select object_name as package_name,
       decode(object_type, 'PACKAGE BODY', 1, 0) as is_body,
       decode(status, 'INVALID', 1, 0) as is_invalid,
       object_id,
       created as created_timestamp, last_ddl_time as modified_timestamp
from all_objects
where owner = :schema_name
  #INC>> and last_ddl_time >= :since
  and object_type in ('PACKAGE','PACKAGE BODY')
order by decode(object_type, 'PACKAGE BODY', 1, 0), created, object_id
/


---- RetrieveInnerRoutines10 ----
with my_objects as ( select owner, object_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type in ('TYPE','PACKAGE') )
select 'Z' as module_kind,
       object_name as module_name,
       0 as position,
       procedure_name as routine_name,
       decode(deterministic, 'YES', 1, 0) as is_deterministic
from my_objects
     natural join
     all_procedures
where procedure_name is not null
/

---- RetrieveInnerRoutines11 ----
with my_objects as ( select owner, object_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type in ('TYPE','PACKAGE') )
select substr(object_type,1,1) as module_kind,
       object_name as module_name,
       subprogram_id as position,
       procedure_name as routine_name,
       decode(deterministic, 'YES', 1, 0) as is_deterministic
from my_objects
     natural join
     all_procedures
where procedure_name is not null
order by object_id, subprogram_id
/


---- RetrieveInnerRoutineArguments10 ----
with my_objects as ( select owner, object_name as package_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type in ('TYPE','PACKAGE') )
select package_name,
       object_name as routine_name,
       position as arg_position,
       argument_name as arg_name,
       case when in_out = 'OUT' and position = 0 and argument_name is null then 'R'
            when in_out = 'IN/OUT' then 'M'
            when in_out = 'IN' and argument_name = 'SELF' and data_type = 'OBJECT' then 'S'
            else substr(in_out,1,1) end as arg_direction,
       type_owner as data_type_schema_name,
       nvl(type_name, data_type) as data_type
from my_objects
     natural join
     all_arguments
where not (argument_name is null and sequence = 0)
order by object_id, position
/

---- RetrieveInnerRoutineArguments11 ----
with my_objects as ( select owner, object_name as package_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type in ('TYPE','PACKAGE') )
select package_name,
       object_name as routine_name,
       position as arg_position,
       argument_name as arg_name,
       case when in_out = 'OUT' and position = 0 and argument_name is null then 'R'
            when in_out = 'IN/OUT' then 'M'
            when in_out = 'IN' and argument_name = 'SELF' and data_type = 'OBJECT' then 'S'
            else substr(in_out,1,1) end as arg_direction,
       type_owner as data_type_schema_name,
       nvl(type_name, data_type) as data_type
from all_arguments
     natural join
     my_objects
where not (argument_name is null and sequence = 0)
order by object_id, subprogram_id, position
/


---- RetrieveSingleRoutines ----
select O.name, kind, is_invalid, object_id, is_deterministic,
       created_timestamp, modified_timestamp
from (  select owner as schema_name,
               object_name as name,
               substr(object_type,1,1) as kind,
               decode(status, 'INVALID', 1, 0) as is_invalid,
               object_id,
               created as created_timestamp, last_ddl_time as modified_timestamp
        from all_objects
        where owner = :schema_name
          #INC>> and last_ddl_time >= :since
          and object_type in ('FUNCTION','PROCEDURE') ) O
     left outer join
     (  select owner as schema_name,
               object_name as name,
               decode(deterministic, 'YES', 1, 0) as is_deterministic
        from all_procedures
        where procedure_name is null ) P
     on O.schema_name = P.schema_name and O.name = P.name
order by created_timestamp, object_id
/


---- RetrieveSingleRoutineArguments10 ----
with my_objects as ( select owner, object_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type in ('FUNCTION','PROCEDURE') )
select object_name as routine_name,
       position as arg_position,
       argument_name as arg_name,
       case when in_out = 'OUT' and position = 0 and argument_name is null then 'R'
            when in_out = 'IN/OUT' then 'M'
            when in_out = 'IN' and argument_name = 'SELF' and data_type = 'OBJECT' then 'S'
            else substr(in_out,1,1) end as arg_direction,
       type_owner as data_type_schema_name,
       nvl(type_name, data_type) as data_type
from my_objects
     natural join
     all_arguments
where not (argument_name is null and sequence = 0)
order by object_id, position
/

---- RetrieveSingleRoutineArguments11 ----
with my_objects as ( select owner, object_name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type in ('FUNCTION','PROCEDURE') )
select object_name as routine_name,
       position as arg_position,
       argument_name as arg_name,
       case when in_out = 'OUT' and position = 0 and argument_name is null then 'R'
            when in_out = 'IN/OUT' then 'M'
            when in_out = 'IN' and argument_name = 'SELF' and data_type = 'OBJECT' then 'S'
            else substr(in_out,1,1) end as arg_direction,
       type_owner as data_type_schema_name,
       nvl(type_name, data_type) as data_type
from my_objects
     natural join
     all_arguments
where not (argument_name is null and sequence = 0)
order by object_id, subprogram_id, position
/


---- RetrieveSynonyms ----
select S.synonym_name,
       O.object_id,
       O.created as created_timestamp,
       O.last_ddl_time as modified_timestamp,
       S.db_link as origin_db_link,
       S.table_owner as origin_schema_name,
       S.table_name as origin_object_name
from all_objects O,
     all_synonyms S
where O.object_type = 'SYNONYM'
  and O.owner = :schema_name
  and O.object_name not like '%/%'
  #INC>> and O.last_ddl_time >= :since
  and O.owner = :schema_name
  and O.object_name = S.synonym_name
order by object_id
/


---- RetrieveDependenciesFast ----
with X as ( select min(obj#) as exclude_standard_id
            from sys.obj$
            where owner# = 0 -- SYS
              and type# = 9  -- package spec
              and name = 'STANDARD' )
select /*+all_rows*/ distinct
       D.d_obj# as source_object_id,
       D.p_obj# as target_object_id
from sys.obj$ O,
     sys.dependency$ D,
     X
where O.owner# = :schema_id
  #INC>> and O.mtime >= :since
  and O.type# between 1 and 42
  and O.type# not between 19 and 41
  and O.type# not in (5,10) -- SYNONYM, NON-EXISTENT
  and O.obj# = D.d_obj#
  and D.p_obj# <> exclude_standard_id
/


---- RetrieveDependenciesSlow ----
select distinct O1.object_id as source_object_id,
                O2.object_id as target_object_id
from all_objects O1,
     all_dependencies D,
     all_objects O2
where O1.owner = :schema_name
  #INC>> and O1.last_ddl_time >= :since
  and O1.object_type not in ('SYNONYM', 'JAVA CLASS', 'JAVA DATA')
  and O1.owner = D.owner
  and O1.object_name = D.name
  and O1.object_type = D.type
  and (D.referenced_owner, D.referenced_name, D.referenced_type) not in (('SYS','STANDARD','PACKAGE'))
  and D.referenced_owner = O2.owner
  and D.referenced_name = O2.object_name
  and D.referenced_type = O2.object_type
  and O1.object_id != O2.object_id
  and O2.object_type not in ('JAVA CLASS', 'JAVA DATA')
  and not (O1.object_type = O2.object_type||' BODY' and O1.owner = O2.owner and O1.object_name = O2.object_name)
/


---- RetrieveViewSources ----
select view_name, text
from
  (
  select owner, view_name, text
  from all_views
  )
  natural join
  (
  select owner, object_name as view_name,
         created, object_id
  from all_objects
  where owner = :schema_name
    #INC>> and last_ddl_time >= :since
    and object_type = 'VIEW'
  )
order by created, object_id
/


---- RetrieveSources ----
with my_objects as ( select owner, object_name as name
                     from all_objects
                     where owner = :schema_name
                       #INC>> and last_ddl_time >= :since
                       and object_type in ('TYPE','TYPE BODY','PACKAGE','PACKAGE BODY','FUNCTION','PROCEDURE') )
select type,
       name,
       line,
       substr(text,1,256) as text
from my_objects
     natural join
     all_source
where type in ('TYPE','TYPE BODY','PACKAGE','PACKAGE BODY','FUNCTION','PROCEDURE')
order by 1,2,3
/


---- RetrieveErrors ----
select name,
       decode(type, 'PROCEDURE', 'R', substr(type,1,1)) as type_code,
       decode(instr(type, 'BODY'), 0,0, 1) as is_body,
       line, position,
       text as message
from all_errors
where owner = :schema_name
order by name, type, sequence, line, position
/

